/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.ejb.mdb;


import org.apache.commons.logging.Log;

import xjavadoc.XClass;

import xdoclet.XDocletException;
import xdoclet.XDocletMessages;
import xdoclet.modules.ejb.AbstractEjbCodeGeneratorSubTask;
import xdoclet.modules.ejb.EjbTagsHandler;
import xdoclet.modules.ejb.XDocletModulesEjbMessages;
import xdoclet.tagshandler.PackageTagsHandler;
import xdoclet.util.LogUtil;

import xdoclet.util.Translator;

/**
 * @author               Jonathan Kovacs (jonk at globaliqx.com)
 * @created              Sept 15, 2003
 * @ant.element          display-name="Mdb Bean" name="mdb" parent="xdoclet.modules.ejb.EjbDocletTask"
 * @version              $Revision: 1.2 $
 * @xdoclet.merge-file   file="mdb-custom.xdt" relates-to="{0}Mdb.java" description="A text file containing custom
 *      template and/or java code to include in the EJB mdb class."
 */
public class MdbSubTask extends AbstractEjbCodeGeneratorSubTask
{
    public final static String DEFAULT_MESSAGE_DRIVEN_CLASS_PATTERN = "{0}Mdb";

    protected final static String DEFAULT_TEMPLATE_FILE = "resources/mdb.xdt";

    /**
     * A configuration parameter for specifying the concrete mdb bean class name pattern. By default the value is used
     * for deciding the concrete mdb bean class name. {0} in the value mean current class's symbolic name which for an
     * EJBean is the EJB name.
     *
     * @see   #getMessageDrivenClassPattern()
     */
    protected String mdbClassPattern;


    /**
     * Describe what the MdbSubTask constructor does
     */
    public MdbSubTask()
    {
        setTemplateURL(getClass().getResource(DEFAULT_TEMPLATE_FILE));
        setDestinationFile(getMessageDrivenClassPattern() + ".java");
        addOfType("javax.ejb.MessageDrivenBean");
        setPackageSubstitutionInheritanceSupported(false);
    }

    /**
     * Returns the configuration parameter for specifying the concrete mdb bean class name pattern. By default the value
     * is used for deciding the concrete mdb bean class name. {0} in the value mean current class's symbolic name which
     * for an EJBean is the EJB name. If nothing explicitly specified by user then "{0}Mdb" is used by default.
     *
     * @return   The MdbClassPattern value
     * @see      #mdbClassPattern
     */
    public String getMessageDrivenClassPattern()
    {
        if (mdbClassPattern != null) {
            return mdbClassPattern;
        }
        else {
            return DEFAULT_MESSAGE_DRIVEN_CLASS_PATTERN;
        }
    }


    /**
     * Sets the Pattern attribute of the MdbSubTask object
     *
     * @param new_pattern  The new Pattern value
     */
    public void setPattern(String new_pattern)
    {
        mdbClassPattern = new_pattern;
    }


    /**
     * Called to validate configuration parameters.
     *
     * @exception XDocletException
     */
    public void validateOptions() throws XDocletException
    {
        super.validateOptions();

        if (getMessageDrivenClassPattern() == null || getMessageDrivenClassPattern().trim().equals("")) {
            throw new XDocletException(Translator.getString(XDocletMessages.class, XDocletMessages.PARAMETER_MISSING_OR_EMPTY, new String[]{"pattern"}));
        }

        if (getMessageDrivenClassPattern().indexOf("{0}") == -1) {
            throw new XDocletException(Translator.getString(XDocletModulesEjbMessages.class, XDocletModulesEjbMessages.PATTERN_HAS_NO_PLACEHOLDER));
        }
    }


    /**
     * Gets the GeneratedFileName attribute of the MdbSubTask object
     *
     * @param clazz                 Describe what the parameter does
     * @return                      The GeneratedFileName value
     * @exception XDocletException
     */
    protected String getGeneratedFileName(XClass clazz) throws XDocletException
    {
        return PackageTagsHandler.packageNameAsPathFor(MdbTagsHandler.getMessageDrivenClassFor(getCurrentClass())) + ".java";
    }


    /**
     * Describe what the method does
     *
     * @exception XDocletException
     */
    protected void engineStarted() throws XDocletException
    {
        System.out.println(Translator.getString(XDocletModulesEjbMessages.class, XDocletModulesEjbMessages.GENERATING_MDB_FOR,
            new String[]{getCurrentClass().getQualifiedName()}));
    }


    /**
     * Describe what the method does
     *
     * @param clazz                 Describe what the parameter does
     * @return                      Describe the return value
     * @exception XDocletException
     */
    protected boolean matchesGenerationRules(XClass clazz) throws XDocletException
    {
        Log log = LogUtil.getLog(MdbSubTask.class, "matchesGenerationRules");

        if (super.matchesGenerationRules(clazz) == false) {
            log.debug("Skip bean " + clazz.getQualifiedName() + " because super.matchesGenerationRules() returned false.");
            return false;
        }

        String generate = getCurrentClass().getDoc().getTagAttributeValue("ejb:bean", "generate", false);

        if ((generate != null) && (generate.equals("false") || generate.equals("no"))) {
            log.debug("Skip mdb class for " + clazz.getQualifiedName() + " because of generate=" + generate + " flag.");
            return false;
        }

        if (EjbTagsHandler.isAConcreteEJBean(getCurrentClass())) {
            return true;
        }
        else {
            log.debug("Skip bean " + clazz.getQualifiedName() + " because it's not a concrete bean.");
            return false;
        }
    }

}
