/***************************************************************************
                     test.c - Test harness for libxbox
                             -------------------
    begin                : Thu Dec 09 23:02:12 BST 2004
    copyright            : (C) 2004 by David Pye
    email                : dmp@davidmpye.dyndns.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <string.h>
#include <stdio.h>
#include <unistd.h>

#include "libxbox.h"
#include "libxbox-eeprom.h"
#include "libxbox-i2c.h"

void printsmcver() {
	char ver[4];
	memset(ver,0x00,4);
	if (xbox_smc_version(ver)) {
		printf("SMC Ver is %s\n",ver);
	}
}

void printinfostring(EEPROMDATA *eeprom) {
	char data[151];
	memset(data,0x00,150);
	xbox_manufacturing_info(eeprom->SerialNumber,data);
	printf("Data is %s\n",data);
}

void print_temps() {
	int cpu=0, mcpx=0;

	if (xbox_temperatures(&cpu, &mcpx)) {
		printf("Temperatures are CPU %i, MCPX %i\n",cpu, mcpx);
	} 
	else printf("Unable to read Xbox temperatures - 1.6 Xbox?");
}

void printfanspeed() {
	int fanspeed=0;
	xbox_fanspeed(&fanspeed);
	printf("Fanspeed is %d\n",fanspeed);
}

void print_chipset_vers() {
	long mcpx=0;
	long nv2a=0;

	xbox_nv2a_version(&nv2a);
	xbox_mcpx_version(&mcpx);

	printf("MCPX version is 0x%08lX\n",mcpx);
	printf("NV2A version is 0x%08lX\n",nv2a);
}

void print_eeprom_chksums(EEPROMDATA *eeprom) {

	printf("Checksum 2 is 0x%8lX\n",*(unsigned long*)eeprom->Checksum2);
	xbox_calc_eeprom_chk2(eeprom);
	printf("Checksum 2 (calculated) is 0x%8lX\n",*(unsigned long*)eeprom->Checksum2);	
	printf("Checksum 3 is 0x%8lX\n",*(unsigned long*)eeprom->Checksum3);
	xbox_calc_eeprom_chk3(eeprom);
	printf("Checksum 3 (calculated) is 0x%8lX\n",*(unsigned long*)eeprom->Checksum3);

}

void print_av_info(EEPROMDATA *eeprom) {

	char encname[15];
	char cname[30];
	char stdname[30];

	XBOX_AVIP_CABLE cable;
	XBOX_TV_ENCODER enc;
	XBOX_VIDEO_STD std;

	xbox_av_cable(&cable);
	xbox_tv_encoder(&enc);
	xbox_video_std(eeprom->VideoStandard, &std);
	
	xbox_tv_encoder_name(enc, encname);
	xbox_av_cable_name(cable, cname);
	xbox_video_std_name(std, stdname);

	
	printf("Encoder is %s\n",encname);
	printf("Cable is %s\n",cname);
	printf("Standard is %s\n",stdname);
}

void print_xbe_region(EEPROMDATA *eeprom) {
	XBOX_XBE_REGION region;
	xbox_xbe_region(eeprom->XBERegion, &region);

	printf("XBE region is ");
	switch (region) {
		case EURO_AUSTRALIA:
			printf("Europe\n");
			break;
		case JAPAN:
			printf("Japan\n");
			break;
		case NORTH_AMERICA:
			printf("North America\n");
			break;
		default:
			printf("Unknown\n");
	}
}

int main() {
	EEPROMDATA eeprom;
	printf("Reading eeprom\n");
	if (xbox_read_eeprom(&eeprom)) printf("Successful\n");
	else printf("Failed\n");

	printf("Decrypting eeprom\n");
	if (xbox_decrypt_eeprom(&eeprom)) printf("Decrypt successful\n");
	else printf("Failed\n");

	print_eeprom_chksums(&eeprom);
	
	print_xbe_region(&eeprom);

	printsmcver();
	print_chipset_vers();
	
	print_temps();
	printfanspeed();

	print_av_info(&eeprom);
	
	printinfostring(&eeprom);
	return 0;
}
