/*
 * This file is part of libtrace
 *
 * Copyright (c) 2007,2008 The University of Waikato, Hamilton, New Zealand.
 * Authors: Daniel Lawson 
 *          Perry Lorier 
 *          
 * All rights reserved.
 *
 * This code has been developed by the University of Waikato WAND 
 * research group. For further information please see http://www.wand.net.nz/
 *
 * libtrace is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libtrace is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libtrace; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: libtrace.h 773 2006-05-01 12:58:09Z perry $
 *
 */

#ifndef LIBTRACE_H
#define LIBTRACE_H

/** @file
 *
 * @brief Trace file processing library header
 *
 * @author Daniel Lawson
 * @author Perry Lorier
 *
 * @version $Id: libtrace.h 773 2006-05-01 12:58:09Z perry $
 *
 * This library provides a per packet interface into a trace file, or a live 
 * captures.  It supports ERF, DAG cards, WAG cards, WAG's event format,
 * pcap etc.
 *
 * @par Usage
 * See the example/ directory in the source distribution for some simple examples
 * @par Linking
 * To use this library you need to link against libtrace by passing -ltrace
 * to your linker. You may also need to link against a version of libpcap 
 * and of zlib which are compiled for largefile support (if you wish to access
 * traces larger than 2 GB). This is left as an exercise for the reader. Debian
 * Woody, at least, does not support large file offsets.
 *
 */

#include <sys/types.h>
#ifndef WIN32
#include <sys/time.h>
#endif

#ifdef _MSC_VER
    /* define the following from MSVC's internal types */
    typedef		__int8	int8_t;
    typedef		__int16	int16_t;
    typedef		__int32	int32_t;
    typedef		__int64	int64_t;
    typedef unsigned 	__int8	uint8_t;
    typedef unsigned	__int16 uint16_t;
    typedef unsigned	__int32 uint32_t;
    typedef unsigned	__int64 uint64_t;
    #ifdef BUILDING_DLL
	#define DLLEXPORT __declspec(dllexport)
    #else
	#define DLLEXPORT __declspec(dllimport)
    #endif
    #define DLLLOCAL
    /* Windows pads bitfields out to to the size of their parent type
     * however gcc warns that this doesn't meet with the iso C specification
     * so produces warnings for this behaviour.  sigh.
     */
    #define LT_BITFIELD8 	uint8_t
    #define LT_BITFIELD16 	uint16_t
    #define LT_BITFIELD32 	uint32_t
    #define LT_BITFIELD64 	uint64_t
#else
#   include <stdint.h>
    #ifdef HAVE_GCCVISIBILITYPATCH
	#define DLLEXPORT __attribute__ (visibility("default"))
	#define DLLLOCAL __attribute__ (visibility("hidden"))
    #else
	#define DLLEXPORT
	#define DLLLOCAL
    #endif
    /* GCC warns if the bitfield type is not "unsigned int", however windows
     * generates incorrect code for this (see above), so we define these
     * macros.  How Hidious.  So much for C's portability.
     */
    #define LT_BITFIELD8 	unsigned int
    #define LT_BITFIELD16 	unsigned int
    #define LT_BITFIELD32 	unsigned int
    #define LT_BITFIELD64 	unsigned int
#endif

#ifdef WIN32
#   include <winsock2.h>
#   include <ws2tcpip.h>
    typedef short sa_family_t;
    /* Make up for a lack of stdbool.h */
#    define bool signed char
#    define false 0
#    define true 1
#    if !defined(ssize_t)
     /* XXX: Not 64-bit safe! */
#    define ssize_t int
#    endif    
#else
#    include <netinet/in.h>

#ifndef __cplusplus
#    include <stdbool.h>
#endif

#    include <sys/types.h>
#    include <sys/socket.h>
#endif

/** API version as 2 byte hex digits, eg 0xXXYYZZ */
#define LIBTRACE_API_VERSION \
            ((3<<16)|(0<<8)|(3))

#define LIBTRACE_SVN_REVISION 0
#define DAG_DRIVER_V ""
    
#ifdef __cplusplus 
extern "C" { 
#endif

/* Function does not depend on anything but its
 * parameters, used to hint gcc's optimisations
 */
#if __GNUC__ >= 3 
#  define DEPRECATED __attribute__((deprecated))
#  define SIMPLE_FUNCTION __attribute__((pure))
#  define UNUSED __attribute__((unused))
#  define PACKED __attribute__((packed))
#  define PRINTF(formatpos,argpos) __attribute__((format(printf,formatpos,argpos)))
#else
#  define DEPRECATED
#  define SIMPLE_FUNCTION
#  define UNUSED
#  define PACKED 
#  define PRINTF(formatpos,argpos) 
#endif
	
/** Opaque structure holding information about an output trace */
typedef struct libtrace_out_t libtrace_out_t;
	
/** Opaque structure holding information about a trace */
typedef struct libtrace_t libtrace_t;
        
/** Opaque structure holding information about a bpf filter */
typedef struct libtrace_filter_t libtrace_filter_t;

/** If a packet has memory allocated
 * If the packet has allocated it's own memory it's buffer_control should
 * be TRACE_CTRL_PACKET, when the packet is destroyed it's memory will be
 * free()'d.  If it's doing zerocopy out of memory owned by something else
 * it should be TRACE_CTRL_EXTERNAL.
 * @note the letters p and e are magic numbers used to detect if the packet
 * wasn't created properly
 */
typedef enum {
	TRACE_CTRL_PACKET='p',
	TRACE_CTRL_EXTERNAL='e' 
} buf_control_t;
/** The size of a packet's buffer when managed by libtrace */
#define LIBTRACE_PACKET_BUFSIZE 65536

/** libtrace error information */
typedef struct trace_err_t{
	int err_num; 		/**< error code */
	char problem[255];	/**< the format, uri etc that caused the error for reporting purposes */
} libtrace_err_t;

/** Enumeration of error codes */
enum {
	/** No Error has occured.... yet. */
	TRACE_ERR_NOERROR 	= 0,
	/** The URI passed to trace_create() is unsupported, or badly formed */
	TRACE_ERR_BAD_FORMAT 	= -1,
	/** The trace failed to initialise */
	TRACE_ERR_INIT_FAILED 	= -2,
	/** Unknown config option */
	TRACE_ERR_UNKNOWN_OPTION= -3,
	/** This output uri cannot write packets of this type */
	TRACE_ERR_NO_CONVERSION = -4,
	/** This packet is corrupt, or unusable for the action required */ 
	TRACE_ERR_BAD_PACKET	= -5,
	/** Option known, but unsupported by this format */
	TRACE_ERR_OPTION_UNAVAIL= -6,
	/** This feature is unsupported */
	TRACE_ERR_UNSUPPORTED	= -7,
	/** Illegal use of the API */
	TRACE_ERR_BAD_STATE	= -8
};

/** Enumeration of DLT supported by libtrace 
 */
typedef enum {
	TRACE_DLT_NULL = 0,
	TRACE_DLT_EN10MB = 1,
	TRACE_DLT_PPP = 9,
	TRACE_DLT_ATM_RFC1483 = 11,
	/* Sigh. This is handled in files with LINKTYPE's */
#ifdef __OpenBSD__
	TRACE_DLT_RAW = 14,	
#else
	TRACE_DLT_RAW = 12,
#endif
	TRACE_DLT_LINKTYPE_RAW = 101,
	TRACE_DLT_C_HDLC = 104,
	TRACE_DLT_IEEE802_11 = 105,
	TRACE_DLT_LINUX_SLL = 113,
	TRACE_DLT_PFLOG = 117,
	TRACE_DLT_IEEE802_11_RADIO = 127 /**< Radiotap */
} libtrace_dlt_t ;

/** Enumeration of link layer types supported by libtrace */
typedef enum { 
    /* TRACE_TYPE_LEGACY = 0 		Obsolete */
       TRACE_TYPE_HDLC_POS = 1, 
       TRACE_TYPE_ETH = 2,		/**< 802.3 style Ethernet */
       TRACE_TYPE_ATM = 3,		/**< ATM frame */
       TRACE_TYPE_80211 = 4,		/**< 802.11 frames */
       TRACE_TYPE_NONE = 5,		/**< Raw IP frames */
       TRACE_TYPE_LINUX_SLL = 6,	/**< Linux "null" framing */
       TRACE_TYPE_PFLOG = 7,		/**< FreeBSD's PFlug */
    /* TRACE_TYPE_LEGACY_DEFAULT	Obsolete */
       TRACE_TYPE_POS = 9,
    /* TRACE_TYPE_LEGACY_ATM 		Obsolete */
    /* TRACE_TYPE_LEGACY_ETH 		Obsolete */
       TRACE_TYPE_80211_PRISM = 12,
       TRACE_TYPE_AAL5 = 13,
       TRACE_TYPE_DUCK = 14,	     /**< Pseudo link layer for DUCK packets */
       TRACE_TYPE_80211_RADIO = 15,  /**< Radiotap + 802.11 */
       TRACE_TYPE_LLCSNAP = 16,      /**< Raw LLC/SNAP */
       TRACE_TYPE_PPP = 17,	     /**< PPP frames */
       TRACE_TYPE_METADATA = 18	     	/**< WDCAP-style meta-data */
	
} libtrace_linktype_t;

/** RT protocol base format identifiers
 * This is used to say what kind of packet is being sent over the rt protocol
 */ 
enum base_format_t {
        TRACE_FORMAT_ERF          =1,
        TRACE_FORMAT_PCAP         =2,
        TRACE_FORMAT_PCAPFILE     =3,
        TRACE_FORMAT_WAG          =4,
        TRACE_FORMAT_RT           =5,
        TRACE_FORMAT_LEGACY_ATM   =6,
	TRACE_FORMAT_LEGACY_POS	  =7,
	TRACE_FORMAT_LEGACY_ETH   =8,
	TRACE_FORMAT_LINUX_NATIVE =9,
	TRACE_FORMAT_DUCK	  =10,
	TRACE_FORMAT_BPF	  =11,
	TRACE_FORMAT_TSH	  =12,
	TRACE_FORMAT_ATMHDR	  =13,
	TRACE_FORMAT_LEGACY_NZIX  =14
};

/* RT protocol packet types */
typedef enum {
	TRACE_RT_HELLO       	=1, /**< Connection accepted */
	TRACE_RT_START		=2, /**< Request for data transmission to begin 
				    */
	TRACE_RT_ACK		=3, /**< Data acknowledgement */
	TRACE_RT_STATUS		=4, /**< Fifo status packet */
	TRACE_RT_DUCK		=5, /**< Dag duck info packet */
	TRACE_RT_END_DATA	=6, /**< Server is exiting message */
	TRACE_RT_CLOSE		=7, /**< Client is exiting message */
	TRACE_RT_DENY_CONN	=8, /**< Connection has been denied */
	TRACE_RT_PAUSE		=9, /**< Request server to suspend sending data 
				     */
	TRACE_RT_PAUSE_ACK	=10,/**< Server is paused message */
	TRACE_RT_OPTION	 	=11,/**< Option request */
	TRACE_RT_KEYCHANGE	=12,/**< Anonymisation key has changed */ 
	TRACE_RT_DUCK_2_4	=13,/**< Dag 2.4 Duck */
	TRACE_RT_DUCK_2_5 	=14,/**< Dag 2.5 Duck */
	TRACE_RT_LOSTCONN 	=15,/**< Lost connection to server */
	TRACE_RT_SERVERSTART	=16,/**< Reliable server has been restarted */
	TRACE_RT_CLIENTDROP	=17,/**< Reliable client was lost */
	TRACE_RT_METADATA	=18,/**< Packet contains server meta-data */

	TRACE_RT_DATA_SIMPLE	= 1000, /**< Trace types that know their link
					  * type
					  */
	TRACE_RT_DATA_ERF	=TRACE_RT_DATA_SIMPLE+TRACE_FORMAT_ERF, 
	TRACE_RT_DATA_WAG	=TRACE_RT_DATA_SIMPLE+TRACE_FORMAT_WAG, 
	TRACE_RT_DATA_LEGACY_ATM=TRACE_RT_DATA_SIMPLE+TRACE_FORMAT_LEGACY_ATM, 
	TRACE_RT_DATA_LEGACY_POS=TRACE_RT_DATA_SIMPLE+TRACE_FORMAT_LEGACY_POS, 
	TRACE_RT_DATA_LEGACY_ETH=TRACE_RT_DATA_SIMPLE+TRACE_FORMAT_LEGACY_ETH, 
	TRACE_RT_DATA_LINUX_NATIVE=TRACE_RT_DATA_SIMPLE+TRACE_FORMAT_LINUX_NATIVE,
	TRACE_RT_DATA_TSH	=TRACE_RT_DATA_SIMPLE+TRACE_FORMAT_TSH,

	TRACE_RT_DATA_ATMHDR = TRACE_RT_DATA_SIMPLE + TRACE_FORMAT_ATMHDR,
	TRACE_RT_DATA_LEGACY_NZIX=TRACE_RT_DATA_SIMPLE + TRACE_FORMAT_LEGACY_NZIX,
	TRACE_RT_DATA_DLT		= 2000, /**< Pcap doesn't store the
						  * linktype per packet, and
						  * thus we have to store it
						  * in here.  sigh.
						  */
	TRACE_RT_DLT_NULL		=TRACE_RT_DATA_DLT+TRACE_DLT_NULL,
	TRACE_RT_DLT_EN10MB		=TRACE_RT_DATA_DLT+TRACE_DLT_EN10MB,
	TRACE_RT_DLT_IEEE802_11		=TRACE_RT_DATA_DLT+TRACE_DLT_IEEE802_11,
	TRACE_RT_DLT_LINUX_SLL		=TRACE_RT_DATA_DLT+TRACE_DLT_LINUX_SLL,
	TRACE_RT_DLT_PFLOG		=TRACE_RT_DATA_DLT+TRACE_DLT_PFLOG,
	TRACE_RT_DLT_ATM_RFC1483 	=TRACE_RT_DATA_DLT+TRACE_DLT_ATM_RFC1483,
	TRACE_RT_DATA_DLT_END		= 2999,
	TRACE_RT_LAST			= (2<<31)
} libtrace_rt_types_t;

/** The libtrace packet structure, applications shouldn't be 
 * meddling around in here 
 */
typedef struct libtrace_packet_t {
	struct libtrace_t *trace; 	/**< pointer to the trace */
	void *header;			/**< pointer to the framing header */
	void *payload;			/**< pointer to the link layer */
	void *buffer;			/**< allocated buffer */
	libtrace_rt_types_t  type; 	/**< rt protocol type for the packet */
	buf_control_t buf_control; 	/**< who owns the memory */
	int capture_length;		/**< Cached capture length */
	void *l3_header;		/**< Cached l3 header */
	uint16_t l3_ethertype;		/**< Cached l3 ethertype */
} libtrace_packet_t;


/** Trace directions 
 * Note that these are the directions used by convention, more directions 
 * are possible, not just these 3, and that they may not conform to this
 * convention.
 */
typedef enum {
	TRACE_DIR_OUTGOING = 0,		/**< Packets originating "inside" */
	TRACE_DIR_INCOMING = 1,		/**< Packets originating "outside" */
	TRACE_DIR_OTHER	   = 2		/**< Packets with an unknown direction, or one that's unknown */
} libtrace_direction_t;

/** Enumeration of Radiotap fields */
typedef enum {
    TRACE_RADIOTAP_TSFT = 0, /**< Timer synchronisation function, in microseconds (uint64) */
    TRACE_RADIOTAP_FLAGS = 1, /**< Wireless flags (uint8) */
    TRACE_RADIOTAP_RATE = 2, /**< Bitrate in units of 500kbps (uint8) */
    TRACE_RADIOTAP_CHANNEL = 3, /**< Frequency in MHz (uint16) and channel flags (uint16) */
    TRACE_RADIOTAP_FHSS = 4, /**< FHSS hop set (uint8) and hopping pattern (uint8) */
    TRACE_RADIOTAP_DBM_ANTSIGNAL = 5, /**< Signal power in dBm (int8) */
    TRACE_RADIOTAP_DBM_ANTNOISE = 6, /**< Noise power in dBm (int8) */
    TRACE_RADIOTAP_LOCK_QUALITY = 7, /**< Barker Code lock quality (uint16) */
    TRACE_RADIOTAP_TX_ATTENUATION = 8, /**< TX attenuation as unitless distance from max power (uint16) */
    TRACE_RADIOTAP_DB_TX_ATTENUATION = 9, /**< TX attenutation as dB from max power (uint16) */
    TRACE_RADIOTAP_DBM_TX_POWER = 10, /**< TX Power in dBm (int8) */
    TRACE_RADIOTAP_ANTENNA = 11, /**< Antenna frame was rx'd or tx'd on (uint8) */
    TRACE_RADIOTAP_DB_ANTSIGNAL = 12, /**< Signal power in dB from a fixed reference (uint8) */
    TRACE_RADIOTAP_DB_ANTNOISE = 13, /**< Noise power in dB from a fixed reference (uint8) */
    TRACE_RADIOTAP_RX_FLAGS = 14, /** Properties of received frame (uint16) */
    TRACE_RADIOTAP_TX_FLAGS = 15, /** Properties of transmitted frame (uint16) */
    TRACE_RADIOTAP_RTS_RETRIES = 16, /** Number of rts retries frame used (uint8) */
    TRACE_RADIOTAP_DATA_RETRIES = 17, /** Number of unicast retries a transmitted frame used (uint8) */
    TRACE_RADIOTAP_EXT = 31
} libtrace_radiotap_field_t;


/** @name Protocol structures
 * These convenience structures are here as they are portable ways of dealing
 * with various protocols.
 * @{
 */

#ifdef WIN32
#pragma pack(push)
#pragma pack(1)
#endif

/** Generic IP header structure */
typedef struct libtrace_ip
{
#if BYTE_ORDER == LITTLE_ENDIAN
    LT_BITFIELD8 ip_hl:4;		/**< Header Length */
    LT_BITFIELD8 ip_v:4;		/**< Version */
#elif BYTE_ORDER == BIG_ENDIAN
    LT_BITFIELD8 ip_v:4;		/**< Version */
    LT_BITFIELD8 ip_hl:4;		/**< Header Length */
#else
#   error "Adjust your <bits/endian.h> defines"
#endif
    uint8_t  ip_tos;			/**< Type of Service */
    uint16_t ip_len;			/**< Total Length */
    int16_t  ip_id;			/**< Identification */
    uint16_t ip_off;			/**< IP Fragment offset (and flags) */
    uint8_t  ip_ttl;			/**< Time to Live */
    uint8_t  ip_p;			/**< Protocol */
    uint16_t ip_sum;			/**< Checksum */
    struct in_addr ip_src;		/**< Source Address */
    struct in_addr ip_dst;		/**< Destination Address */
} PACKED libtrace_ip_t;

/** IPv6 header extension structure */
typedef struct libtrace_ip6_ext
{
	uint8_t nxt;
	uint8_t len;
} PACKED libtrace_ip6_ext_t;

/** Generic IPv6 header structure */
typedef struct libtrace_ip6
{ 
    uint32_t flow;
    uint16_t plen;			/**< Payload length */
    uint8_t nxt;			/**< Next header */
    uint8_t hlim;			/**< Hop limit */
    struct in6_addr ip_src;		/**< source address */
    struct in6_addr ip_dst;		/**< dest address */
} PACKED libtrace_ip6_t;

/** Generic TCP header structure */
typedef struct libtrace_tcp
  {
    uint16_t source;		/**< Source Port */
    uint16_t dest;		/**< Destination port */
    uint32_t seq;		/**< Sequence number */
    uint32_t ack_seq;		/**< Acknowledgement Number */
#  if BYTE_ORDER == LITTLE_ENDIAN
    LT_BITFIELD8 res1:4;	/**< Reserved bits */
    LT_BITFIELD8 doff:4;	/**< Data Offset */	
    LT_BITFIELD8 fin:1;		/**< FIN */
    LT_BITFIELD8 syn:1;		/**< SYN flag */
    LT_BITFIELD8 rst:1;		/**< RST flag */
    LT_BITFIELD8 psh:1;		/**< PuSH flag */
    LT_BITFIELD8 ack:1;		/**< ACK flag */
    LT_BITFIELD8 urg:1;		/**< URG flag */
    LT_BITFIELD8 res2:2;	/**< Reserved */
#  elif BYTE_ORDER == BIG_ENDIAN
    LT_BITFIELD8 doff:4;	/**< Data offset */
    LT_BITFIELD8 res1:4;	/**< Reserved bits */
    LT_BITFIELD8 res2:2;	/**< Reserved */
    LT_BITFIELD8 urg:1;		/**< URG flag */
    LT_BITFIELD8 ack:1;		/**< ACK flag */
    LT_BITFIELD8 psh:1;		/**< PuSH flag */
    LT_BITFIELD8 rst:1;		/**< RST flag */
    LT_BITFIELD8 syn:1;		/**< SYN flag */
    LT_BITFIELD8 fin:1;		/**< FIN flag */
#  else
#   error "Adjust your <bits/endian.h> defines"
#  endif
    uint16_t window;		/**< Window Size */
    uint16_t check;		/**< Checksum */
    uint16_t urg_ptr;		/**< Urgent Pointer */
} PACKED libtrace_tcp_t;

/** Generic UDP header structure */
typedef struct libtrace_udp {
  uint16_t	source;		/**< Source port */
  uint16_t	dest;		/**< Destination port */
  uint16_t	len;		/**< Length */
  uint16_t	check;		/**< Checksum */
} PACKED libtrace_udp_t;

/** Generic ICMP header structure */
typedef struct libtrace_icmp
{
  uint8_t type;		/**< Message Type */
  uint8_t code;		/**< Type Sub-code */
  uint16_t checksum;		/**< Checksum */
  union
  {
    struct
    {
      uint16_t	id;
      uint16_t	sequence;
    } echo;			/**< Echo Datagram */
    uint32_t	gateway;	/**< Gateway Address */
    struct
    {
      uint16_t	unused;
      uint16_t	mtu;
    } frag;			/**< Path MTU Discovery */
  } un;				/**< Union for Payloads of Various ICMP Codes */
} PACKED libtrace_icmp_t;

/** Generic LLC/SNAP header structure */
typedef struct libtrace_llcsnap
{
/* LLC */
  uint8_t dsap;			/**< Destination Service Access Point */
  uint8_t ssap;			/**< Source Service Access Point */
  uint8_t control;
/* SNAP */
  LT_BITFIELD32 oui:24;		/**< Organisationally Unique Identifier (scope)*/
  uint16_t type;		/**< Protocol within OUI */
} PACKED libtrace_llcsnap_t;

/** 802.3 frame */
typedef struct libtrace_ether
{
  uint8_t ether_dhost[6];	/**< Destination Ether Addr */
  uint8_t ether_shost[6];	/**< Source Ether Addr */
  uint16_t ether_type;		/**< Packet Type ID Field (next-header) */
} PACKED libtrace_ether_t;

/** 802.1Q frame */
typedef struct libtrace_8021q 
{
  LT_BITFIELD16 vlan_pri:3;	 /**< VLAN User Priority */
  LT_BITFIELD16 vlan_cfi:1; 	 /**< VLAN Format Indicator, 
				   * 0 for ethernet, 1 for token ring */
  LT_BITFIELD16 vlan_id:12; 	 /**< VLAN Id */
  uint16_t vlan_ether_type;	 /**< VLAN Sub-packet Type ID Field 
				   * (next-header)*/
} PACKED libtrace_8021q_t;

/** ATM User Network Interface (UNI) Cell. */
typedef struct libtrace_atm_cell
{
  LT_BITFIELD32 gfc:4;		/**< Generic Flow Control */
  LT_BITFIELD32 vpi:8;		/**< Virtual Path Identifier */
  LT_BITFIELD32 vci:16;		/**< Virtual Channel Identifier */
  LT_BITFIELD32 pt:3;		/**< Payload Type */
  LT_BITFIELD32 clp:1;		/**< Cell Loss Priority */
  LT_BITFIELD32 hec:8;		/**< Header Error Control */
} PACKED libtrace_atm_cell_t;

/** ATM Network Node/Network Interface (NNI) Cell. */
typedef struct libtrace_atm_nni_cell
{
  LT_BITFIELD32 vpi:12;		/**< Virtual Path Identifier */
  LT_BITFIELD32 vci:16;		/**< Virtual Channel Identifier */
  LT_BITFIELD32 pt:3;		/**< Payload Type */
  LT_BITFIELD32 clp:1;		/**< Cell Loss Priority */
  LT_BITFIELD32 hec:8;		/**< Header Error Control */
} PACKED libtrace_atm_nni_cell_t;

/** Captured UNI cell.
 *
 * Endance don't capture the HEC, presumably to keep alignment.  This 
 * version of the \ref libtrace_atm_cell  is used when dealing with dag
 * captures of uni cells.
 *
 */
typedef struct libtrace_atm_capture_cell
{
  LT_BITFIELD32 gfc:4;		/**< Generic Flow Control */
  LT_BITFIELD32 vpi:8;		/**< Virtual Path Identifier */
  LT_BITFIELD32 vci:16;		/**< Virtual Channel Identifier */
  LT_BITFIELD32 pt:3;		/**< Payload Type */
  LT_BITFIELD32 clp:1;		/**< Cell Loss Priority */
} PACKED libtrace_atm_capture_cell_t;

/** Captured NNI cell.
 *
 * Endance don't capture the HEC, presumably to keep alignment.  This 
 * version of the \ref libtrace_atm_nni_cell  is used when dealing with dag
 * captures of nni cells.
 *
 */
typedef struct libtrace_atm_nni_capture_cell
{
  LT_BITFIELD32 vpi:12;		/**< Virtual Path Identifier */
  LT_BITFIELD32 vci:16;		/**< Virtual Channel Identifier */
  LT_BITFIELD32 pt:3;		/**< Payload Type */
  LT_BITFIELD32 clp:1;		/**< Cell Loss Priority */
  LT_BITFIELD32 hec:8;		/**< Header Error Control */
} PACKED libtrace_atm_nni_capture_cell_t;

/** PPP header */
typedef struct libtrace_pps
{
 uint8_t addres;		/**< PPP Address (0xFF - All stations) */
 uint8_t header;		/**< PPP Control (0x03 - Unnumbered info) */
 uint16_t protocol;		/**< PPP Protocol (htons(0x0021) - IPv4 */
} PACKED libtrace_ppp_t;

/** 802.11 header */
typedef struct libtrace_80211_t {
#if BYTE_ORDER == LITTLE_ENDIAN
        LT_BITFIELD32      protocol:2;
        LT_BITFIELD32      type:2;
        LT_BITFIELD32      subtype:4;
#else
	LT_BITFIELD32	   subtype:4;
	LT_BITFIELD32      type:2;
	LT_BITFIELD32      protocol:2;
#endif

#if BYTE_ORDER == LITTLE_ENDIAN
        LT_BITFIELD32      to_ds:1;	/**< Packet to Distribution Service */
        LT_BITFIELD32      from_ds:1;	/**< Packet from Distribution Service */
        LT_BITFIELD32      more_frag:1;	/**< Packet has more fragments */
        LT_BITFIELD32      retry:1;	/**< Packet is a retry */
        LT_BITFIELD32      power:1;
        LT_BITFIELD32      more_data:1;
        LT_BITFIELD32      wep:1;
        LT_BITFIELD32      order:1;
#else
        LT_BITFIELD32      order:1;
        LT_BITFIELD32      wep:1;
        LT_BITFIELD32      more_data:1;
        LT_BITFIELD32      power:1;
        LT_BITFIELD32      retry:1;	/**< Packet is a retry */
        LT_BITFIELD32      more_frag:1;	/**< Packet has more fragments */
        LT_BITFIELD32      from_ds:1;	/**< Packet from Distribution Service */
        LT_BITFIELD32      to_ds:1;	/**< Packet to Distribution Service */
#endif
        uint16_t     duration;
        uint8_t      mac1[6];
        uint8_t      mac2[6];
        uint8_t      mac3[6];
        uint16_t     SeqCtl;
        uint8_t      mac4[6];
} PACKED libtrace_80211_t;

/** The Radiotap header pre-amble
 *
 * All Radiotap headers start with this pre-amble, followed by the fields
 * specified in the it_present bitmask. If bit 31 of it_present is set, then
 * another bitmask follows.
 * @note All of the radiotap data fields are in little-endian byte-order.
 */
typedef struct libtrace_radiotap_t {
    uint8_t     it_version; /**< Radiotap version */
    uint8_t     it_pad; /**< Padding for natural alignment */
    uint16_t    it_len; /**< Length in bytes of the entire Radiotap header */
    uint32_t    it_present; /**< Which Radiotap fields are present */
} PACKED libtrace_radiotap_t;


#ifdef WIN32
#pragma pack(pop)
#endif


/*@}*/

/** Prints help information for libtrace 
 *
 * Function prints out some basic help information regarding libtrace,
 * and then prints out the help() function registered with each input module
 */
DLLEXPORT void trace_help(void);

/** @name Trace management
 * These members deal with creating, configuring, starting, pausing and
 * cleaning up a trace object
 *@{
 */

/** Create a trace file from a URI
 * 
 * @param uri containing a valid libtrace URI
 * @return an opaque pointer to a libtrace_t
 *
 * Valid URI's are:
 *  - erf:/path/to/erf/file
 *  - erf:-  (stdin)
 *  - dag:/dev/dagcard                  
 *  - pcapint:pcapinterface                (eg: pcap:eth0)
 *  - pcap:/path/to/pcap/file
 *  - pcap:-
 *  - rt:hostname
 *  - rt:hostname:port
 *  - rtclient:hostname	(deprecated)
 *  - rtclient:hostname:port (deprecated)
 *
 *  If an error occured when attempting to open the trace file, an error
 *  trace is returned and trace_get_error should be called to find out
 *  if an error occured, and what that error was.  The trace is created in the
 *  configuration state, you must call trace_start to start the capture.
 */
DLLEXPORT libtrace_t *trace_create(const char *uri);

/** Creates a "dummy" trace file that has only the format type set.
 *
 * @return an opaque pointer to a (sparsely initialised) libtrace_t
 *
 * IMPORTANT: Do not attempt to call trace_read_packet or other such functions
 * with the dummy trace. Its intended purpose is to act as a packet->trace for
 * libtrace_packet_t's that are not associated with a libtrace_t structure.
 */
DLLEXPORT libtrace_t *trace_create_dead(const char *uri);

/** Creates a trace output file from a URI. 
 *
 * @param uri	the uri string describing the output format and destination
 * @return an opaque pointer to a libtrace_output_t
 *
 * Valid URI's are:
 *  - erf:/path/to/erf/file
 *  - pcap:/path/to/pcap/file
 *  - wtf:/path/to/wtf/file
 *
 *  If an error occured when attempting to open the output trace, NULL is returned 
 *  and trace_errno is set. Use trace_perror_output() to get more information
 */
DLLEXPORT libtrace_out_t *trace_create_output(const char *uri);

/** Start the capture
 * @param libtrace	The trace to start
 * @return 0 on success, -1 on failure
 *
 * This does the actual work with starting the trace capture, and applying
 * all the config options.  This may fail.
 */
DLLEXPORT int trace_start(libtrace_t *libtrace);

/** Pause the capture
 * @param libtrace	The trace to pause
 * @return 0 on success, -1 on failure
 *
 * This stops a capture in progress and returns you to the configuration
 * state.  Any packets that arrive after trace_pause() has been called
 * will be discarded.  To resume capture, call trace_start().
 */
DLLEXPORT int trace_pause(libtrace_t *libtrace);

/** Start an output trace
 * @param libtrace	The trace to start
 * @return 0 on success, -1 on failure
 *
 * This does the actual work with starting a trace for write.  This generally
 * creates the file.
 */
DLLEXPORT int trace_start_output(libtrace_out_t *libtrace);

/** Valid trace capture options */
typedef enum {
	TRACE_OPTION_SNAPLEN, 	/**< Number of bytes captured */
	TRACE_OPTION_PROMISC, 	/**< Capture packets to other hosts */
	TRACE_OPTION_FILTER,  	/**< Apply this filter to all packets */
	TRACE_OPTION_META_FREQ,	/**< Frequency of meta-data information, e.g. DUCK packets */
	/** trace_event ignores gaps between packets when reading traces off disk */
	TRACE_OPTION_EVENT_REALTIME
} trace_option_t;

/** Sets an input config option
 * @param libtrace	the trace object to apply the option to
 * @param option	the option to set
 * @param value		the value to set the option to
 * @return -1 if option configuration failed, 0 otherwise
 * This should be called after trace_create, and before trace_start
 */
DLLEXPORT int trace_config(libtrace_t *libtrace,
		trace_option_t option,
		void *value);

typedef enum {
	TRACE_OPTION_OUTPUT_FILEFLAGS, /**< File flags to open the trace file
					* with.  eg O_APPEND
					*/
	TRACE_OPTION_OUTPUT_COMPRESS   /**< Compression level, eg 6. */
} trace_option_output_t;

/** Sets an output config option
 *
 * @param libtrace	the output trace object to apply the option to
 * @param option	the option to set
 * @param value		the value to set the option to
 * @return -1 if option configuration failed, 0 otherwise
 * This should be called after trace_create_output, and before 
 * trace_start_output
 */
DLLEXPORT int trace_config_output(libtrace_out_t *libtrace, 
		trace_option_output_t option,
		void *value
		);

/** Close a trace file, freeing up any resources it may have been using
 *
 */
DLLEXPORT void trace_destroy(libtrace_t *trace);

/** Close a trace file, freeing up any resources it may have been using
 * @param trace		trace file to be destroyed
 */
DLLEXPORT void trace_destroy_dead(libtrace_t *trace);

/** Close a trace output file, freeing up any resources it may have been using
 * @param trace		the output trace file to be destroyed
 */
DLLEXPORT void trace_destroy_output(libtrace_out_t *trace);

/** Check (and clear) the current error state of an input trace
 * @param trace		the trace file to check the error state on
 * @return Error report
 * This reads and returns the current error state and sets the current error 
 * to "no error".
 */
DLLEXPORT libtrace_err_t trace_get_err(libtrace_t *trace);

/** Return if there is an error
 * @param trace		the trace file to check the error state on
 * This does not clear the error status, and only returns true or false.
 */
DLLEXPORT bool trace_is_err(libtrace_t *trace);

/** Output an error message to stderr and clear the error status.
 * @param trace		the trace with the error to output
 * @param msg		the message to prefix to the error
 * This function does clear the error status.
 */
DLLEXPORT void trace_perror(libtrace_t *trace, const char *msg,...) PRINTF(2,3);

/** Check (and clear) the current error state of an output trace
 * @param trace		the output trace file to check the error state on
 * @return Error report
 * This reads and returns the current error state and sets the current error 
 * to "no error".
 */
DLLEXPORT libtrace_err_t trace_get_err_output(libtrace_out_t *trace);

/** Return if there is an error
 * @param trace		the trace file to check the error state on
 * This does not clear the error status, and only returns true or false.
 */
DLLEXPORT bool trace_is_err_output(libtrace_out_t *trace);

/** Output an error message to stderr and clear the error status.
 * @param trace		the trace with the error to output
 * @param msg		the message to prefix to the error
 * This function does clear the error status.
 */
DLLEXPORT void trace_perror_output(libtrace_out_t *trace, const char *msg,...)
	PRINTF(2,3);

/** Return the number of captured packets 
 * Includes the number of packets counted as early as possible, before
 * filtering, and includes dropped packets.
 *
 * @param trace		Trace to examine
 * @returns number of packets seen at the capture point before filtering.
 *
 * If this is not known, this will return UINT64_MAX
 */
uint64_t trace_get_received_packets(libtrace_t *trace);

/** Return the number of filtered packets
 * Returns the number of packets that were captured, but discarded for not
 * matching a trace filter.  This includes packets 
 *
 * @param trace		Trace file to examine
 * @returns the number of packets that were successfully captured, but filtered
 *
 * If this is not known, this will return UINT64_MAX
 */
uint64_t trace_get_filtered_packets(libtrace_t *trace);

/** Return the number of packets that have been dropped for lack of packets
 * @param trace		Trace file to examine
 * @returns The number of packets captured, but dropped due to buffer overruns
 */
uint64_t trace_get_dropped_packets(libtrace_t *trace);

/** Return the number of packets that have been returned to library user
 * @param trace		Trace file to examine
 * @returns The number of packets returned to the user of the library.
 */
uint64_t trace_get_accepted_packets(libtrace_t *trace);


/*@}*/

/** @name Reading/Writing packets
 * These members deal with creating, reading and writing packets
 *
 * @{
 */

/** Create a new packet object
 *
 * @return a pointer to an initialised libtrace_packet_t object
 */
DLLEXPORT libtrace_packet_t *trace_create_packet(void);

/** Copy a packet
 * @param packet	the source packet to copy
 * @return a new packet which has the same content as the source packet
 * @note This always involves a copy, which can be slow.  Use of this 
 * function should be avoided where possible.
 * @par The reason you would want to use this function is that a zerocopied
 * packet from a device is using the devices memory which may be a limited
 * resource.  Copying the packet will cause it to be copied into the systems
 * memory.
 */
DLLEXPORT libtrace_packet_t *trace_copy_packet(const libtrace_packet_t *packet);

/** Destroy a packet object
 *
 * sideeffect: sets packet to NULL
 */
DLLEXPORT void trace_destroy_packet(libtrace_packet_t *packet);


/** Read one packet from the trace 
 *
 * @param trace 	the libtrace opaque pointer
 * @param packet  	the packet opaque pointer
 * @return 0 on EOF, negative value on error, number of bytes read when
 * successful.
 *
 * @note the number of bytes read is usually (but not always) the same as
 * trace_get_framing_length()+trace_get_capture_length() depending on the
 * trace format.
 * @note the trace must have been started with trace_start before calling
 * this function
 */
DLLEXPORT int trace_read_packet(libtrace_t *trace, libtrace_packet_t *packet);

/** Event types 
 * see \ref libtrace_eventobj_t and \ref trace_event
 */
typedef enum {
	TRACE_EVENT_IOWAIT,	/**< Need to block on fd */
	TRACE_EVENT_SLEEP,	/**< Sleep for some time */
	TRACE_EVENT_PACKET,	/**< packet has arrived */
	TRACE_EVENT_TERMINATE	/**< End of trace */
} libtrace_event_t;

/** Structure returned by libtrace_event explaining what the current event is */
typedef struct libtrace_eventobj_t {
	libtrace_event_t type; /**< event type (iowait,sleep,packet) */
	int fd;		       /**< if IOWAIT, the fd to sleep on */
	double seconds;	       /**< if SLEEP, the amount of time to sleep for 
				*/
	int size; 	       /**< if PACKET, the value returned from 
				*  trace_read_packet 
				*/
} libtrace_eventobj_t;

/** Processes the next libtrace event
 * @param trace the libtrace opaque pointer
 * @param packet the libtrace_packet opaque pointer
 * @return libtrace_event struct containing the type, and potential
 * 	fd or seconds to sleep on
 *
 * Type can be:
 *  TRACE_EVENT_IOWAIT	Waiting on I/O on fd
 *  TRACE_EVENT_SLEEP	Next event in seconds
 *  TRACE_EVENT_PACKET	Packet arrived in buffer with size size
 *  TRACE_EVENT_TERMINATE Trace terminated (perhaps with an error condition)
 */
DLLEXPORT libtrace_eventobj_t trace_event(libtrace_t *trace,
		libtrace_packet_t *packet);


/** Write one packet out to the output trace
 *
 * @param trace		the libtrace_out opaque pointer
 * @param packet	the packet opaque pointer
 * @return the number of bytes written out, if zero or negative then an error has occured.
 */
DLLEXPORT int trace_write_packet(libtrace_out_t *trace, libtrace_packet_t *packet);
/*@}*/

/** @name Protocol decodes
 * These functions locate and return a pointer to various headers inside a
 * packet
 * @{
 */


/** Gets a pointer to the first byte of the packet as it was captured and
 * returns its corresponding linktype and capture length.
 *
 * Use this function instead of the deprecated trace_get_link().
 *
 * @param packet the packet pointer
 * @param[out] linktype the linktype of the returned pointer
 * @param[out] remaining the capture length (the number of captured bytes from
 * the returned pointer)
 * @return a pointer to the first byte of the packet
 */
DLLEXPORT void *trace_get_packet_buffer(const libtrace_packet_t *packet,
                libtrace_linktype_t *linktype, uint32_t *remaining);

/** get a pointer to the link layer
 * @param packet  	the packet opaque pointer
 *
 * @return a pointer to the link layer, or NULL if there is no link layer
 *
 * @deprecated This function is deprecated: Use trace_get_packet_buffer() or
 * one of the trace_get_layer*() functions instead.
 * @note you should call trace_get_link_type to find out what type of link
 * layer this is
 */
DLLEXPORT SIMPLE_FUNCTION DEPRECATED
void *trace_get_link(const libtrace_packet_t *packet);

/** get a pointer to the IPv4 header (if any)
 * @param packet  	the packet opaque pointer
 *
 * @return a pointer to the IPv4 header, or NULL if there is no IPv4 header
 *
 * You should consider using \ref trace_get_layer3 instead of this function.
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_ip_t *trace_get_ip(libtrace_packet_t *packet);

/** get a pointer to the IPv6 header (if any)
 * @param packet  	the packet opaque pointer
 *
 * @return a pointer to the IPv6 header, or NULL if there is no IPv6 header
 *
 * You should consider using \ref trace_get_layer3 instead of this function.
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_ip6_t *trace_get_ip6(libtrace_packet_t *packet);

/** Return a pointer to the first metadata header in a packet, if present.
 *
 * This function takes a pointer to a libtrace packet and if any metadata
 * headers exist, returns a pointer to the first one, along with its
 * corresponding linktype. 
 *
 * If no metadata headers exist in the packet, NULL is returned.
 *
 * A metadata header is a header that was prepended by the capturing device,
 * such as a Linux SLL header, or a Radiotap wireless monitoring header.
 * Subsequent metadata headers may be accessed with the
 * trace_get_payload_from_meta(...) function. 
 *
 * @param packet the libtrace packet
 * @param[out] linktype the linktype of the returned metadata header
 * @param[out] remaining the number of bytes captured after the returned
 * pointer.
 * @return a pointer to the first metadata header, or NULL if there are no
 * metadata headers present.
 *
 * remaining may be NULL, however linktype must be provided.
 */
DLLEXPORT void *trace_get_packet_meta(const libtrace_packet_t *packet,
                libtrace_linktype_t *linktype,
                uint32_t *remaining);

/** Returns the payload of a metadata header.
 * 
 * This function takes a pointer to the start of a metadata header (either
 * obtained via trace_get_packet_meta(...) or by a previous call to
 * trace_get_payload_from_meta(...)) along with its corresponding linktype and
 * returns the payload, i.e. the next header. It will also update the linktype
 * parameter to indicate the type of payload.
 *  
 * If the linktype indicates that the header passed in is not a metadata
 * header, the function returns NULL to indicate this. The linktype remains
 * unchanged in this case.
 *
 * This function allows the user to iterate through metadata headers which are
 * sometimes present before the actual packet as it was received on the wire.
 * Examples of metadata headers include the Linux SLL header and the Radiotap
 * wireless monitoring header.
 *
 * @param[in] meta a pointer to a header
 * @param[in,out] linktype the linktype of meta (updated to indicate the
 * linktype of the returned header if applicable).
 * @param[in,out] remaining the number of bytes after the meta pointer.
 * @return a pointer to the payload of the metadata header. If meta is not a
 * pointer to a metadata header, NULL is returned and linktype remains
 * unchanged.
 *
 * All parameters are mandatory. NULL will be returned if any are NULL.
 */
DLLEXPORT void *trace_get_payload_from_meta(const void *meta,
                libtrace_linktype_t *linktype,
                uint32_t *remaining);


/** Get a pointer to the layer 2 header. Generally this is the first byte of the
 * packet as it was seen on the wire.
 * 
 * This function takes a libtrace packet and skips over any metadata headers if
 * present (such as Linux SLL or Radiotap) and returns a pointer to the first
 * byte of the packet that was actually received by the network interface.
 *
 * @param packet the libtrace packet
 * @param[out] linktype the linktype of the returned layer 2 header
 * @param[out] remaining the number of bytes left in the packet after the
 * returned pointer.
 * @return a pointer to the first byte of the packet as it was seen on the
 * wire.
 *
 * remaining may be NULL, otherwise it will be filled in by the function.
 */
DLLEXPORT void *trace_get_layer2(const libtrace_packet_t *packet,
                libtrace_linktype_t *linktype,
                uint32_t *remaining);

/** Gets a pointer to the next header given a pointer to a layer2 header
 *
 * @param l2            	The pointer to the current layer2 header
 * @param linktype		The type of the layer2 header
 * @param[out] ethertype 	An optional output variable of the ethernet type
 * @param[in,out] remaining 	Optionally updated with the length remaining
 *
 * @return a pointer to the transport layer header, or NULL if header isn't
 * present.
 *
 * type may be NULL if not needed.
 *
 * Remaining may be NULL.  If Remaining is not NULL it must point to the number
 * of bytes captured of the layer2 header and beyond.  It will be decremented
 * by the number of bytes skipped to find the payload.
 *
 */
DLLEXPORT void *trace_get_payload_from_layer2(void *l2,
		libtrace_linktype_t linktype,
		uint16_t *ethertype,
		uint32_t *remaining);


/** Get a pointer to the layer 3 header.
 * @param packet  		The packet opaque pointer
 * @param[out] ethertype	The ethertype of the layer 3 header
 * @param[out] remaining	The amount of space available after this header
 * 				has been removed.
 *
 * @return a pointer to the layer 3 header.
 * remaining may be NULL, otherwise it will be set to the number of captured
 * bytes after the pointer returned.
 */
DLLEXPORT 
void *trace_get_layer3(const libtrace_packet_t *packet,
		uint16_t *ethertype, uint32_t *remaining);

/** Gets a pointer to the transport layer header (if any)
 * @param packet        a pointer to a libtrace_packet structure
 * @param[out] proto	transport layer protocol
 *
 * @return a pointer to the transport layer header, or NULL if there is no
 * header
 *
 * @note proto may be NULL if proto is unneeded.
 */
DLLEXPORT void *trace_get_transport(const libtrace_packet_t *packet, 
		uint8_t *proto, uint32_t *remaining);

/** Gets a pointer to the payload given a pointer to the IP header
 * @param ip            The IP Header
 * @param[out] proto	An output variable of the IP protocol
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the transport layer header, or NULL if header isn't
 * present.
 *
 * Remaining may be NULL.  If Remaining is not NULL it must point to the number
 * of bytes captured of the IP header and beyond.  It will be decremented by
 * the length of the IPv4 header (including any options).
 *
 * proto may be NULL if not needed.
 *
 * @note This is similar to trace_get_transport_from_ip in libtrace2
 */
DLLEXPORT void *trace_get_payload_from_ip(libtrace_ip_t *ip, uint8_t *proto,
		uint32_t *remaining);

/** Gets a pointer to the payload given a pointer to the IPv6 header
 * @param ipptr         The IPv6 Header
 * @param[out] proto    An output variable of the protocol of the next header
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the transport layer header, or NULL if the IPv6 header
 * isn't complete.
 *
 * Remaining may be NULL.  If Remaining is not NULL it must point to the number
 * of bytes captured of the IP header and beyond.  It will be decremented by
 * this function by the length of the IPV6 header.
 *
 * proto may be NULL if not needed.
 *
 */
DLLEXPORT void *trace_get_payload_from_ip6(libtrace_ip6_t *ipptr,
                uint8_t *prot, uint32_t *remaining);

/** Gets a pointer to the payload given a pointer to the link header
 * @param ip            The link pointer
 * @param[out] type	An output variable of the ethernet type
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the transport layer header, or NULL if header isn't
 * present.
 *
 * Remaining may be NULL.  If Remaining is not NULL it must point to the number
 * of bytes captured of the linklayer and beyond.  It will be updated after
 * this function to the number of bytes remaining after the IP header (and any
 * IP options) have been removed.
 *
 * type may be NULL if not needed.
 *
 */
DLLEXPORT void *trace_get_payload_from_link(void *linkptr,
		libtrace_linktype_t linktype, 
		uint16_t *type, uint32_t *remaining);

/** Skips over any 802.1q headers, if present.
 * @param ethernet      A pointer to the payload following an ethernet header
 * -usually the result of calling trace_get_payload_from_link
 * @param[in,out] type  The ethernet type, replaced with the vlan ether type
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the header beyond the vlan header, if present.
 * Otherwise, returns the ethernet payload that was passed in
 *
 * Remaining may be NULL. If Remaining is not NULL it must point to the number
 * of bytes captured past (but not including) the link layer. This function
 * will decrement it by the length of the 802.1q headers if present.
 *
 * Type must point to the value of the ethernet type. Libtrace will assert
 * fail if type is NULL.
 *
 */
DLLEXPORT void *trace_get_vlan_payload_from_ethernet_payload(
                void *ethernet_payload, uint16_t *type, uint32_t *remaining);

/** Gets a pointer to the payload given a pointer to a tcp header
 * @param tcp           The tcp Header
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the tcp payload, or NULL if the payload isn't present.
 *
 * Remaining may be NULL.  If remaining is not NULL it must point to the number
 * of bytes captured of the TCP header and beyond.  It will be decremented by
 * this function by the length of the TCP header (including any options).
 *
 */
DLLEXPORT void *trace_get_payload_from_tcp(libtrace_tcp_t *tcp, 
		uint32_t *remaining);

/** Gets a pointer to the payload given a pointer to a udp header
 * @param udp           The udp Header
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the udp payload, or NULL if the payload isn't present.
 *
 * Remaining may be NULL.  If Remaining is not NULL it must point to the number
 * of bytes captured of the UDP header and beyond.  It will be decremented by
 * this function to the number of bytes remaining after the UDP header.
 *
 */
DLLEXPORT void *trace_get_payload_from_udp(libtrace_udp_t *udp, uint32_t *remaining);

/** Gets a pointer to the payload given a pointer to a icmp header
 * @param icmp          The icmp Header
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the icmp payload, or NULL if the payload isn't present.
 *
 * Remaining may be NULL.  If remaining is not NULL it must point to the number
 * of bytes captured of the ICMP header and beyond.  It will be decremented
 * by the number of bytes in the ICMP header.
 * 
 */
DLLEXPORT void *trace_get_payload_from_icmp(libtrace_icmp_t *icmp, 
		uint32_t *remaining);

/** get a pointer to the TCP header (if any)
 * @param packet  	the packet opaque pointer
 *
 * @return a pointer to the TCP header, or NULL if there is not a TCP packet
 *
 * @note you should probably use trace_get_transport()
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_tcp_t *trace_get_tcp(libtrace_packet_t *packet);

/** get a pointer to the TCP header (if any) given a pointer to the IP header
 * @param ip		The IP header
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the TCP header, or NULL if this is not a TCP packet
 *
 * Remaining may be NULL.  If Remaining is not NULL it must point to the number
 * of bytes captured of the TCP header and beyond.  It will be decremented by
 * the number of bytes in the TCP header (including any TCP options).
 *
 * @note The last parameter has changed from libtrace2
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_tcp_t *trace_get_tcp_from_ip(libtrace_ip_t *ip, uint32_t *remaining);

/** get a pointer to the UDP header (if any)
 * @param packet  	the packet opaque pointer
 *
 * @return a pointer to the UDP header, or NULL if this is not a UDP packet
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_udp_t *trace_get_udp(libtrace_packet_t *packet);

/** get a pointer to the UDP header (if any) given a pointer to the IP header
 * @param 	ip	The IP header
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the UDP header, or NULL if this is not an UDP packet
 *
 * Remaining may be NULL.  If Remaining is not NULL it must point to the number
 * of bytes captured of the UDP header and beyond.  This function will
 * decremented it by the length of the UDP header.
 *
 * @note Beware the change from libtrace2 from skipped to remaining
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_udp_t *trace_get_udp_from_ip(libtrace_ip_t *ip,uint32_t *remaining);

/** get a pointer to the ICMP header (if any)
 * @param packet  	the packet opaque pointer
 *
 * @return a pointer to the ICMP header, or NULL if this is not a ICMP packet
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_icmp_t *trace_get_icmp(libtrace_packet_t *packet);

/** get a pointer to the ICMP header (if any) given a pointer to the IP header
 * @param ip		The IP header
 * @param[in,out] remaining Updated with the number of bytes remaining
 *
 * @return a pointer to the ICMP header, or NULL if this is not an ICMP packet
 *
 * Remaining may be NULL.  If Remaining is not NULL it must point to the number
 * of bytes captured of the ICMP header and beyond.  It will be decremented by
 * the length of the ICMP head in bytes.
 *
 * @note Beware the change from libtrace2 from skipped to remaining
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_icmp_t *trace_get_icmp_from_ip(libtrace_ip_t *ip,uint32_t *remaining);

/** Get the destination MAC address
 * @param packet  	the packet opaque pointer
 * @return a pointer to the destination mac, (or NULL if there is no 
 * destination MAC)
 */
DLLEXPORT SIMPLE_FUNCTION
uint8_t *trace_get_destination_mac(libtrace_packet_t *packet);

/** Get the source MAC address
 * @param packet  	the packet opaque pointer
 * @return a pointer to the source mac, (or NULL if there is no source MAC)
 */
DLLEXPORT SIMPLE_FUNCTION
uint8_t *trace_get_source_mac(libtrace_packet_t *packet);

/** Get the source IP address
 * @param packet  	the packet opaque pointer
 * @param addr		a pointer to a sockaddr to store the address in, or NULL to use
 * 			static storage.
 * @return NULL if there is no source address, or a sockaddr holding a v4 or v6
 * address
 */
DLLEXPORT SIMPLE_FUNCTION
struct sockaddr *trace_get_source_address(const libtrace_packet_t *packet,
		struct sockaddr *addr);

/** Get the destination IP address
 * @param packet  	the packet opaque pointer
 * @param addr		a pointer to a sockaddr to store the address in, or NULL to use
 * 			static storage.
 * @return NULL if there is no destination address, or a sockaddr holding a v4
 * or v6 address
 */
DLLEXPORT SIMPLE_FUNCTION
struct sockaddr *trace_get_destination_address(const libtrace_packet_t *packet,
		struct sockaddr *addr);

/*@}*/

/** parse an ip or tcp option
 * @param[in,out] ptr	the pointer to the current option
 * @param[in,out] len	the length of the remaining buffer
 * @param[out] type	the type of the option
 * @param[out] optlen 	the length of the option
 * @param[out] data	the data of the option
 *
 * @return bool true if there is another option (and the fields are filled in)
 *               or false if this was the last option.
 *
 * This updates ptr to point to the next option after this one, and updates
 * len to be the number of bytes remaining in the options area.  Type is updated
 * to be the code of this option, and data points to the data of this option,
 * with optlen saying how many bytes there are.
 *
 * @note Beware of fragmented packets.
 */
DLLEXPORT int trace_get_next_option(unsigned char **ptr,int *len,
			unsigned char *type,
			unsigned char *optlen,
			unsigned char **data);


/** @name Time
 * These functions deal with time that a packet arrived and return it
 * in various formats
 * @{
 */
/** Get the current time in DAG time format 
 * @param packet  	the packet opaque pointer
 *
 * @return a 64 bit timestamp in DAG ERF format (upper 32 bits are the seconds
 * past 1970-01-01, the lower 32bits are partial seconds)
 */
DLLEXPORT SIMPLE_FUNCTION
uint64_t trace_get_erf_timestamp(const libtrace_packet_t *packet);

/** Get the current time in struct timeval
 * @param packet  	the packet opaque pointer
 *
 * @return time that this packet was seen in a struct timeval
 */ 
DLLEXPORT SIMPLE_FUNCTION
struct timeval trace_get_timeval(const libtrace_packet_t *packet);

/** Get the current time in floating point seconds
 * @param packet  	the packet opaque pointer
 *
 * @return time that this packet was seen in 64bit floating point seconds from
 * the unix epoch (1970-01-01 00:00:00 UTC).
 */
DLLEXPORT SIMPLE_FUNCTION
double trace_get_seconds(const libtrace_packet_t *packet);

/** Seek within a trace
 * @param trace		trace to seek
 * @param seconds	time to seek to
 * @return 0 on success.
 * Make the next packet read to be the first packet to occur at or after the
 * time searched for.  This must be called in the configuration state (ie,
 * before trace_start() or after trace_pause().
 * @note This function may be extremely slow.
 */
DLLEXPORT int trace_seek_seconds(libtrace_t *trace, double seconds);

/** Seek within a trace
 * @param trace		trace to seek
 * @param tv		time to seek to
 * @return 0 on success.
 * Make the next packet read to be the first packet to occur at or after the
 * time searched for.  This must be called in the configuration state (ie,
 * before trace_start() or after trace_pause().
 * @note This function may be extremely slow.
 */
DLLEXPORT int trace_seek_timeval(libtrace_t *trace, struct timeval tv);

/** Seek within a trace
 * @param trace		trace to seek
 * @param ts		erf timestamp
 * @return 0 on success.
 * Make the next packet read to be the first packet to occur at or after the
 * time searched for.  This must be called in the configuration state (ie,
 * before trace_start() or after trace_pause().
 * @note This function may be extremely slow.
 */
DLLEXPORT int trace_seek_erf_timestamp(libtrace_t *trace, uint64_t ts);

/*@}*/

/** @name Sizes
 * This section deals with finding or setting the various different lengths
 * a packet can have
 * @{
 */
/** Get the size of the packet in the trace (in bytes)
 * @param packet  	the packet opaque pointer
 * @return the size of the packet in the trace
 * @note Due to this being a header capture, or anonymisation, this may not
 * be the same size as the original packet.  See get_wire_length() for the
 * original size of the packet.
 * @note This can (and often is) different for different packets in a trace!
 * @note This is sometimes called the "snaplen".
 * @note The return size refers to the network-level payload of the packet and
 * does not include any capture framing headers. For example, an Ethernet 
 * packet with an empty TCP packet will return sizeof(ethernet_header) + 
 * sizeof(ip_header) + sizeof(tcp_header), but not the capture file 
 * (pcap/erf/etc) header.
 */
DLLEXPORT SIMPLE_FUNCTION
size_t trace_get_capture_length(const libtrace_packet_t *packet);

/** Get the size of the packet as it was seen on the wire (in bytes).
 * @param packet  	the packet opaque pointer
 * @return the size of the packet as it was on the wire.
 * @note Due to the trace being a header capture, or anonymisation this may
 * not be the same as the Capture Len.
 * @note trace_getwire_length \em{includes} FCS.
 * @note The return size refers to the network-level payload of the packet and
 * does not include any capture framing headers. For example, an Ethernet 
 * packet with an empty TCP packet will return sizeof(ethernet_header) + 
 * sizeof(ip_header) + sizeof(tcp_header), but not the capture file 
 * (pcap/erf/etc) header.
 */ 
DLLEXPORT SIMPLE_FUNCTION
size_t trace_get_wire_length(const libtrace_packet_t *packet);

/** Get the length of the capture framing headers (in bytes).
 * @param packet  	the packet opaque pointer
 * @return the size of the packet as it was on the wire.
 * @note this length corresponds to the difference between the size of a 
 * captured packet in memory, and the captured length of the packet
 */ 
DLLEXPORT SIMPLE_FUNCTION
size_t trace_get_framing_length(const libtrace_packet_t *packet);

/** Truncate ("snap") the packet at the suggested length
 * @param packet	the packet opaque pointer
 * @param size		the new length of the packet (in bytes)
 * @return the new capture length of the packet, or the original capture
 * length of the packet if unchanged
 */
DLLEXPORT size_t trace_set_capture_length(libtrace_packet_t *packet, size_t size);

/*@}*/


/** Get the type of the link layer
 * @param packet  	the packet opaque pointer
 * @return libtrace_linktype_t
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_linktype_t trace_get_link_type(const libtrace_packet_t *packet);

/** Set the direction flag, if it has one
 * @param packet  	the packet opaque pointer
 * @param direction	the new direction 
 * @returns -1 on error, or the direction that was set.
 */
DLLEXPORT libtrace_direction_t trace_set_direction(libtrace_packet_t *packet, libtrace_direction_t direction);

/** Get the direction flag, if it has one
 * @param packet  	the packet opaque pointer
 * @return a value containing the direction flag, or -1 if this is not supported
 * The direction is defined as 0 for packets originating locally (ie, outbound)
 * and 1 for packets originating remotely (ie, inbound).
 * Other values are possible, which might be overloaded to mean special things
 * for a special trace.
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_direction_t trace_get_direction(const libtrace_packet_t *packet);

/** @name BPF
 * This section deals with using Berkley Packet Filters
 * @{
 */
/** setup a BPF filter
 * @param filterstring a char * containing the bpf filter string
 * @return opaque pointer pointer to a libtrace_filter_t object
 * @note The filter is not actually compiled at this point, so no correctness
 * tests are performed here. trace_create_filter will always return ok, but
 * if the filter is poorly constructed an error will be generated when the 
 * filter is actually used
 */
DLLEXPORT SIMPLE_FUNCTION
libtrace_filter_t *trace_create_filter(const char *filterstring);

/** Setup a BPF filter based on pre-compiled byte-code.
 * @param bf_insns	A pointer to the start of the byte-code
 * @param bf_len	The number of BPF instructions	
 * @returns		an opaque pointer to a libtrace_filter_t object
 * @note		The supplied byte-code is not checked for correctness.
 * @author		Scott Raynel
 */
DLLEXPORT libtrace_filter_t *
trace_create_filter_from_bytecode(void *bf_insns, unsigned int bf_len);

/** Apply a BPF filter to a packet
 * @param filter 	the filter opaque pointer
 * @param packet	the packet opaque pointer
 * @return >0 if the filter matches, 0 if it doesn't, -1 on error.
 * @note Due to the way BPF filters are built, the filter is not actually
 * compiled until the first time trace_create_filter is called. If your filter
 * is incorrect, it will generate an error message and assert, exiting the
 * program. This behaviour may change to more graceful handling of this error
 * in the future.
 */
DLLEXPORT int trace_apply_filter(libtrace_filter_t *filter,
		const libtrace_packet_t *packet);

/** Destroy a BPF filter
 * @param filter 	the filter opaque pointer
 * Deallocate all the resources associated with a BPF filter
 */
DLLEXPORT void trace_destroy_filter(libtrace_filter_t *filter);
/*@}*/

/** @name Portability
 * This section has functions that causes annoyances to portability for one
 * reason or another.
 * @{
 */

/** Convert an ethernet address to a string 
 * @param addr 	Ethernet address in network byte order
 * @param buf	Buffer to store the ascii representation, or NULL
 * @return buf, or if buf is NULL then a statically allocated buffer.
 *
 * This function is similar to the GNU ether_ntoa_r function, with a few
 * minor differences.  if NULL is passed as buf, then the function will
 * use an internal static buffer, if NULL isn't passed then the function
 * will use that buffer instead.
 *
 * @note the type of addr isn't struct ether_addr as it is with ether_ntoa_r,
 * however it is bit compatible so that a cast will work.
 */ 
DLLEXPORT char *trace_ether_ntoa(const uint8_t *addr, char *buf);

/** Convert a string to an ethernet address
 * @param buf	Ethernet address in hex format delimited with :'s.
 * @param addr	buffer to store the binary representation, or NULL
 * @return addr, or if addr is NULL, then a statically allocated buffer.
 *
 * This function is similar to the GNU ether_aton_r function, with a few
 * minor differences.  if NULL is passed as addr, then the function will
 * use an internal static buffer, if NULL isn't passed then the function will 
 * use that buffer instead.
 * 
 * @note the type of addr isn't struct ether_addr as it is with ether_aton_r,
 * however it is bit compatible so that a cast will work.
 */
DLLEXPORT uint8_t *trace_ether_aton(const char *buf, uint8_t *addr);

/*@}*/


/** Which port is the server port */
typedef enum {
	USE_DEST, 	/**< Destination port is the server port */
	USE_SOURCE	/**< Source port is the server port */
} serverport_t;

/** Get the source port
 * @param packet	the packet to read from
 * @return a port in \em HOST byte order, or equivalent to ports for this
 * protocol, or 0 if this protocol has no ports.
 */
DLLEXPORT SIMPLE_FUNCTION
uint16_t trace_get_source_port(const libtrace_packet_t *packet);

/** Get the destination port
 * @param packet	the packet to read from
 * @return a port in \em HOST byte order, or equivilent to ports for this
 * protocol, or 0 if this protocol has no ports.
 */
DLLEXPORT SIMPLE_FUNCTION
uint16_t trace_get_destination_port(const libtrace_packet_t *packet);

/** hint at the server port in specified protocol
 * @param protocol	the IP layer protocol, eg 6 (tcp), 17 (udp)
 * @param source	the source port from the packet
 * @param dest		the destination port from the packet
 * @return one of USE_SOURCE or USE_DEST depending on which one you should use
 * @note ports must be in \em HOST byte order!
 */
DLLEXPORT SIMPLE_FUNCTION
int8_t trace_get_server_port(uint8_t protocol, uint16_t source, uint16_t dest);

/** Takes a uri and splits it into a format and uridata component. 
 * @param uri		the uri to be parsed
 * @param format	destination location for the format component of the uri 
 * @return 0 if an error occured, otherwise return the uridata component
 */
DLLEXPORT const char *trace_parse_uri(const char *uri, char **format);

/** Gets the format type for a given packet.
 * @param packet	the packet opaque pointer
 * @return the format of the packet
 */
DLLEXPORT 
enum base_format_t trace_get_format(struct libtrace_packet_t *packet);

/** Construct a packet from a buffer.
 * @param packet[in,out]	Libtrace Packet object to update with the new 
 * 				data.
 * @param linktype		The linktype of the packet.
 * @param[in] data		The packet data (including linklayer)
 * @param len			Length of packet data
 */
DLLEXPORT
void trace_construct_packet(libtrace_packet_t *packet,
		libtrace_linktype_t linktype, const void *data, uint16_t len);

/*@}*/

/** @name Wireless trace support
 * Functions to access wireless information from packets that have wireless
 * monitoring headers such as Radiotap or Prism.
 * 
 * The trace_get_wireless_* functions provide an abstract interface for
 * retrieving information from wireless traces. They take a pointer to the
 * wireless monitoring header (usually found with trace_get_link(packet)) and
 * the linktype of the header passed in.
 * 
 * All of the trace_get_wireless_* functions return false if the requested
 * information was unavailable, or true if it was. The actual data is stored
 * in an output variable supplied by the caller. Values returned into the 
 * output variable will always be returned in host byte order.
 * @{
 */


#ifndef ARPHRD_80211_RADIOTAP
/* libc doesn't define this yet, but it seems to be what everyone is using
 */
#define ARPHRD_80211_RADIOTAP 803
#endif

/** Get the wireless Timer Syncronisation Function
 *
 * Gets the value of the timer syncronisation function for this frame, which
 * is a value in microseconds indicating the time that the first bit of the
 * MPDU was received by the MAC.
 *
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in 
 * @param[out] tsft the value of the timer syncronisation function. 
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_tsft(void *linkptr,
        libtrace_linktype_t linktype, uint64_t *tsft);

/** Get the wireless rate 
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] rate the data-rate of the frame in units of 500kbps
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_rate(void *linkptr,
        libtrace_linktype_t linktype, uint8_t *rate);

/** Get the wireless channel frequency
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] freq the frequency in MHz of the channel the frame was transmitted
 * or received on.
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_freq(void *linkptr,
        libtrace_linktype_t linktype, uint16_t *freq);

/** Get the wireless signal strength in dBm 
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] strength the RF signal power at the antenna, in dB difference
 * from 1mW.
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_signal_strength_dbm(void *linkptr,
        libtrace_linktype_t linktype, int8_t *strength);

/** Get the wireless noise strength in dBm
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] strength the RF noise power at the antenna, in dB difference 
 * from 1mW. 
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_noise_strength_dbm(void *linkptr,
        libtrace_linktype_t linktype, int8_t *strength);

/** Get the wireless signal strength in dB
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] strength the RF signal power at the antenna,in dB difference 
 * from a fixed reference. 
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_signal_strength_db(void *linkptr,
        libtrace_linktype_t linktype, uint8_t *strength);

/** Get the wireless noise strength in dB 
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] strength the RF noise power at the antenna, in dB difference 
 * from a fixed reference. 
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_noise_strength_db(void *linkptr,
        libtrace_linktype_t linktype, uint8_t *strength);

/** Get the wireless transmit attenuation 
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] attenuation the transmit power as a unitless distance from maximum 
 * power set at factory calibration. 0 indicates maximum transmission power.
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_tx_attenuation(void *linkptr,
        libtrace_linktype_t linktype, uint16_t *attenuation);

/** Get the wireless transmit attenuation in dB
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] attenuation the transmit power as dB difference from maximum power 
 * set at factory calibration. 0 indicates maximum power.
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_tx_attenuation_db(void *linkptr,
        libtrace_linktype_t linktype, uint16_t *attenuation);

/** Get the wireless transmit power in dBm @param link the wireless header
 * @param linktype the linktype of the wireless header passed in 
 * @param[out] txpower the transmit power as dB from a 1mW reference. This is the absolute power level measured at the antenna port.  
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_tx_power_dbm(void *linkptr, libtrace_linktype_t
		linktype, int8_t *txpower);

/** Get the wireless antenna 
 * @param link the wireless header
 * @param linktype the linktype of the wireless header passed in
 * @param[out] antenna which antenna was used to transmit or receive the frame.
 * @return true if the field was available, false if not.
 */
DLLEXPORT bool trace_get_wireless_antenna(void *linkptr,
        libtrace_linktype_t linktype, uint8_t *antenna);

/*@}*/

#ifdef __cplusplus
} /* extern "C" */
#endif /* #ifdef __cplusplus */
#endif /* LIBTRACE_H_ */
