/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "libpolyxmass-acidobasic.h"
#include "libpolyxmass-chemgroup.h"
#include "pxmchem-monomer.h"


PxmMonomer *
libpolyxmass_acidobasic_render_xml_node_mnm (xmlDocPtr xml_doc,
					     xmlNodePtr xml_node,
					     gpointer data)
{
  /* The DTD for the mnm element says this:

  <!ELEMENT mnm (code,chemgroup*)>
  <!ELEMENT chemgroup ((pka,acidcharged)?,polrule*,chemgrouprule*)>
  <!ELEMENT chemgrouprule (entity,name,outcome)>
  

  xml_node points to element <mnm>

  We'll have to get the code of the mnm (monomer) and, for each chemgroup
  element, create a prop object by name "CHEMGROUP" that we'll add to the
  array of prop objects to the monomer object of code 'code' in the
  mnmGPA given as parameter.

  Here is an example of an xml file:
 
    +-- xml_node
    |
    V
    <mnm>
      <code>G</code>
      <chemgroup>
	<pka>9.6</pka>
	<acidcharged>TRUE</acidcharged>
	<polrule>left_trapped</polrule>
	<chemgrouprule>
	  <entity>LE_PLM_MODIF</entity>
	  <name>LE_Acetylation</name>
	  <outcome>LOST</outcome>
	</chemgrouprule>
      </chemgroup>
      <chemgroup>
	<pka>2.35</pka>
	<acidcharged>FALSE</acidcharged>
	<polrule>right_trapped</polrule>
      </chemgroup>
    </mnm>

  */
  gchar *help = NULL;

  PxmMonomer *monomer = NULL;
  PxmChemgroup *chemgroup = NULL;
  PxmProp *prop = NULL;
  
  
  
  /* Make sure we have parameters pointing bona fide to the right xml
     element.
  */
  g_assert (xml_node != NULL);
  g_assert (0 == xmlStrcmp (xml_node->name, 
			    (const xmlChar *) "mnm"));
  
  /* Now go to the first child of current node.
   */
  xml_node = xml_node->children;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* We now MUST have the <code> element.
   */
  g_assert (0 == xmlStrcmp (xml_node->name, (const xmlChar *) "code"));
  
  help = xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (help != NULL);

  /* We know the code of the monomer, we can allocate the monomer.
   */
  monomer = pxmchem_monomer_new ();
  
  pxmchem_monomer_set_code (monomer, help);
  
  g_free (help);
  
  /* Now we know the monomer into which chemgroup objects will be stored in
     the form of PxmProp instances. So we can go to the next element,
     which might not exist, or exist once or exist many times (chemgroup*).
  */
  xml_node = xml_node->next;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  while (xml_node != NULL)
    {
      /* A node is found, which should be <chemgroup>.
       */
      g_assert (0 == xmlStrcmp (xml_node->name, 
				(const xmlChar *) "chemgroup"));
      
      chemgroup = libpolyxmass_chemgroup_render_xml_node_chemgroup (xml_doc,
						  xml_node,
						  NULL);

      if (chemgroup == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 "%s@%d: failed creating a chemgroup instance "
		 "from xml data\n",
		 __FILE__, __LINE__);

	  pxmchem_monomer_free (monomer);
	  
	  return NULL;
	}
      
      /* We got an allocated chemgroup instance, so we can create a prop
	 object by name "CHEMGROUP".
       */
      prop = libpolyxmass_chemgroup_prop_new (chemgroup);
      
      g_ptr_array_add (monomer->propGPA, prop);
      
      /* Now we can go to the next child.
       */
      xml_node = xml_node->next;
      
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  /* We have finished parsing the chemgroup element(s) (if any) contained
     into the mnm element. Our work is done: the monomer is now
     complete with the data read from the xml file. 

     Note that for any single monomer in the polymer chemistry
     definition, there can be only one PxmChemgroup instance (in its
     monomer->propGPA) that might contain PxmChemgroupRule instance(s)
     of entity LE_PLM_MODIF/RE_PLM_MODIF.

     This is because a monomer cannot be modified more than once on
     its left/right_end group(s) (by permanent modification of the
     polymer sequence, either on its left_end or on its
     right_end). Thus, it might not be possible that a given monomer
     lists more than zero-or-one chemical entity LE_PLM_MODIF
     chemgrouprules for its chemgroups.

     So, we still have to make sure that there is not more than one
     PxmChemgroup instance in this monomer->propGPA array of
     "CHEMGROUP"-named PxmProp instances that contains one or more
     PxmChemgroupRule instances of entity LE_PLM_MODIF/RE_PLM_MODIF.

     That is, one monomer can have only one chemgroup being possibly
     modified with a chemgrouprule of entity LE_PLM_MODIF. It is
     possible that a given chemgroup lists more than one entity
     LE_PLM_MODIF, for example a protein could list two entities
     LE_PLM_MODIF of names Acetylation and Formylation for the
     NH2-terminal group of each monomer of the protein chemistry
     definition. But it might not be possible that two chemgroup
     objects of any single monomer list one or more entities
     LE_PLM_MODIF. Same applies for the RE_PLM_MODIF.
  */
  if (1 < 
      libpolyxmass_chemgroup_ensure_single_containing_entity (monomer, 
							      "LE_PLM_MODIF"))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     "%s@%d: there can be at most one chemgroup in a monomer,"
	     "that contains chemgrouprule instances with entity "
	     "%s. The others have been removed.\n", 
	     __FILE__, __LINE__, "LE_PLM_MODIF");
    }

  if (1 < 
      libpolyxmass_chemgroup_ensure_single_containing_entity (monomer, 
							      "RE_PLM_MODIF"))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     "%s@%d: there can be at most one chemgroup in a monomer,"
	     "that contains chemgrouprule instances with entity "
	     "%s. The others have been removed.\n", 
	     __FILE__, __LINE__, "RE_PLM_MODIF");
    }
  
  return monomer;
}



gboolean
libpolyxmass_acidobasic_render_xml_node_monomers (xmlDocPtr xml_doc,
						  xmlNodePtr xml_node,
						  GPtrArray *mnmGPA,
						  gpointer data)
{
  /* The DTD for the monomers element says this:

  <!ELEMENT monomers (mnm*)>
 
  with 

  <!ELEMENT mnm (code,chemgroup*)>

  Here is an example of an xml file:

  <acidobasicdata>
 
  +-- xml_node
  |
  V
  <monomers>
    <mnm>
      <code>G</code>
      <chemgroup>
	<value>9.6</value>
	<endrule>left_trapped</endrule>
	<acidcharged>TRUE</acidcharged>
	<chemgrouprule>
	  <entity>LE_Acetylation</entity>
	  <outcome>LOST</outcome>
	</chemgrouprule>
      </chemgroup>
      <chemgroup>
	<value>2.35</value>
	<endrule>right_trapped</endrule>
	<acidcharged>FALSE</acidcharged>
      </chemgroup>
    </mnm>
  */
  PxmMonomer *monomer = NULL;
  

  g_assert (mnmGPA != NULL);
  
  /* Make sure we have parameters pointing bona fide to the right xml
     element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == xmlStrcmp (xml_node->name, 
			    (const xmlChar *) "monomers"));
  
  /* Now go to the first child of current node. Since we do not know
     if there are elements of any kind, the best is to iterate in
     loop...
   */
  xml_node = xml_node->children;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  while (xml_node != NULL)
    {
      /* Check that we have effectively a <mnm> element here.
       */
      g_assert (0 == xmlStrcmp (xml_node->name, 
				(const xmlChar*) "mnm"));
      
      monomer = libpolyxmass_acidobasic_render_xml_node_mnm (xml_doc,
							     xml_node,
							     NULL);
      if (monomer == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 "%s@%d: could not parse a <monomers> element\n",
		 __FILE__, __LINE__);
	  
	  return FALSE;
	}

      g_ptr_array_add (mnmGPA, monomer);
      
      /* We can iterate in the next xml node.
       */
      xml_node = xml_node->next;
      
      /* From a rigorous XML parsing point of view, the blanks found
	 in the XML document are considered to be nodes, and we have
	 to detect these and take proper action: go next sibling (next
	 blank) as long as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }

  /* At this point we have finished parsing the <monomers> element.
   */
      
  return TRUE;
}

  
PxmModif *
libpolyxmass_acidobasic_render_xml_node_mdf (xmlDocPtr xml_doc,
					     xmlNodePtr xml_node,
					     gpointer data)
{
   /* The DTD for the mdf element says this:
      
      <!ELEMENT mdf (name,chemgroup*)>
      <!ELEMENT chemgrouprule (entity,outcome)>
      <!ELEMENT chemgroup (value,endrule*,acidcharged,chemgrouprule*)>

    <modifs>
      <mdf>
        <name>Phosphorylation</name>
        <chemgroup>
  	  <value>12</value>
  	  <acidcharged>FALSE</acidcharged>
        </chemgroup>
        <chemgroup>
  	  <value>7</value>
  	  <acidcharged>FALSE</acidcharged>
        </chemgroup>
      </mdf>
    </modifs>
    
    xml_node is located at <mdf>
   */
  gchar *help = NULL;

  PxmModif *modif = NULL;
  
  PxmChemgroup *chemgroup = NULL;
  PxmProp *prop = NULL;
  

  /* Make sure we have parameters pointing bona fide to the right xml
     element.
  */
  g_assert (xml_node != NULL);
  g_assert (0 == xmlStrcmp (xml_node->name, (const xmlChar *) "mdf"));
  
  /* Now go to the first child of current node: <value>.
   */
  xml_node = xml_node->children;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* We now MUST have the <name> element.
   */
  g_assert (0 == xmlStrcmp (xml_node->name, (const xmlChar *) "name"));
  
  help = xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (help != NULL);
  
  /* We know the name of the modification, we can allocate the modif.
   */
  modif = pxmchem_modif_new ();
  
  pxmchem_modif_set_name (modif, help);
  
  g_free (help);
  
  /* Now we know the modif into which chemgroup objects will be stored in
     the form of PxmProp instances. So we can go to the next element,
     which might not exist, or exist once or exist many times (chemgroup*).
  */
  xml_node = xml_node->next;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  while (xml_node != NULL)
    {
      /* A node is found, which should be <chemgroup>.
       */
      g_assert (0 == xmlStrcmp (xml_node->name, 
				(const xmlChar *) "chemgroup"));
      
      chemgroup = libpolyxmass_chemgroup_render_xml_node_chemgroup (xml_doc,
						  xml_node,
						  NULL);
      if (chemgroup == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 "%s@%d: failed creating a chemgroup instance "
		 "from xml data\n",
		 __FILE__, __LINE__);

	  pxmchem_modif_free (modif);
	  
	  return NULL;
	}
      
      /* We got an allocated chemgroup instance, so we can create a prop
	 object by name "CHEMGROUP".
       */
      prop = libpolyxmass_chemgroup_prop_new (chemgroup);
      
      g_ptr_array_add (modif->propGPA, prop);
      
      /* Now we can go to the next child.
       */
      xml_node = xml_node->next;
      
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  /* We have finished parsing the chemgroup element(s) (if any) contained
     into the mdf element. Our work is done, we return that mdf object.
  */
  return modif;
}

  
gboolean
libpolyxmass_acidobasic_render_xml_node_modifs (xmlDocPtr xml_doc,
						xmlNodePtr xml_node,
						GPtrArray *mdfGPA,
						gpointer data)
{
   /* The DTD for the modifs element says this:
 
   <!ELEMENT modifs (mdf*)>

   with 

   <!ELEMENT mdf (name,chemgroup*)>
   
  Here is an example of an xml file:
  
    <acidobasicdata>
    .
    .
    .
    </monomers>
      <modifs>
        <mdf>
          <name>Phosphorylation</name>
          <chemgroup>
    	<value>12</value>
    	<acidcharged>FALSE</acidcharged>
          </chemgroup>
          <chemgroup>
    	<value>7</value>
    	<acidcharged>FALSE</acidcharged>
          </chemgroup>
        </mdf>
      </modifs>
    </acidobasicdata>
   */
  PxmModif *modif = NULL;
  

  /* Make sure we have parameters pointing bona fide to the right xml
     element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == xmlStrcmp (xml_node->name, 
			    (const xmlChar *) "modifs"));
  
  /* Now go to the first child of current node. Since we do not know
     if there are elements of any kind, the best is to iterate in
     loop...
   */
  xml_node = xml_node->children;
  
  /* From a rigorous XML parsing point of view, the blanks found in
     the XML document are considered to be nodes, and we have to
     detect these and take proper action: go next sibling (next blank)
     as long as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  while (xml_node != NULL)
    {
      /* Check that we have effectively a <mdf> element here.
       */
      g_assert (0 == xmlStrcmp (xml_node->name, 
				(const xmlChar*) "mdf"));
      
      modif = libpolyxmass_acidobasic_render_xml_node_mdf (xml_doc,
							   xml_node,
							   NULL);
      
      if (modif == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 "%s@%d: could not parse a <modifs> element\n",
		 __FILE__, __LINE__);
	  
	  return FALSE;
	}
      
      g_ptr_array_add (mdfGPA, modif);

      /* We can iterate in the next xml node.
       */
      xml_node = xml_node->next;
      
      /* From a rigorous XML parsing point of view, the blanks found
	 in the XML document are considered to be nodes, and we have
	 to detect these and take proper action: go next sibling (next
	 blank) as long as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }

  /* At this point we have finished parsing the <modifs>
     element. All we have to do is return.
   */
  return TRUE;
}


gboolean
libpolyxmass_acidobasic_render_xml_file (gchar *file,
					 GPtrArray *mnmGPA,
					 GPtrArray *mdfGPA,
					 gpointer data)
{
  /* We get a file name which must be a valid xml file. Then we have
     to parse this file and construct a polymer definition that is
     compliant with the polyxmass data conventions.
   */
  xmlDocPtr xml_doc = NULL;
  xmlNodePtr xml_node = NULL;
  
  g_assert (file != NULL);
  
  if (strlen (file) <= 0)
    return FALSE;
  
  if (FALSE == g_file_test (file, G_FILE_TEST_EXISTS))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     "%s@%d: file does not exist: %s\n",
	     __FILE__, __LINE__, file);
      
      return FALSE;
    }
      
  /* The very first thing we could do is check that the file is an
   * XML file: <?xml version="1.0"?> should be the first item in the
   * file.
   */
  if (FALSE == libpolyxmass_globals_check_xml_file (file))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     "%s@%d: file is not valid xml format: %s\n",
	     __FILE__, __LINE__, file);
      
      return FALSE;
    }
  
  /* Build an XML tree from a the file.
   */
  xml_doc = xmlParseFile (file);

  if (xml_doc == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     "%s@%d: file is not valid xml format: %s\n",
	     __FILE__, __LINE__, file);
      
     return FALSE;
    }

  /* Check if the document is of the right kind.
   */
  xml_node = xmlDocGetRootElement (xml_doc);

  if (xml_node == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     "%s@%d: file is an empty xml file: %s\n",
	     __FILE__, __LINE__, file);

      xmlFreeDoc (xml_doc);
      
      return FALSE;
    }

  if (xmlStrcmp (xml_node->name, (const xmlChar *) "acidobasicdata"))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     "%s@%d: file is of wrong type, "
	     "root node != \"acidobasicdata\": %s\n",
	     __FILE__, __LINE__, file);
      
      xmlFreeDoc (xml_doc);
      
      return FALSE;
    }
  
  /* Now go to the first child of current node: <monomers> or <modifs>
     or none.
   */
  xml_node = xml_node->children;

  /* From a rigorous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  while (xml_node != NULL)
    {
      if (0 == xmlStrcmp (xml_node->name, (const xmlChar *) "monomers"))
	{
	  if (FALSE == 
	      libpolyxmass_acidobasic_render_xml_node_monomers (xml_doc,
								xml_node,
								mnmGPA,
								NULL))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     "%s@%d: could not render xml monomers node\n",
		     __FILE__, __LINE__);
	      
	      return FALSE;
	    }
	}
      else if (0 == xmlStrcmp (xml_node->name, (const xmlChar *) "modifs"))
	{
	  if (FALSE == 
	      libpolyxmass_acidobasic_render_xml_node_modifs (xml_doc,
							      xml_node,
							      mdfGPA,
							      NULL))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     "%s@%d: could not render xml modifs node\n",
		     __FILE__, __LINE__);
	      
	      return FALSE;
	    }
	}
      else
	g_assert_not_reached ();

      /* At this point, go to the next sibling.
       */
      xml_node = xml_node->next;
      
      /* From a rigorous XML parsing point of view, the blanks found
	 in the XML document are considered to be nodes, and we have
	 to detect these and take proper action: go next sibling (next
	 blank) as long as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  /* At this point we have successfully parsed the whole acidobasic
     element, that is the whole file. We just return TRUE.
   */
  return TRUE;






#if 0

  /* Test the contents of the arrays to ensure that the file was read 
     properly. This is a debugging code.
  */
  {
    gint iter = 0;
    gint jter = 0;
    gint kter = 0;
    
    PxmMonomer *monomer = NULL;
    PxmModif *modif = NULL;
    
    PxmChemgroup *chemgroup = NULL;
    PxmChemgroupRule *pr = NULL;

    PxmProp *prop = NULL;
    

    GPtrArray *mnmGPA = g_ptr_array_new ();
    GPtrArray *mdfGPA = g_ptr_array_new ();
    
    printf ("should be doing this\n");
    
    libpolyxmass_acidobasic_render_xml_file ("/home/rusconi/devel/"
					     "polyxmass-common/polchem-defs/"
					     "protein/acidobasic.xml",
					     mnmGPA,
					     mdfGPA,
					     NULL);

    for (iter = 0 ; iter < mnmGPA->len; iter++)
      {
	monomer = g_ptr_array_index (mnmGPA, iter);
	
	debug_printf (("monomer %s\n",monomer->code));

	for (jter = 0; jter < monomer->propGPA->len ; jter++)
	  {
	    prop = g_ptr_array_index (monomer->propGPA, jter);

	    chemgroup = prop->data;
	    
	    debug_printf (("\t prop is '%s'\n", prop->name));
	    debug_printf (("\t chemgroup value is '%lf'; chemgroup end rule is '%d'; "
			   "acidcharged is '%d'\n", 

			   chemgroup->value,
			   chemgroup->endrule,
			   chemgroup->acidcharged));
	    	    
	    for (kter = 0 ; kter < chemgroup->prGPA->len; kter++)
	      {
		pr = g_ptr_array_index (chemgroup->prGPA, kter);
		
		debug_printf (("\t\t pr entity is '%s' and outcome is '%d'\n",
			       pr->entity, pr->outcome));
	      }
	    
	  }
      }

    for (iter = 0 ; iter < mdfGPA->len; iter++)
      {
	modif = g_ptr_array_index (mdfGPA, iter);
	
	debug_printf (("modif %s\n",modif->name));

	for (jter = 0; jter < modif->propGPA->len ; jter++)
	  {
	    prop = g_ptr_array_index (modif->propGPA, jter);

	    chemgroup = prop->data;
	    
	    debug_printf (("\t prop is '%s'\n", prop->name));
	    debug_printf (("\t chemgroup value is '%lf'; chemgroup end rule is '%d'; "
			   "acidcharged is '%d'\n", 

			   chemgroup->value,
			   chemgroup->endrule,
			   chemgroup->acidcharged));
	    	    
	    for (kter = 0 ; kter < chemgroup->prGPA->len; kter++)
	      {
		pr = g_ptr_array_index (chemgroup->prGPA, kter);
		
		debug_printf (("\t\t pr entity is '%s' and outcome is '%d'\n",
			       pr->entity, pr->outcome));
	      }
	    
	  }
      }
  }
#endif
}



gdouble
libpolyxmass_acidobasic_compute_charge_ratio (gdouble pka,
					      gdouble ph,
					      gboolean acidcharged)
{
  gdouble a_over_ah = 0;
  
  if (pka < 0)
    return 0;
  if (pka > 14)
    return 0;
  
  if (ph < 0)
    return 0;
  if (ph > 14)
    return 0;
    
  a_over_ah = (gdouble) pow (10, (ph - pka));
  
  
  if (a_over_ah < 1)
    {
      /* The solution contains more acid forms (AH) than basic forms
	 (A).
      */
      if (acidcharged == TRUE)
	return (1 - a_over_ah);
      else
	return (- a_over_ah);
    }
  
  else if (a_over_ah > 1)
    {
      /* The solution contains more basic forms (A) than acid forms
	 (AH).
      */
      if (acidcharged == TRUE)
	return (1 / a_over_ah);
      else
	return (- (1 - (1 / a_over_ah)));
    }
  else if (a_over_ah == 1)
    {
      /* The solution contains as many acid forms (AH) as basic forms
	 (H).
      */
      if (acidcharged == TRUE)
	return (a_over_ah / 2);
      else
	return (- a_over_ah / 2);
    }
  else
    g_assert_not_reached ();
  
  return 0;
}

