/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributors: Jason Day <jasonday@worldnet.att.net>
 */

#include <stdio.h>

#ifdef _WIN32
#include <getopt.h>
#else
#include <unistd.h>
#ifndef _HPUX_SOURCE
#include <dlfcn.h>
#endif
#endif

#ifdef HAVE_GETOPTLONG
#include <getopt.h>
#endif

#include <AGNet.h>
#include <AGUtil.h>
#include <AGClientProcessor.h>
#include <AGProtocol.h>
#include <AGBufferReader.h>
#include <AGPalmProtocol.h>
#include <AGUserConfig.h>
#include <AGServerConfig.h>
#include <AGSyncCommon.h>
#include <AGCommandProcessor.h>
#include <AGDesktopInfoPalm.h>
#include <AGTimeoutMessageBox.h>
#include <AGMobileLinkSettings.h>
#include <MAL31UserConfig.h>

#include <pi-source.h>
#include <pi-socket.h>
#include <pi-file.h>
#include <pi-dlp.h>
#include <pi-version.h>
#include <pi-header.h>

#include "libmal.h"

static char *progname       = NULL;
#ifdef _DEBUG
#define debug_print(x) printf("%s\n", (x));
#else
#define debug_print(x)
#endif


#ifdef HAVE_GETOPTLONG
static struct option long_options[] =
{
    {"proxyaddress", 1, 0, 1},
    {"proxyport",    1, 0, 2},
    {"proxyname",    1, 0, 3},
    {"proxypasswd",  1, 0, 4},
    
    {"socksproxy",   1, 0, 5},
    {"socksport",    1, 0, 6},
    
    {"debug",        0, 0, 7},
    {"daemon",       0, 0, 8},
    {"help",         0, 0, 9},
    {"lowres",       0, 0, 10},
    
    {0, 0, 0, 0}
};
#endif

/*
 * This variable is left here, but daemon mode is not enabled
 * any more.
 */
static int daemon_mode = 0;


static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask,
                     AGBool bufferable);
static int32 cmdITEM(void *out,
                     int32 *returnErrorCode,
                     int32 currentItemNumber,
                     int32 totalItemCount,
                     char *currentItem);
static int processCommandLine(int argc, char *argv[], PalmSyncInfo *pInfo);



/*----------------------------------------------------------------------------*/

/*
 * TASK callback function. This is actually the default function defined
 * in libmal, but reimplemented here for demonstration.
 */
static int32
cmdTASK(void *out, int32 *returnErrorCode, char *currentTask, 
        AGBool bufferable)
{
    if (currentTask)
        printf("%s\n", currentTask);
    return AGCLIENT_CONTINUE;
}

/*
 * ITEM callback function. This is actually the default function defined
 * in libmal, but reimplemented here for demonstration.
 */
static int32 
cmdITEM(void *out, int32 *returnErrorCode, int32 currentItemNumber,
        int32 totalItemCount, char *currentItem)
{
    printf(".");
    fflush(stdout);
    if (currentItemNumber == totalItemCount)
        printf("\n");
    
    return AGCLIENT_CONTINUE;
}


int
main(int argc, char *argv[])
{
    PalmSyncInfo *pInfo;
    
    pInfo = syncInfoNew();
    if (NULL == pInfo)
        return -1;

    processCommandLine(argc, argv, pInfo);

    pInfo->taskFunc = cmdTASK;
    pInfo->itemFunc = cmdITEM;

    setbuf (stdout, 0);

    Connect(pInfo);

    malsync (pInfo);

    Disconnect(pInfo);
    syncInfoFree(pInfo);

    if (pInfo->httpProxy)
        free(pInfo->httpProxy);
    
    return 0;
}
/*---------------------------------------------------------------------------*/
static void
Help(void)
{
      printf("Usage: %s [OPTIONS]\n\n", progname);
#ifdef HAVE_GETOPTLONG
      printf("  -v, --version        Version.\n");
      printf("  -p, --proxyaddress   Proxy address.\n");
      printf("  -r, --proxyport      Proxy port.\n");
      printf("  -u, --proxyname      Proxy username.\n");
      printf("  -d, --proxypasswd    Proxy password.\n");
      printf("  -s, --socksproxy     Socks proxy.\n");
      printf("  -o, --socksport      Socks port.\n");
      printf("  -g, --debug          Print out a lot of debug stuff.\n");
      printf("  -l, --lowres         Low resolution images.\n");
      printf("  -D, --daemon         Work as a daemon (disconnect from shell).\n");
      printf("  -h, --help           Print this help.\n\n");
#else
      printf("  -v,     Version.\n");
      printf("  -p,     Proxy address.\n");
      printf("  -r,     Proxy port.\n");
      printf("  -u,     Proxy username.\n");
      printf("  -d,     Proxy password.\n");
      printf("  -s,     Socks proxy.\n");
      printf("  -o,     Socks port.\n");
      printf("  -g,     Print out a lot of debug stuff.\n");
      printf("  -l,     Low Resoultion Images.\n");
      printf("  -D,     Work as a daemon (disconnect from shell).\n");
      printf("  -h,     Print this help.\n\n");
#endif      
      printf("The serial port to connect to may be specified by the PILOTPORT");
      printf("\nenvironment variable. If not specified it will default to \n");
      printf("/dev/pilot \n");
      printf("\n");
      printf("The baud rate to connect with may be specified by the PILOTRATE\n");
      printf("environment variable. If not specified, it will default to 9600.\n");
      printf("Please use caution setting it to higher values, as several types\n");
      printf("of workstations have problems with higher rates.\n");
      printf("\n");
      exit(0);
}
/*----------------------------------------------------------------------------*/
static int 
getProxyInfo(char *str, PalmSyncInfo *pInfo)
{
    char *ptr;
    char *proxyname = NULL;
    int port;

    /* chop off the leading stuff if there */
    ptr = strstr(str, "http://");
    
    if (ptr) 
        proxyname = strdup(ptr+7);
    else 
        proxyname = strdup(str);

    if (!proxyname)
        return -1;

    /* look for the port */
    ptr = index(proxyname, ':');
    if (NULL == ptr) {
        free(proxyname);
        return -1;
    }
    
    /* get el porto */
    port = atoi(ptr+1);

    if (port <= 0 || port > 65535) {
        free(proxyname);
        return -1;
    }

    /* get rid of the port info in the string */
    *ptr = 0;

    /* yea! set the correct proxy values */
    pInfo->httpProxy     = proxyname;
    pInfo->httpProxyPort = port;

    return 0;
}
/*----------------------------------------------------------------------------*/
static int 
processCommandLine(int argc, char *argv[], PalmSyncInfo *pInfo)
{
    char *str;
    int c;
#ifdef HAVE_GETOPTLONG
    int optind;
#endif
    
    str = getenv("PILOTPORT");
    if (str != NULL)
        pInfo->device = str;
    
    /* Look for the http_proxy env variable */
    str = getenv("http_proxy");
    if (NULL != str)
        getProxyInfo(str, pInfo);

    progname = argv[0];
    do {
#ifdef HAVE_GETOPTLONG
        c = getopt_long( argc, argv, "v:p:r:u:d:s:o:gDhvl",
                         long_options, &optind);
#else
        c = getopt( argc, argv, "p:r:u:d:s:o:gDhvl");
#endif
        
        switch( c ) {
        case 'p': case 1:
            if (pInfo->httpProxy)
                free(pInfo->httpProxy);
            pInfo->httpProxy = strdup(optarg);
            break;
        case 'r': case 2:
            pInfo->httpProxyPort = atoi(optarg);
            break;
        case 'u': case 3:
            pInfo->proxyUsername = optarg;
            break;
        case 'd': case 4:
            pInfo->proxyPassword = optarg;
            break;
        case 's': case 5:
            pInfo->socksProxy = optarg;
            break;
        case 'o': case 6:
            pInfo->socksProxyPort = atoi(optarg);
            break;
            case 'g': case 7:
            /*
             * FIXME
             * Not practical right now
             */
            /*verbose = 1;*/
            break;
        case 'D': case 8:
            /*verbose = 0;*/
            daemon_mode = 1;
            break;
        case 'h': case 9:
            Help();
            break;
        case 'v': case 10:
            printf("%s\n", VERSION_STRING);
            exit(0);
        case 'l':
        case 11:
            pInfo->lowres = 1;
            break;
        }
        
    } while (c != EOF);

    return 0;
}
