/**
 * \file allumettes.cpp
 * \brief Implmentation de la classe etat_allumettes.
 * \author Julien Jorge
 */
#include "allumettes.hpp"
#include <iostream>

template<> const int claw::ai::game::game_state<int>::s_min_score = -100;
template<> const int claw::ai::game::game_state<int>::s_max_score =  100;

/*---------------------------------------------------------------------------*/
/**
 * \brief Affiche un tat sur un flux.
 * \param os Flux sur lequel crire.
 * \param e Etat a crire.
 * \return os.
 */
std::ostream& operator<<(std::ostream& os, const etat_allumettes& e)
{
  int i;

  for (i=0; i!=e.m_nb_allumettes; ++i)
    {
      os << "|";
      if ( i%3 == 2 )
        os << " ";
    }

  os << std::endl;

  return os;
} // operator<<()

/*---------------------------------------------------------------------------*/
/**
 * \brief Constructeur.
 * \param nb_allumettes Nombre d'allumettes.
 * \param ordi_joue Indique si l'ordi commence  jouer.
 */
etat_allumettes::etat_allumettes( int nb_allumettes, bool ordi_joue )
  : m_ordi(ordi_joue), m_nb_allumettes(nb_allumettes)
{

} // etat_allumettes() [constructeur]

/*---------------------------------------------------------------------------*/
/**
 * \brief Joue un nombre d'allumettes et passe au joueur suivant.
 * \return vrai si l'action termine le jeu.
 */
bool etat_allumettes::jouer( int c ) 
{ 
  m_nb_allumettes -= std::min(m_nb_allumettes, c);
  m_ordi = !m_ordi;     // on passe  l'autre joueur

  return m_nb_allumettes == 0;
} // jouer()

/*---------------------------------------------------------------------------*/
/**
 * \brief Effectue une action.
 * \param a action a effectuer.
 * \return L'tat rsultant de cette action.
 */
etat_allumettes::state* etat_allumettes::do_action( const int& a ) const
{
  state* em = new etat_allumettes(*this);

  ((etat_allumettes*)em)->jouer(a);
  return em;
} // do_action()

/*---------------------------------------------------------------------------*/
/**
 * \brief Indique si la partie est termine.
 */
bool etat_allumettes::final() const
{
  return m_nb_allumettes == 0;
} // final()

/*---------------------------------------------------------------------------*/
/**
 * \brief Calcule la liste des actions jouables depuis l'tat actuel.
 * \param l (sortie) liste des coups jouables.
 */
void etat_allumettes::nexts_actions( std::list<int>& l ) const
{
  for (int i=0; i!=3; ++i)
    if (m_nb_allumettes > i)
      l.push_back(i+1);
} // nexts_actions()

/*---------------------------------------------------------------------------*/
/**
 * \brief Evalue l'tat actuel, du point de vue du programme.
 */
int etat_allumettes::evaluate() const
{
  int score = 0;

  // il ne reste plus d'allumettes
  if (m_nb_allumettes == 0)
    {
      // c'est le joueur qui a jou -> gagn
      if( m_ordi )
        score = 100;
      else // ce l'ordi qui a jou -> perdu
        score = -100;
    } 

  return score;
} // evaluate()
