{
    $Id: aasmtai.pas,v 1.96 2005/02/18 23:37:51 jonas Exp $
    Copyright (c) 1998-2002 by Florian Klaempfl

    This unit implements an abstract asmoutput class for all processor types

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 ****************************************************************************
}
{ @abstract(This unit implements an abstract asm output class for all processor types)
  This unit implements an abstract assembler output class for all processors, these
  are then overriden for each assembler writer to actually write the data in these
  classes to an assembler file.
}

unit aasmtai;

{$i fpcdefs.inc}

interface

    uses
       cutils,cclasses,
       globtype,globals,systems,
       cpuinfo,cpubase,
       cgbase,cgutils,
       symtype,
       aasmbase;

    type
       taitype = (
          ait_none,
          ait_align,
          ait_section,
          ait_comment,
          ait_direct,
          ait_string,
          ait_instruction,
          ait_datablock,
          ait_symbol,
          ait_symbol_end, { needed to calc the size of a symbol }
          ait_label,
          { the const_xx must be below each other so it can be used as
            array index }
          ait_const_128bit,
          ait_const_64bit,
          ait_const_32bit,
          ait_const_16bit,
          ait_const_8bit,
          ait_const_sleb128bit,
          ait_const_uleb128bit,
          ait_const_rva_symbol, { win32 only }
          ait_const_indirect_symbol, { darwin only }
          ait_real_32bit,
          ait_real_64bit,
          ait_real_80bit,
          ait_comp_64bit,
          ait_real_128bit,
{$ifdef GDB}
          ait_stabn,
          ait_stabs,
          ait_force_line,
          ait_stab_function_name,
{$endif GDB}
{$ifdef alpha}
          { the follow is for the DEC Alpha }
          ait_frame,
          ait_ent,
{$endif alpha}
{$ifdef ia64}
          ait_bundle,
          ait_stop,
{$endif ia64}
{$ifdef m68k}
          ait_labeled_instruction,
{$endif m68k}
          { used to split into tiny assembler files }
          ait_cutobject,
          ait_regalloc,
          ait_tempalloc,
          { used to mark assembler blocks and inlined functions }
          ait_marker,
          { special symbol for darwin pic code }
          ait_non_lazy_symbol_pointer
          );

    const
{$ifdef cpu64bit}
       ait_const_aint = ait_const_64bit;
       ait_const_ptr  = ait_const_64bit;
{$else cpu64bit}
       ait_const_aint = ait_const_32bit;
       ait_const_ptr  = ait_const_32bit;
{$endif cpu64bit}

       taitypestr : array[taitype] of string[24] = (
          '<none>',
          'align',
          'section',
          'comment',
          'direct',
          'string',
          'instruction',
          'datablock',
          'symbol',
          'symbol_end',
          'label',
          'const_128bit',
          'const_64bit',
          'const_32bit',
          'const_16bit',
          'const_8bit',
          'const_sleb128bit',
          'const_uleb128bit',
          'const_rva_symbol',
          'const_indirect_symbol',
          'real_32bit',
          'real_64bit',
          'real_80bit',
          'comp_64bit',
          'real_128bit',
{$ifdef GDB}
          'stabn',
          'stabs',
          'force_line',
          'stab_funcname',
{$endif GDB}
{$ifdef alpha}
          { the follow is for the DEC Alpha }
          'frame',
          'ent',
{$endif alpha}
{$ifdef ia64}
          'bundle',
          'stop',
{$endif ia64}
{$ifdef m68k}
          'labeled_instr',
{$endif m68k}
          'cut',
          'regalloc',
          'tempalloc',
          'marker',
          'non_lazy_symbol_pointer'
          );

    type
      { Types of operand }
      toptype=(top_none,top_reg,top_ref,top_const,top_bool,top_local,
       { ARM only }
       top_regset,
       top_shifterop,
       { m68k only }
       top_reglist);

      { kinds of operations that an instruction can perform on an operand }
      topertype = (operand_read,operand_write,operand_readwrite);

      tlocaloper = record
        localsym : pointer;
        localsymderef : tderef;
        localsymofs : longint;
        localindexreg : tregister;
        localscale : byte;
        localgetoffset,
        localforceref : boolean
      end;
      plocaloper = ^tlocaloper;

      { please keep the size of this record <=12 bytes and keep it properly aligned }
      toper = record
        ot : longint;
        case typ : toptype of
          top_none   : ();
          top_reg    : (reg:tregister);
          top_ref    : (ref:preference);
          top_const  : (val:aint);
          top_bool   : (b:boolean);
          { local varsym that will be inserted in pass_2 }
          top_local  : (localoper:plocaloper);
      {$ifdef arm}
          top_regset : (regset:^tcpuregisterset);
          top_shifterop : (shifterop : pshifterop);
      {$endif arm}
      {$ifdef m68k}
          top_regset : (regset:^tcpuregisterset);
      {$endif m68k}
      end;
      poper=^toper;

{ ait_* types which don't result in executable code or which don't influence   }
{ the way the program runs/behaves, but which may be encountered by the        }
{ optimizer (= if it's sometimes added to the exprasm list). Update if you add }
{ a new ait type!                                                              }
    const
      SkipInstr = [ait_comment, ait_symbol,ait_section
{$ifdef GDB}
                   ,ait_stabs, ait_stabn, ait_stab_function_name, ait_force_line
{$endif GDB}
                   ,ait_regalloc, ait_tempalloc, ait_symbol_end];

{ ait_* types which do not have line information (and hence which are of type
  tai, otherwise, they are of type tailineinfo }
      SkipLineInfo =[ait_label,
                     ait_regalloc,ait_tempalloc,
{$ifdef GDB}
                  ait_stabn,ait_stabs,ait_stab_function_name,
{$endif GDB}
                  ait_cutobject,ait_marker,ait_align,ait_section,ait_comment,
                  ait_const_8bit,ait_const_16bit,ait_const_32bit,ait_const_64bit,ait_const_128bit,
                  ait_const_sleb128bit,ait_const_uleb128bit,
                  ait_const_rva_symbol,ait_const_indirect_symbol,
                  ait_real_32bit,ait_real_64bit,ait_real_80bit,ait_comp_64bit,ait_real_128bit,
                  ait_non_lazy_symbol_pointer
                  ];


    type
       { cut type, required for alphanumeric ordering of the assembler filenames }
       TCutPlace=(cut_normal,cut_begin,cut_end);

       TRegAllocType = (ra_alloc,ra_dealloc,ra_sync,ra_resize);

       TMarker = (NoPropInfoStart,NoPropInfoEnd,
                  AsmBlockStart,AsmBlockEnd,
                  InlineStart,InlineEnd,marker_blockstart,
                  marker_position);

       { Buffer type used for alignment }
       tfillbuffer = array[0..63] of char;

       Tspill_temp_list=array[tsuperregister] of Treference;

       { abstract assembler item }
       tai = class(TLinkedListItem)
{$ifndef NOOPT}
          { pointer to record with optimizer info about this tai object }
          optinfo  : pointer;
{$endif NOOPT}
          typ      : taitype;
          constructor Create;
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);virtual;
          procedure ppuwrite(ppufile:tcompilerppufile);virtual;
          procedure buildderefimpl;virtual;
          procedure derefimpl;virtual;
       end;

       { abstract assembler item with line information }
       tailineinfo = class(tai)
        fileinfo : tfileposinfo;
        constructor Create;
        constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
        procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;

       tai_simple = class(tai)
         constructor create(_typ : taitype);
       end;

       taiclass = class of tai;

       taiclassarray = array[taitype] of taiclass;

       { Generates an assembler string }
       tai_string = class(tailineinfo)
          str : pchar;
          { extra len so the string can contain an \0 }
          len : longint;
          constructor Create(const _str : string);
          constructor Create_pchar(_str : pchar);
          constructor Create_length_pchar(_str : pchar;length : longint);
          destructor Destroy;override;
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          function getcopy:tlinkedlistitem;override;
       end;

       { Generates a common label }
       tai_symbol = class(tailineinfo)
          is_global : boolean;
          sym       : tasmsymbol;
          size      : longint;
          constructor Create(_sym:tasmsymbol;siz:longint);
          constructor Create_Global(_sym:tasmsymbol;siz:longint);
          constructor Createname(const _name : string;_symtyp:Tasmsymtype;siz:longint);
          constructor Createname_global(const _name : string;_symtyp:Tasmsymtype;siz:longint);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          procedure derefimpl;override;
       end;

       tai_symbol_end = class(tailineinfo)
          sym : tasmsymbol;
          constructor Create(_sym:tasmsymbol);
          constructor Createname(const _name : string);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          procedure derefimpl;override;
       end;

       { Generates an assembler label }
       tai_label = class(tai)
          is_global : boolean;
          l         : tasmlabel;
          constructor Create(_l : tasmlabel);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          procedure derefimpl;override;
       end;

       { Directly output data to final assembler file }
       tai_direct = class(tailineinfo)
          str : pchar;
          constructor Create(_str : pchar);
          destructor Destroy; override;
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          function getcopy:tlinkedlistitem;override;
       end;

       { Generates an assembler comment }
       tai_comment = class(tai)
          str : pchar;
          constructor Create(_str : pchar);
          destructor Destroy; override;
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          function getcopy:tlinkedlistitem;override;
       end;


       { Generates a section / segment directive }
       tai_section = class(tai)
          sectype : TAsmSectionType;
          secalign : byte;
          name    : pstring;
          sec     : TAsmSection; { used in binary writer }
          constructor Create(Asectype:TAsmSectionType;Aname:string;Aalign:byte);
          destructor Destroy;override;
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;


       { Generates an uninitializised data block }
       tai_datablock = class(tailineinfo)
          is_global : boolean;
          sym       : tasmsymbol;
          size      : longint;
          constructor Create(const _name : string;_size : longint);
          constructor Create_global(const _name : string;_size : longint);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          procedure derefimpl;override;
       end;


       { Generates an integer const }
       tai_const = class(tai)
          sym,
          endsym  : tasmsymbol;
          value   : int64;
          { we use for the 128bit int64/qword for now because I can't imagine a
            case where we need 128 bit now (FK) }
          constructor Create(_typ:taitype;_value : int64);
          constructor Create_128bit(_value : int64);
          constructor Create_64bit(_value : int64);
          constructor Create_32bit(_value : longint);
          constructor Create_16bit(_value : word);
          constructor Create_8bit(_value : byte);
          constructor Create_sleb128bit(_value : int64);
          constructor Create_uleb128bit(_value : qword);
          constructor Create_aint(_value : aint);
          constructor Create_sym(_sym:tasmsymbol);
          constructor Create_sym_offset(_sym:tasmsymbol;ofs:aint);
          constructor Create_rel_sym(_typ:taitype;_sym,_endsym:tasmsymbol);
          constructor Create_rva_sym(_sym:tasmsymbol);
          constructor Create_indirect_sym(_sym:tasmsymbol);
          constructor Createname(const name:string;_symtyp:Tasmsymtype;ofs:aint);
          constructor Createname_rva(const name:string);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          procedure derefimpl;override;
          function getcopy:tlinkedlistitem;override;
          function size:longint;
       end;

       { Generates a single float (32 bit real) }
       tai_real_32bit = class(tai)
          value : ts32real;
          constructor Create(_value : ts32real);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;

       tformatoptions = (fo_none,fo_hiloswapped);

       { Generates a double float (64 bit real) }
       tai_real_64bit = class(tai)
          value : ts64real;
{$ifdef ARM}
          formatoptions : tformatoptions;
          constructor Create_hiloswapped(_value : ts64real);
{$endif ARM}
          constructor Create(_value : ts64real);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;


       { Generates an extended float (80 bit real) }
       tai_real_80bit = class(tai)
          value : ts80real;
          constructor Create(_value : ts80real);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;


       { Generates an float128 (128 bit real) }
       tai_real_128bit = class(tai)
          value : ts128real;
          constructor Create(_value : ts128real);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;

       { Generates a comp int (integer over 64 bits)

          This is Intel 80x86 specific, and is not
          really supported on other processors.
       }
       tai_comp_64bit = class(tai)
          value : ts64comp;
          constructor Create(_value : ts64comp);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;

       { Insert a cut to split assembler into several smaller files }
       tai_cutobject = class(tai)
          place : tcutplace;
          constructor Create;
          constructor Create_begin;
          constructor Create_end;
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;

       { Insert a marker for assembler and inline blocks }
       tai_marker = class(tai)
          Kind: TMarker;
          Constructor Create(_Kind: TMarker);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;

       tai_tempalloc = class(tai)
          allocation : boolean;
{$ifdef EXTDEBUG}
          problem : pstring;
{$endif EXTDEBUG}
          temppos,
          tempsize   : longint;
          constructor alloc(pos,size:longint);
          constructor dealloc(pos,size:longint);
{$ifdef EXTDEBUG}
          constructor allocinfo(pos,size:longint;const st:string);
{$endif EXTDEBUG}
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          destructor destroy;override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;

       tai_regalloc = class(tai)
          reg     : tregister;
          ratype  : TRegAllocType;
          { reg(de)alloc belongs to this instruction, this
            is only used for automatic inserted (de)alloc for
            imaginary register and required for spilling code }
          instr   : tai;
          constructor alloc(r : tregister;ainstr:tai);
          constructor dealloc(r : tregister;ainstr:tai);
          constructor sync(r : tregister);
          constructor resize(r : tregister);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
       end;

      Taasmoutput=class;

      tadd_reg_instruction_proc=procedure(instr:Tai;r:tregister) of object;
      Trggetproc=procedure(list:Taasmoutput;position:Tai;subreg:Tsubregister;var result:Tregister) of object;
      Trgungetproc=procedure(list:Taasmoutput;position:Tai;r:Tregister) of object;

       { Class template for assembler instructions
       }
       tai_cpu_abstract = class(tailineinfo)
       protected
          procedure ppuloadoper(ppufile:tcompilerppufile;var o:toper);virtual;abstract;
          procedure ppuwriteoper(ppufile:tcompilerppufile;const o:toper);virtual;abstract;
          procedure ppubuildderefimploper(var o:toper);virtual;abstract;
          procedure ppuderefoper(var o:toper);virtual;abstract;
       public
          { Condition flags for instruction }
          condition : TAsmCond;
          { Number of operands to instruction }
          ops       : byte;
          { Number of allocate oper structures }
          opercnt   : byte;
          { Operands of instruction }
          oper      : array[0..max_operands-1] of poper;
          { Actual opcode of instruction }
          opcode    : tasmop;
{$ifdef x86}
          segprefix : tregister;
{$endif x86}
          { true if instruction is a jmp }
          is_jmp    : boolean; { is this instruction a jump? (needed for optimizer) }
          Constructor Create(op : tasmop);virtual;
          Destructor Destroy;override;
          function getcopy:TLinkedListItem;override;
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          procedure buildderefimpl;override;
          procedure derefimpl;override;
          procedure SetCondition(const c:TAsmCond);
          procedure allocate_oper(opers:longint);
          procedure loadconst(opidx:longint;l:aint);
          procedure loadsymbol(opidx:longint;s:tasmsymbol;sofs:longint);
          procedure loadlocal(opidx:longint;s:pointer;sofs:longint;indexreg:tregister;scale:byte;getoffset,forceref:boolean);
          procedure loadref(opidx:longint;const r:treference);
          procedure loadreg(opidx:longint;r:tregister);
          procedure loadoper(opidx:longint;o:toper);
          procedure clearop(opidx:longint);
          { register allocator }
          function is_same_reg_move(regtype: Tregistertype):boolean;virtual;
          function spilling_get_operation_type(opnr: longint): topertype;virtual;
          function spilling_get_operation_type_ref(opnr: longint; reg: tregister): topertype;virtual;
       end;
       tai_cpu_class = class of tai_cpu_abstract;

       { alignment for operator }
       tai_align_abstract = class(tai)
          aligntype : byte;   { 1 = no align, 2 = word align, 4 = dword align }
          fillsize  : byte;   { real size to fill }
          fillop    : byte;   { value to fill with - optional }
          use_op    : boolean;
          constructor Create(b:byte);virtual;
          constructor Create_op(b: byte; _op: byte);virtual;
          constructor Create_zeros(b:byte);
          constructor ppuload(t:taitype;ppufile:tcompilerppufile);override;
          procedure ppuwrite(ppufile:tcompilerppufile);override;
          function calculatefillbuf(var buf : tfillbuffer):pchar;virtual;
       end;
       tai_align_class = class of tai_align_abstract;

       taasmoutput = class(tlinkedlist)
          constructor create;
          function  empty : boolean;
          function  getlasttaifilepos : pfileposinfo;
          procedure InsertAfter(Item,Loc : TLinkedListItem);override;
       end;


    var
      { array with all class types for tais }
      aiclass : taiclassarray;

      { Current expression list }
      exprasmlist : taasmoutput;

      { labels for BREAK and CONTINUE }
      aktbreaklabel,aktcontinuelabel : tasmlabel;

      { label when the result is true or false }
      truelabel,falselabel : tasmlabel;

      { hook to notify uses of registers }
      add_reg_instruction_hook : tadd_reg_instruction_proc;

      { default lists }
      datasegment,codesegment,bsssegment,
      debuglist,withdebuglist,consts,
      importssection,exportssection,
      resourcesection,rttilist,
      dwarflist,
      { data used by pic code }
      picdata,
      resourcestringlist         : taasmoutput;

      cai_align : tai_align_class;
      cai_cpu   : tai_cpu_class;

    function  use_smartlink_section:boolean;
    function  maybe_smartlink_symbol:boolean;
    procedure maybe_new_object_file(list:taasmoutput);
    procedure new_section(list:taasmoutput;Asectype:TAsmSectionType;Aname:string;Aalign:byte);

    function ppuloadai(ppufile:tcompilerppufile):tai;
    procedure ppuwriteai(ppufile:tcompilerppufile;n:tai);


implementation

    uses
      strings,
      verbose;

    const
      pputaimarker = 254;


{****************************************************************************
                                 Helpers
 ****************************************************************************}

    function ppuloadai(ppufile:tcompilerppufile):tai;
      var
        b : byte;
        t : taitype;
      begin
        { marker }
        b:=ppufile.getbyte;
        if b<>pputaimarker then
          internalerror(200208181);
        { load nodetype }
        t:=taitype(ppufile.getbyte);
        if t<>ait_none then
         begin
           if t>high(taitype) then
             internalerror(200208182);
           if not assigned(aiclass[t]) then
             internalerror(200208183);
           {writeln('taiload: ',taitypestr[t]);}
           { generate tai of the correct class }
           ppuloadai:=aiclass[t].ppuload(t,ppufile);
         end
        else
         ppuloadai:=nil;
      end;


    procedure ppuwriteai(ppufile:tcompilerppufile;n:tai);
      begin
        { marker, read by ppuloadnode }
        ppufile.putbyte(pputaimarker);
        if assigned(n) then
         begin
           { type, read by ppuloadnode }
           ppufile.putbyte(byte(n.typ));
           {writeln('taiwrite: ',taitypestr[n.typ]);}
           n.ppuwrite(ppufile);
         end
        else
         ppufile.putbyte(byte(ait_none));
      end;


    function use_smartlink_section:boolean;
      begin
        result:=(af_smartlink_sections in target_asm.flags) and
                (tf_smartlink_sections in target_info.flags) and
                not(cs_debuginfo in aktmoduleswitches);
      end;


    function maybe_smartlink_symbol:boolean;
      begin
        result:=(cs_create_smart in aktmoduleswitches) or
                use_smartlink_section;
      end;


    procedure maybe_new_object_file(list:taasmoutput);
      begin
        if (cs_create_smart in aktmoduleswitches) and
           (not use_smartlink_section) then
          list.concat(tai_cutobject.create);
      end;


    procedure new_section(list:taasmoutput;Asectype:TAsmSectionType;Aname:string;Aalign:byte);
      begin
        list.concat(tai_section.create(Asectype,Aname,Aalign));
        list.concat(cai_align.create(Aalign));
      end;


{****************************************************************************
                             TAI
 ****************************************************************************}

    constructor tai.Create;
      begin
{$ifndef NOOPT}
        optinfo:=nil;
{$endif NOOPT}
      end;


    constructor tai.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        typ:=t;
{$ifndef NOOPT}
        optinfo:=nil;
{$endif}
      end;


    procedure tai.ppuwrite(ppufile:tcompilerppufile);
      begin
      end;


    procedure tai.buildderefimpl;
      begin
      end;


    procedure tai.derefimpl;
      begin
      end;


{****************************************************************************
                              TAILINEINFO
 ****************************************************************************}

    constructor tailineinfo.create;
     begin
       inherited create;
       if not(inlining_procedure and
              (cs_gdb_valgrind in aktglobalswitches)) then
         fileinfo:=aktfilepos;
     end;


    constructor tailineinfo.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        ppufile.getposinfo(fileinfo);
      end;


    procedure tailineinfo.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putposinfo(fileinfo);
      end;


{****************************************************************************
                              TAI_SIMPLE
 ****************************************************************************}

    constructor tai_simple.create(_typ : taitype);
      begin
        inherited create;
        typ:=_typ;
      end;


{****************************************************************************
                             TAI_SECTION
 ****************************************************************************}

    constructor tai_section.Create(Asectype:TAsmSectionType;Aname:string;Aalign:byte);
      begin
        inherited Create;
        typ:=ait_section;
        sectype:=asectype;
        secalign:=Aalign;
        name:=stringdup(Aname);
        sec:=nil;
      end;


    constructor tai_section.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        sectype:=tasmsectiontype(ppufile.getbyte);
        secalign:=ppufile.getbyte;
        name:=stringdup(ppufile.getstring);
        sec:=nil;
      end;


    destructor tai_section.Destroy;
      begin
        stringdispose(name);
      end;


    procedure tai_section.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putbyte(byte(sectype));
        ppufile.putbyte(secalign);
        ppufile.putstring(name^);
      end;


{****************************************************************************
                             TAI_DATABLOCK
 ****************************************************************************}

    constructor tai_datablock.Create(const _name : string;_size : longint);

      begin
         inherited Create;
         typ:=ait_datablock;
         sym:=objectlibrary.newasmsymbol(_name,AB_LOCAL,AT_DATA);
         { keep things aligned }
         if _size<=0 then
           _size:=4;
         size:=_size;
         is_global:=false;
      end;


    constructor tai_datablock.Create_global(const _name : string;_size : longint);
      begin
         inherited Create;
         typ:=ait_datablock;
         sym:=objectlibrary.newasmsymbol(_name,AB_GLOBAL,AT_DATA);
         { keep things aligned }
         if _size<=0 then
           _size:=4;
         size:=_size;
         is_global:=true;
      end;


    constructor tai_datablock.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited Create;
        sym:=ppufile.getasmsymbol;
        size:=ppufile.getlongint;
        is_global:=boolean(ppufile.getbyte);
      end;


    procedure tai_datablock.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putasmsymbol(sym);
        ppufile.putlongint(size);
        ppufile.putbyte(byte(is_global));
      end;


    procedure tai_datablock.derefimpl;
      begin
        objectlibrary.DerefAsmsymbol(sym);
      end;


{****************************************************************************
                               TAI_SYMBOL
 ****************************************************************************}

    constructor tai_symbol.Create(_sym:tasmsymbol;siz:longint);
      begin
         inherited Create;
         typ:=ait_symbol;
         sym:=_sym;
         size:=siz;
         sym.defbind:=AB_LOCAL;
         is_global:=false;
      end;

    constructor tai_symbol.Create_global(_sym:tasmsymbol;siz:longint);
      begin
         inherited Create;
         typ:=ait_symbol;
         sym:=_sym;
         size:=siz;
         sym.defbind:=AB_GLOBAL;
         is_global:=true;
      end;

    constructor tai_symbol.Createname(const _name : string;_symtyp:Tasmsymtype;siz:longint);
      begin
         inherited Create;
         typ:=ait_symbol;
         sym:=objectlibrary.newasmsymbol(_name,AB_LOCAL,_symtyp);
         size:=siz;
         is_global:=false;
      end;

    constructor tai_symbol.Createname_global(const _name : string;_symtyp:Tasmsymtype;siz:longint);
      begin
         inherited Create;
         typ:=ait_symbol;
         sym:=objectlibrary.newasmsymbol(_name,AB_GLOBAL,_symtyp);
         size:=siz;
         is_global:=true;
      end;

    constructor tai_symbol.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        sym:=ppufile.getasmsymbol;
        size:=ppufile.getlongint;
        is_global:=boolean(ppufile.getbyte);
      end;


    procedure tai_symbol.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putasmsymbol(sym);
        ppufile.putlongint(size);
        ppufile.putbyte(byte(is_global));
      end;


    procedure tai_symbol.derefimpl;
      begin
        objectlibrary.DerefAsmsymbol(sym);
      end;


{****************************************************************************
                               TAI_SYMBOL_END
 ****************************************************************************}

    constructor tai_symbol_end.Create(_sym:tasmsymbol);
      begin
         inherited Create;
         typ:=ait_symbol_end;
         sym:=_sym;
      end;

    constructor tai_symbol_end.Createname(const _name : string);
      begin
         inherited Create;
         typ:=ait_symbol_end;
         sym:=objectlibrary.newasmsymbol(_name,AB_GLOBAL,AT_NONE);
      end;

    constructor tai_symbol_end.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        sym:=ppufile.getasmsymbol;
      end;


    procedure tai_symbol_end.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putasmsymbol(sym);
      end;


    procedure tai_symbol_end.derefimpl;
      begin
        objectlibrary.DerefAsmsymbol(sym);
      end;


{****************************************************************************
                               TAI_CONST
 ****************************************************************************}

    constructor tai_const.Create(_typ:taitype;_value : int64);
      begin
         inherited Create;
         typ:=_typ;
         value:=_value;
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_128bit(_value : int64);
      begin
         inherited Create;
         typ:=ait_const_128bit;
         value:=_value;
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_64bit(_value : int64);
      begin
         inherited Create;
         typ:=ait_const_64bit;
         value:=_value;
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_32bit(_value : longint);
      begin
         inherited Create;
         typ:=ait_const_32bit;
         value:=_value;
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_16bit(_value : word);
      begin
         inherited Create;
         typ:=ait_const_16bit;
         value:=_value;
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_8bit(_value : byte);
      begin
         inherited Create;
         typ:=ait_const_8bit;
         value:=_value;
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_sleb128bit(_value : int64);
      begin
         inherited Create;
         typ:=ait_const_sleb128bit;
         value:=_value;
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_uleb128bit(_value : qword);
      begin
         inherited Create;
         typ:=ait_const_uleb128bit;
         value:=int64(_value);
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_aint(_value : aint);
      begin
         inherited Create;
         typ:=ait_const_aint;
         value:=_value;
         sym:=nil;
         endsym:=nil;
      end;


    constructor tai_const.Create_sym(_sym:tasmsymbol);
      begin
         inherited Create;
         typ:=ait_const_ptr;
         { sym is allowed to be nil, this is used to write nil pointers }
         sym:=_sym;
         endsym:=nil;
         value:=0;
         { update sym info }
         if assigned(sym) then
           sym.increfs;
      end;


    constructor tai_const.Create_sym_offset(_sym:tasmsymbol;ofs:aint);
      begin
         inherited Create;
         typ:=ait_const_ptr;
         if not assigned(_sym) then
           internalerror(200404121);
         sym:=_sym;
         endsym:=nil;
         value:=ofs;
         { update sym info }
         sym.increfs;
      end;


    constructor tai_const.Create_rel_sym(_typ:taitype;_sym,_endsym:tasmsymbol);
      begin
         inherited Create;
         typ:=_typ;
         sym:=_sym;
         endsym:=_endsym;
         value:=0;
         { update sym info }
         sym.increfs;
         endsym.increfs;
      end;


    constructor tai_const.Create_rva_sym(_sym:tasmsymbol);
      begin
         inherited Create;
         typ:=ait_const_rva_symbol;
         sym:=_sym;
         endsym:=nil;
         value:=0;
         { update sym info }
         sym.increfs;
      end;


    constructor tai_const.Create_indirect_sym(_sym:tasmsymbol);
      begin
         inherited Create;
         typ:=ait_const_indirect_symbol;
         sym:=_sym;
         endsym:=nil;
         value:=0;
         { update sym info }
         sym.increfs;
      end;


    constructor tai_const.Createname(const name:string;_symtyp:Tasmsymtype;ofs:aint);
      begin
         inherited Create;
         typ:=ait_const_ptr;
         sym:=objectlibrary.newasmsymbol(name,AB_EXTERNAL,_symtyp);
         endsym:=nil;
         value:=ofs;
         { update sym info }
         sym.increfs;
      end;


    constructor tai_const.Createname_rva(const name:string);
      begin
         inherited Create;
         typ:=ait_const_rva_symbol;
         sym:=objectlibrary.newasmsymbol(name,AB_EXTERNAL,AT_FUNCTION);
         endsym:=nil;
         value:=0;
         { update sym info }
         sym.increfs;
      end;


    constructor tai_const.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        sym:=ppufile.getasmsymbol;
        endsym:=ppufile.getasmsymbol;
        value:=ppufile.getint64;
      end;


    procedure tai_const.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putasmsymbol(sym);
        ppufile.putasmsymbol(endsym);
        ppufile.putint64(value);
      end;


    procedure tai_const.derefimpl;
      begin
        objectlibrary.DerefAsmsymbol(sym);
        objectlibrary.DerefAsmsymbol(endsym);
      end;


    function tai_const.getcopy:tlinkedlistitem;
      begin
        getcopy:=inherited getcopy;
        { we need to increase the reference number }
        sym.increfs;
        if assigned(endsym) then
          endsym.increfs;
      end;


    function tai_const.size:longint;
      begin
        case typ of
          ait_const_8bit :
            result:=1;
          ait_const_16bit :
            result:=2;
          ait_const_32bit :
            result:=4;
          ait_const_64bit :
            result:=8;
          ait_const_indirect_symbol,
          ait_const_rva_symbol :
            result:=sizeof(aint);
        end;
      end;


{****************************************************************************
                               TAI_real_32bit
 ****************************************************************************}

    constructor tai_real_32bit.Create(_value : ts32real);

      begin
         inherited Create;
         typ:=ait_real_32bit;
         value:=_value;
      end;

    constructor tai_real_32bit.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        value:=ppufile.getreal;
      end;


    procedure tai_real_32bit.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putreal(value);
      end;


{****************************************************************************
                               TAI_real_64bit
 ****************************************************************************}

    constructor tai_real_64bit.Create(_value : ts64real);

      begin
         inherited Create;
         typ:=ait_real_64bit;
         value:=_value;
      end;


{$ifdef ARM}
    constructor tai_real_64bit.Create_hiloswapped(_value : ts64real);

      begin
         inherited Create;
         typ:=ait_real_64bit;
         value:=_value;
         formatoptions:=fo_hiloswapped;
      end;
{$endif ARM}

    constructor tai_real_64bit.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        value:=ppufile.getreal;
{$ifdef ARM}
        formatoptions:=tformatoptions(ppufile.getbyte);
{$endif ARM}
      end;


    procedure tai_real_64bit.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putreal(value);
{$ifdef ARM}
        ppufile.putbyte(byte(formatoptions));
{$endif ARM}
      end;


{****************************************************************************
                               TAI_real_80bit
 ****************************************************************************}

    constructor tai_real_80bit.Create(_value : ts80real);

      begin
         inherited Create;
         typ:=ait_real_80bit;
         value:=_value;
      end;


    constructor tai_real_80bit.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        value:=ppufile.getreal;
      end;


    procedure tai_real_80bit.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putreal(value);
      end;


{****************************************************************************
                               TAI_real_80bit
 ****************************************************************************}

    constructor tai_real_128bit.Create(_value : ts128real);

      begin
         inherited Create;
         typ:=ait_real_128bit;
         value:=_value;
      end;


    constructor tai_real_128bit.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        value:=ppufile.getreal;
      end;


    procedure tai_real_128bit.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putreal(value);
      end;


{****************************************************************************
                               Tai_comp_64bit
 ****************************************************************************}

    constructor tai_comp_64bit.Create(_value : ts64comp);

      begin
         inherited Create;
         typ:=ait_comp_64bit;
         value:=_value;
      end;


    constructor tai_comp_64bit.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        ppufile.putdata(value,sizeof(value));
      end;


    procedure tai_comp_64bit.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.getdata(value,sizeof(value));
      end;


{****************************************************************************
                               TAI_STRING
 ****************************************************************************}

     constructor tai_string.Create(const _str : string);

       begin
          inherited Create;
          typ:=ait_string;
          len:=length(_str);
          getmem(str,len+1);
          strpcopy(str,_str);
       end;

     constructor tai_string.Create_pchar(_str : pchar);

       begin
          inherited Create;
          typ:=ait_string;
          str:=_str;
          len:=strlen(_str);
       end;

    constructor tai_string.Create_length_pchar(_str : pchar;length : longint);

       begin
          inherited Create;
          typ:=ait_string;
          str:=_str;
          len:=length;
       end;

    destructor tai_string.destroy;

      begin
         { you can have #0 inside the strings so }
         if str<>nil then
           freemem(str,len+1);
         inherited Destroy;
      end;


    constructor tai_string.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        len:=ppufile.getlongint;
        getmem(str,len+1);
        ppufile.getdata(str^,len);
        str[len]:=#0;
      end;


    procedure tai_string.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putlongint(len);
        ppufile.putdata(str^,len);
      end;


    function tai_string.getcopy : tlinkedlistitem;
      var
        p : tlinkedlistitem;
      begin
        p:=inherited getcopy;
        getmem(tai_string(p).str,len+1);
        move(str^,tai_string(p).str^,len+1);
        getcopy:=p;
      end;


{****************************************************************************
                               TAI_LABEL
 ****************************************************************************}

    constructor tai_label.create(_l : tasmlabel);
      begin
        inherited Create;
        typ:=ait_label;
        l:=_l;
        l.is_set:=true;
        is_global:=(l.defbind=AB_GLOBAL);
      end;


    constructor tai_label.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        l:=tasmlabel(ppufile.getasmsymbol);
        is_global:=boolean(ppufile.getbyte);
      end;


    procedure tai_label.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putasmsymbol(l);
        ppufile.putbyte(byte(is_global));
      end;


    procedure tai_label.derefimpl;
      begin
        objectlibrary.DerefAsmsymbol(tasmsymbol(l));
        l.is_set:=true;
      end;


{****************************************************************************
                              TAI_DIRECT
 ****************************************************************************}

     constructor tai_direct.Create(_str : pchar);

       begin
          inherited Create;
          typ:=ait_direct;
          str:=_str;
       end;

    destructor tai_direct.destroy;

      begin
         strdispose(str);
         inherited Destroy;
      end;

    constructor tai_direct.ppuload(t:taitype;ppufile:tcompilerppufile);
      var
        len : longint;
      begin
        inherited ppuload(t,ppufile);
        len:=ppufile.getlongint;
        getmem(str,len+1);
        ppufile.getdata(str^,len);
        str[len]:=#0;
      end;


    procedure tai_direct.ppuwrite(ppufile:tcompilerppufile);
      var
        len : longint;
      begin
        inherited ppuwrite(ppufile);
        len:=strlen(str);
        ppufile.putlongint(len);
        ppufile.putdata(str^,len);
      end;


    function tai_direct.getcopy : tlinkedlistitem;
      var
        p : tlinkedlistitem;
      begin
        p:=inherited getcopy;
        getmem(tai_direct(p).str,strlen(str)+1);
        move(str^,tai_direct(p).str^,strlen(str)+1);
        getcopy:=p;
      end;


{****************************************************************************
          tai_comment  comment to be inserted in the assembler file
 ****************************************************************************}

     constructor tai_comment.Create(_str : pchar);

       begin
          inherited Create;
          typ:=ait_comment;
          str:=_str;
       end;

    destructor tai_comment.destroy;

      begin
         strdispose(str);
         inherited Destroy;
      end;

    constructor tai_comment.ppuload(t:taitype;ppufile:tcompilerppufile);
      var
        len : longint;
      begin
        inherited ppuload(t,ppufile);
        len:=ppufile.getlongint;
        getmem(str,len+1);
        ppufile.getdata(str^,len);
        str[len]:=#0;
      end;


    procedure tai_comment.ppuwrite(ppufile:tcompilerppufile);
      var
        len : longint;
      begin
        inherited ppuwrite(ppufile);
        len:=strlen(str);
        ppufile.putlongint(len);
        ppufile.putdata(str^,len);
      end;


    function tai_comment.getcopy : tlinkedlistitem;
      var
        p : tlinkedlistitem;
      begin
        p:=inherited getcopy;
        getmem(tai_comment(p).str,strlen(str)+1);
        move(str^,tai_comment(p).str^,strlen(str)+1);
        getcopy:=p;
      end;


{****************************************************************************
                              TAI_CUTOBJECT
 ****************************************************************************}

     constructor tai_cutobject.Create;
       begin
          inherited Create;
          typ:=ait_cutobject;
          place:=cut_normal;
       end;


     constructor tai_cutobject.Create_begin;
       begin
          inherited Create;
          typ:=ait_cutobject;
          place:=cut_begin;
       end;


     constructor tai_cutobject.Create_end;
       begin
          inherited Create;
          typ:=ait_cutobject;
          place:=cut_end;
       end;


    constructor tai_cutobject.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        place:=TCutPlace(ppufile.getbyte);
      end;


    procedure tai_cutobject.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putbyte(byte(place));
      end;


{****************************************************************************
                             Tai_Marker
 ****************************************************************************}

    constructor Tai_Marker.Create(_Kind: TMarker);
      begin
        Inherited Create;
        typ := ait_marker;
        Kind := _Kind;
      end;


    constructor Tai_Marker.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        kind:=TMarker(ppufile.getbyte);
      end;


    procedure Tai_Marker.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putbyte(byte(kind));
      end;


{*****************************************************************************
                                tai_tempalloc
*****************************************************************************}

    constructor tai_tempalloc.alloc(pos,size:longint);
      begin
        inherited Create;
        typ:=ait_tempalloc;
        allocation:=true;
        temppos:=pos;
        tempsize:=size;
{$ifdef EXTDEBUG}
        problem:=nil;
{$endif EXTDEBUG}
      end;


    destructor tai_tempalloc.destroy;
      begin
{$ifdef EXTDEBUG}
        stringdispose(problem);
{$endif EXTDEBUG}
        inherited destroy;
      end;


    constructor tai_tempalloc.dealloc(pos,size:longint);
      begin
        inherited Create;
        typ:=ait_tempalloc;
        allocation:=false;
        temppos:=pos;
        tempsize:=size;
{$ifdef EXTDEBUG}
        problem:=nil;
{$endif EXTDEBUG}
      end;


{$ifdef EXTDEBUG}
    constructor tai_tempalloc.allocinfo(pos,size:longint;const st:string);
      begin
        inherited Create;
        typ:=ait_tempalloc;
        allocation:=false;
        temppos:=pos;
        tempsize:=size;
        problem:=stringdup(st);
      end;
{$endif EXTDEBUG}


    constructor tai_tempalloc.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        temppos:=ppufile.getlongint;
        tempsize:=ppufile.getlongint;
        allocation:=boolean(ppufile.getbyte);
{$ifdef EXTDEBUG}
        problem:=nil;
{$endif EXTDEBUG}
      end;


    procedure tai_tempalloc.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putlongint(temppos);
        ppufile.putlongint(tempsize);
        ppufile.putbyte(byte(allocation));
      end;


{*****************************************************************************
                                 tai_regalloc
*****************************************************************************}

    constructor tai_regalloc.alloc(r : tregister;ainstr:tai);
      begin
        inherited create;
        typ:=ait_regalloc;
        ratype:=ra_alloc;
        reg:=r;
        { ainstr must be an instruction }
        if assigned(ainstr) and
           (ainstr.typ<>ait_instruction) then
          internalerror(200411011);
        instr:=ainstr;
      end;


    constructor tai_regalloc.dealloc(r : tregister;ainstr:tai);
      begin
        inherited create;
        typ:=ait_regalloc;
        ratype:=ra_dealloc;
        reg:=r;
        { ainstr must be an instruction }
        if assigned(ainstr) and
           (ainstr.typ<>ait_instruction) then
          internalerror(200411012);
        instr:=ainstr;
      end;


    constructor tai_regalloc.sync(r : tregister);
      begin
        inherited create;
        typ:=ait_regalloc;
        ratype:=ra_sync;
        reg:=r;
      end;


    constructor tai_regalloc.resize(r : tregister);
      begin
        inherited create;
        typ:=ait_regalloc;
        ratype:=ra_resize;
        reg:=r;
      end;


    constructor tai_regalloc.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        ppufile.getdata(reg,sizeof(Tregister));
        ratype:=tregalloctype(ppufile.getbyte);
      end;


    procedure tai_regalloc.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putdata(reg,sizeof(Tregister));
        ppufile.putbyte(byte(ratype));
      end;


{*****************************************************************************
                               TaiInstruction
*****************************************************************************}

    constructor tai_cpu_abstract.Create(op : tasmop);

      begin
         inherited create;
         typ:=ait_instruction;
         is_jmp:=false;
         opcode:=op;
         ops:=0;
         fillchar(condition,sizeof(condition),0);
         fillchar(oper,sizeof(oper),0);
      end;


    destructor tai_cpu_abstract.Destroy;
      var
        i : integer;
      begin
        for i:=0 to opercnt-1 do
          begin
            clearop(i);
            dispose(oper[i]);
          end;
        inherited destroy;
      end;


{ ---------------------------------------------------------------------
    Loading of operands.
  ---------------------------------------------------------------------}

    procedure tai_cpu_abstract.allocate_oper(opers:longint);
      begin
        while (opers>opercnt) do
          begin
            new(oper[opercnt]);
            fillchar(oper[opercnt]^,sizeof(toper),0);
            inc(opercnt);
          end;
      end;


    procedure tai_cpu_abstract.loadconst(opidx:longint;l:aint);
      begin
        allocate_oper(opidx+1);
        with oper[opidx]^ do
         begin
           if typ<>top_const then
             clearop(opidx);
           val:=l;
           typ:=top_const;
         end;
      end;


    procedure tai_cpu_abstract.loadsymbol(opidx:longint;s:tasmsymbol;sofs:longint);
      var
        r : treference;
      begin
        reference_reset_symbol(r,s,sofs);
        r.refaddr:=addr_full;
        loadref(opidx,r);
      end;


    procedure tai_cpu_abstract.loadlocal(opidx:longint;s:pointer;sofs:longint;indexreg:tregister;scale:byte;getoffset,forceref:boolean);
      begin
        if not assigned(s) then
         internalerror(200204251);
        allocate_oper(opidx+1);
        with oper[opidx]^ do
         begin
           if typ<>top_local then
             begin
               clearop(opidx);
               new(localoper);
             end;
           with oper[opidx]^.localoper^ do
             begin
               localsym:=s;
               localsymofs:=sofs;
               localindexreg:=indexreg;
               localscale:=scale;
               localgetoffset:=getoffset;
               localforceref:=forceref;
             end;
           typ:=top_local;
         end;
      end;


    procedure tai_cpu_abstract.loadref(opidx:longint;const r:treference);
      begin
        allocate_oper(opidx+1);
        with oper[opidx]^ do
          begin
            if typ<>top_ref then
              begin
                clearop(opidx);
                new(ref);
              end;

            ref^:=r;
{$ifdef x86}
            { We allow this exception for x86, since overloading this would be
              too much of a a speed penalty}
            if (ref^.segment<>NR_NO) and (ref^.segment<>NR_DS) then
              segprefix:=ref^.segment;
{$endif}
{$ifdef extdebug}
            if (cs_create_pic in aktmoduleswitches) and
              assigned(r.symbol) and
              (r.refaddr=addr_no) then
              internalerror(200502052);
{$endif}
            typ:=top_ref;
            if assigned(add_reg_instruction_hook) then
              begin
                add_reg_instruction_hook(self,ref^.base);
                add_reg_instruction_hook(self,ref^.index);
              end;
            { mark symbol as used }
            if assigned(ref^.symbol) then
              ref^.symbol.increfs;
          end;
      end;


    procedure tai_cpu_abstract.loadreg(opidx:longint;r:tregister);
      begin
        allocate_oper(opidx+1);
        with oper[opidx]^ do
         begin
           if typ<>top_reg then
             clearop(opidx);
           reg:=r;
           typ:=top_reg;
         end;
        if assigned(add_reg_instruction_hook) then
          add_reg_instruction_hook(self,r);
{$ifdef ARM}
        { R15 is the PC on the ARM thus moves to R15 are jumps.
          Due to speed considerations we don't use a virtual overridden method here.
          Because the pc/r15 isn't handled by the reg. allocator this should never cause
          problems with iregs getting r15.
        }
        is_jmp:=(opcode=A_MOV) and (opidx=0) and (r=NR_R15);
{$endif ARM}
      end;


    procedure tai_cpu_abstract.loadoper(opidx:longint;o:toper);
      begin
        allocate_oper(opidx+1);
        clearop(opidx);
        oper[opidx]^:=o;
        { copy also the reference }
        with oper[opidx]^ do
          begin
            case typ of
              top_reg:
                begin
                  if assigned(add_reg_instruction_hook) then
                    add_reg_instruction_hook(self,reg);
                end;
              top_ref:
                begin
                  new(ref);
                  ref^:=o.ref^;
                  if assigned(add_reg_instruction_hook) then
                    begin
                      add_reg_instruction_hook(self,ref^.base);
                      add_reg_instruction_hook(self,ref^.index);
                    end;
                end;
{$ifdef ARM}
              top_shifterop:
                begin
                  new(shifterop);
                  shifterop^:=o.shifterop^;
                  if assigned(add_reg_instruction_hook) then
                    add_reg_instruction_hook(self,shifterop^.rs);
                end;
{$endif ARM}
             end;
          end;
      end;


    procedure tai_cpu_abstract.clearop(opidx:longint);
      begin
        with oper[opidx]^ do
          begin
            case typ of
              top_ref:
                dispose(ref);
              top_local:
                dispose(localoper);
{$ifdef ARM}
              top_shifterop:
                dispose(shifterop);
              top_regset:
                dispose(regset);
{$endif ARM}
            end;
            typ:=top_none;
          end;
      end;


{ ---------------------------------------------------------------------
    Miscellaneous methods.
  ---------------------------------------------------------------------}

    procedure tai_cpu_abstract.SetCondition(const c:TAsmCond);
      begin
         condition:=c;
      end;


    Function tai_cpu_abstract.getcopy:TLinkedListItem;
      var
        i : longint;
        p : tai_cpu_abstract;
      begin
        p:=tai_cpu_abstract(inherited getcopy);
        { make a copy of the references }
        p.opercnt:=0;
        p.allocate_oper(ops);
        for i:=0 to ops-1 do
          begin
            p.oper[i]^:=oper[i]^;
            case oper[i]^.typ of
              top_local :
                begin
                  new(p.oper[i]^.localoper);
                  p.oper[i]^.localoper^:=oper[i]^.localoper^;
                end;
              top_ref :
                begin
                  new(p.oper[i]^.ref);
                  p.oper[i]^.ref^:=oper[i]^.ref^;
                end;
{$ifdef ARM}
              top_shifterop:
                begin
                  new(p.oper[i]^.shifterop);
                  p.oper[i]^.shifterop^:=oper[i]^.shifterop^;
                end;
{$endif ARM}
            end;
          end;
        getcopy:=p;
      end;


    function tai_cpu_abstract.is_same_reg_move(regtype: Tregistertype):boolean;
      begin
        { When the generic RA is used this needs to be overriden, we don't use
          virtual;abstract; to prevent a lot of warnings of unimplemented abstract methods
          when tai_cpu is created (PFV) }
        internalerror(200404091);
      end;


    function tai_cpu_abstract.spilling_get_operation_type(opnr: longint): topertype;
      begin
        internalerror(200404091);
      end;


    function tai_cpu_abstract.spilling_get_operation_type_ref(opnr: longint; reg: tregister): topertype;
      begin
        result := operand_read;
      end;


    constructor tai_cpu_abstract.ppuload(t:taitype;ppufile:tcompilerppufile);
      var
        i : integer;
      begin
        inherited ppuload(t,ppufile);
        { hopefully, we don't get problems with big/litte endian here when cross compiling :/ }
        ppufile.getdata(condition,sizeof(tasmcond));
        allocate_oper(ppufile.getbyte);
        for i:=0 to ops-1 do
          ppuloadoper(ppufile,oper[i]^);
        opcode:=tasmop(ppufile.getword);
{$ifdef x86}
        ppufile.getdata(segprefix,sizeof(Tregister));
{$endif x86}
        is_jmp:=boolean(ppufile.getbyte);
      end;


    procedure tai_cpu_abstract.ppuwrite(ppufile:tcompilerppufile);
      var
        i : integer;
      begin
        inherited ppuwrite(ppufile);
        ppufile.putdata(condition,sizeof(tasmcond));
        ppufile.putbyte(ops);
        for i:=0 to ops-1 do
          ppuwriteoper(ppufile,oper[i]^);
        ppufile.putword(word(opcode));
{$ifdef x86}
        ppufile.putdata(segprefix,sizeof(Tregister));
{$endif x86}
        ppufile.putbyte(byte(is_jmp));
      end;


    procedure tai_cpu_abstract.buildderefimpl;
      var
        i : integer;
      begin
        for i:=0 to ops-1 do
          ppubuildderefimploper(oper[i]^);
      end;


    procedure tai_cpu_abstract.derefimpl;
      var
        i : integer;
      begin
        for i:=0 to ops-1 do
          ppuderefoper(oper[i]^);
      end;


{****************************************************************************
                              tai_align_abstract
 ****************************************************************************}

     constructor tai_align_abstract.Create(b: byte);
       begin
          inherited Create;
          typ:=ait_align;
          if b in [1,2,4,8,16,32] then
            aligntype := b
          else
            aligntype := 1;
          fillsize:=0;
          fillop:=0;
          use_op:=false;
       end;


     constructor tai_align_abstract.Create_op(b: byte; _op: byte);
       begin
          inherited Create;
          typ:=ait_align;
          if b in [1,2,4,8,16,32] then
            aligntype := b
          else
            aligntype := 1;
          fillsize:=0;
          fillop:=_op;
          use_op:=true;
       end;


     constructor tai_align_abstract.Create_zeros(b: byte);
       begin
          inherited Create;
          typ:=ait_align;
          if b in [1,2,4,8,16,32] then
            aligntype := b
          else
            aligntype := 1;
         use_op:=true;
         fillsize:=0;
         fillop:=0;
       end;


     function tai_align_abstract.calculatefillbuf(var buf : tfillbuffer):pchar;
       begin
         if fillsize>sizeof(buf) then
           internalerror(200404293);
         fillchar(buf,high(buf),fillop);
         calculatefillbuf:=pchar(@buf);
       end;


    constructor tai_align_abstract.ppuload(t:taitype;ppufile:tcompilerppufile);
      begin
        inherited ppuload(t,ppufile);
        aligntype:=ppufile.getbyte;
        fillsize:=0;
        fillop:=ppufile.getbyte;
        use_op:=boolean(ppufile.getbyte);
      end;


    procedure tai_align_abstract.ppuwrite(ppufile:tcompilerppufile);
      begin
        inherited ppuwrite(ppufile);
        ppufile.putbyte(aligntype);
        ppufile.putbyte(fillop);
        ppufile.putbyte(byte(use_op));
      end;


{*****************************************************************************
                                 TAAsmOutput
*****************************************************************************}

    constructor taasmoutput.create;
      begin
        inherited create;
        { make sure the optimizer won't remove the first tai of this list}
        insert(tai_marker.create(marker_blockstart));
      end;


    function taasmoutput.empty : boolean;
      begin
        { there is always a marker_blockstart available,
          see taasmoutput.create }
        result:=(count<=1);
      end;


    function taasmoutput.getlasttaifilepos : pfileposinfo;
      var
       hp : tlinkedlistitem;
      begin
         getlasttaifilepos := nil;
         if assigned(last) then
           begin
              { find the last file information record }
              if not (tai(last).typ in SkipLineInfo) then
                getlasttaifilepos:=@tailineinfo(last).fileinfo
              else
               { go through list backwards to find the first entry
                 with line information
               }
               begin
                 hp:=tai(last);
                 while assigned(hp) and (tai(hp).typ in SkipLineInfo) do
                    hp:=hp.Previous;
                 { found entry }
                 if assigned(hp) then
                   getlasttaifilepos:=@tailineinfo(hp).fileinfo
               end;
           end;
      end;

    procedure Taasmoutput.InsertAfter(Item,Loc : TLinkedListItem);

      begin
        { This is not possible because it is not sure that the
          tai at Loc has taifileinfo as parent }
        {if assigned(Loc) then
          tailineinfo(Item).fileinfo:=tailineinfo(Loc).fileinfo;}
        inherited InsertAfter(Item,Loc);
      end;

begin
  cai_cpu:=tai_cpu_abstract;
  cai_align:=tai_align_abstract;
end.
{
  $Log: aasmtai.pas,v $
  Revision 1.96  2005/02/18 23:37:51  jonas
    * fixed spilling for several ppc instructions which only read registers
    + added support for registers in references that get changed (load/store
      with update)

  Revision 1.95  2005/02/14 17:13:06  peter
    * truncate log

  Revision 1.94  2005/02/06 00:05:56  florian
    + x86_64 pic draft

  Revision 1.93  2005/01/31 17:07:50  peter
    * fix [regpara] in intel assembler

}
