/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.remoting.transport.multiplex.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.HashMap;

import javax.net.ServerSocketFactory;
import javax.net.SocketFactory;
import javax.net.ssl.SSLServerSocket;

import junit.framework.TestCase;

import org.jboss.logging.Logger;
import org.jboss.remoting.security.SSLSocketBuilder;
import org.jboss.remoting.transport.PortUtil;
import org.jboss.remoting.transport.multiplex.MasterServerSocket;
import org.jboss.remoting.transport.multiplex.Multiplex;
import org.jboss.remoting.transport.multiplex.VirtualServerSocket;
import org.jboss.remoting.transport.multiplex.VirtualSocket;

/**
 * 
 * @author <a href="mailto:ron.sigal@jboss.com">Ron Sigal</a>
 * <p>
 * Copyright (c) Jun 30, 2006
 * </p>
 */
public class SSLSocketTestCase extends TestCase
{
   protected Logger log = Logger.getLogger(SSLSocketTestCase.class);
   protected ServerSocket serverSocket;
   protected InetAddress serverAddress;
   protected int serverPort;
   
   public void setUp()
   {
      try
      {
         HashMap config = new HashMap();
         config.put(SSLSocketBuilder.REMOTING_KEY_STORE_TYPE, "JKS");
         String keyStoreFilePath = this.getClass().getResource(".keystore").getFile();
         config.put(SSLSocketBuilder.REMOTING_KEY_STORE_FILE_PATH, keyStoreFilePath);
         config.put(SSLSocketBuilder.REMOTING_KEY_STORE_PASSWORD, "unit-tests-server");
         SSLSocketBuilder builder = new SSLSocketBuilder(config);
         builder.setUseSSLServerSocketFactory(false);
         ServerSocketFactory serverSocketFactory = builder.createSSLServerSocketFactory();
         config.put(Multiplex.SERVER_SOCKET_FACTORY, serverSocketFactory);
         
         serverPort = PortUtil.findFreePort(getHostName());
         serverAddress = InetAddress.getByName(getHostName());
         serverSocket = new MasterServerSocket(serverPort, 100, serverAddress, config);
         
         new Thread()
         {
            public void run()
            {
               try
               {
                  Socket socket = serverSocket.accept();
                  InputStream is = socket.getInputStream();
                  OutputStream os = socket.getOutputStream();
                  os.write(is.read());
                  socket.close();
                  serverSocket.close();
               }
               catch (IOException e)
               {
                  log.error(e);
                  e.printStackTrace();
               }
            }
         }.start();
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
   }
   
   
   public void testBindThenConnect()
   {
      log.info("entering " + getName());
      
      try
      {
         HashMap config = new HashMap();
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_TYPE, "JKS");
         String trustStoreFilePath = this.getClass().getResource(".truststore").getFile();
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_FILE_PATH, trustStoreFilePath);
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_PASSWORD, "unit-tests-client");
         SSLSocketBuilder builder = new SSLSocketBuilder(config);
         builder.setUseSSLSocketFactory(false);
         SocketFactory socketFactory = builder.createSSLSocketFactory();
         config.put(Multiplex.SOCKET_FACTORY, socketFactory);
         
         VirtualSocket socket = new VirtualSocket(config);
         InetAddress bindAddress = InetAddress.getByName(getHostName());
         int bindPort = PortUtil.findFreePort(getHostName());
         InetSocketAddress bindSocketAddress = new InetSocketAddress(bindAddress, bindPort);
         InetSocketAddress connectSocketAddress = new InetSocketAddress(serverAddress, serverPort);
         
         socket.bind(bindSocketAddress);
         socket.connect(connectSocketAddress);
         InputStream is = socket.getInputStream();
         OutputStream os = socket.getOutputStream();
         os.write(7);
         assertEquals(7, is.read());
         socket.close();
         
         log.info(getName() + " PASSES");
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         log.error(getName() + " FAILS");
         fail();
      }
   }
   
   
   public void testBindAndConnect()
   {
      log.info("entering " + getName());
      
      try
      {
         HashMap config = new HashMap();
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_TYPE, "JKS");
         String trustStoreFilePath = this.getClass().getResource(".truststore").getFile();
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_FILE_PATH, trustStoreFilePath);
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_PASSWORD, "unit-tests-client");
         SSLSocketBuilder builder = new SSLSocketBuilder(config);
         builder.setUseSSLSocketFactory(false);
         SocketFactory socketFactory = builder.createSSLSocketFactory();
         config.put(Multiplex.SOCKET_FACTORY, socketFactory);
         
         VirtualSocket socket = new VirtualSocket(config);
         InetAddress bindAddress = InetAddress.getByName(getHostName());
         int bindPort = PortUtil.findFreePort(getHostName());
         InetSocketAddress bindSocketAddress = new InetSocketAddress(bindAddress, bindPort);
         InetSocketAddress connectSocketAddress = new InetSocketAddress(serverAddress, serverPort);
         
         socket.connect(connectSocketAddress, bindSocketAddress, 0);
         InputStream is = socket.getInputStream();
         OutputStream os = socket.getOutputStream();
         os.write(7);
         assertEquals(7, is.read());
         socket.close();
         
         log.info(getName() + " PASSES");
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         log.error(getName() + " FAILS");
         fail();
      }
   }
   
   
   public void testConnect()
   {
      log.info("entering " + getName());
      
      try
      {
         HashMap config = new HashMap();
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_TYPE, "JKS");
         String trustStoreFilePath = this.getClass().getResource(".truststore").getFile();
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_FILE_PATH, trustStoreFilePath);
         config.put(SSLSocketBuilder.REMOTING_TRUST_STORE_PASSWORD, "unit-tests-client");
         SSLSocketBuilder builder = new SSLSocketBuilder(config);
         builder.setUseSSLSocketFactory(false);
         SocketFactory socketFactory = builder.createSSLSocketFactory();
         config.put(Multiplex.SOCKET_FACTORY, socketFactory);
         
         VirtualSocket socket = new VirtualSocket(config);
         InetSocketAddress connectSocketAddress = new InetSocketAddress(serverAddress, serverPort);
         socket.connect(connectSocketAddress);
         InputStream is = socket.getInputStream();
         OutputStream os = socket.getOutputStream();
         os.write(7);
         assertEquals(7, is.read());
         socket.close();
         
         log.info(getName() + " PASSES");
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         log.error(getName() + " FAILS");
         fail();
      }
   }
   
   
   protected String getHostName()
   {
      return "localhost";
   }
}
