
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.component;

import java.awt.event.*;
import java.beans.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import org.gjt.btools.image.Standard16;

/**
 * An edit menu that operates upon a particular component.
 * Standard commands such as cut, copy, paste and select all are provided.
 * The target component may be changed repeatedly during the lifetime
 * of this edit menu.
 */
public class EditMenu extends JMenu implements ActionListener,
        ChangeListener, PropertyChangeListener {
    /**
     * A specific menu item.
     */
    private JMenuItem cut;
    /**
     * A specific menu item.
     */
    private JMenuItem copy;
    /**
     * A specific menu item.
     */
    private JMenuItem paste;
    /**
     * A specific menu item.
     */
    private JMenuItem selectAll;

    /**
     * The text component upon which we operate.
     */
    private JTextComponent text;
    /**
     * Is the associated text component read-only?
     */
    private boolean readonly;

    /**
     * Create a new edit menu that does not operate upon any component.
     */
    public EditMenu() {
        this(null);
    }

    /**
     * Create a new edit menu that operates upon the given text component.
     *
     * @param text the text component upon which this menu will operate.
     */
    public EditMenu(JTextComponent text) {
        super("Edit");
        setMnemonic(KeyEvent.VK_E);
        this.text = null;
        this.readonly = false;

        cut = new JMenuItem("Cut", Standard16.cut.image());
        cut.setMnemonic(KeyEvent.VK_U);
        cut.setAccelerator(KeyStroke.getKeyStroke(
            KeyEvent.VK_X, ActionEvent.CTRL_MASK));
        copy = new JMenuItem("Copy", Standard16.copy.image());
        copy.setMnemonic(KeyEvent.VK_C);
        copy.setAccelerator(KeyStroke.getKeyStroke(
            KeyEvent.VK_C, ActionEvent.CTRL_MASK));
        paste = new JMenuItem("Paste", Standard16.paste.image());
        paste.setMnemonic(KeyEvent.VK_P);
        paste.setAccelerator(KeyStroke.getKeyStroke(
            KeyEvent.VK_V, ActionEvent.CTRL_MASK));
        selectAll = new JMenuItem("Select All");
        selectAll.setMnemonic(KeyEvent.VK_A);
        selectAll.setAccelerator(KeyStroke.getKeyStroke(
            KeyEvent.VK_A, ActionEvent.CTRL_MASK));
        add(cut);
        add(copy);
        add(paste);
        add(selectAll);
        cut.addActionListener(this);
        copy.addActionListener(this);
        paste.addActionListener(this);
        selectAll.addActionListener(this);

        setTarget(text);
    }

    /**
     * Reset this edit menu to operate upon the given text component.
     * If this menu was previously operating upon a different component,
     * it will abandon the old component in favour of the new one.
     *
     * @param newText the text component upon which this menu should
     * now operate, or <tt>null</tt> if this menu should not operate upon
     * any component.
     */
    public void setTarget(JTextComponent newText) {
        if (text != null) {
            text.getCaret().removeChangeListener(this);
            text.removePropertyChangeListener("editable", this);
        }

        text = newText;

        if (text == null)
            setEnabled(false);
        else {
            readonly = ! text.isEditable();
            text.getCaret().addChangeListener(this);
            text.addPropertyChangeListener("editable", this);
            updateButtonStates(text.getCaret());
            setEnabled(true);
        }
    }

    /**
     * Called when one of the menu items is selected.
     *
     * @param e the corresponding event.
     */
    public void actionPerformed(ActionEvent e) {
        if (text != null) {
            if (e.getSource() == cut)
                text.cut();
            else if (e.getSource() == copy)
                text.copy();
            else if (e.getSource() == paste)
                text.paste();
            else if (e.getSource() == selectAll)
                text.selectAll();
        }
    }

    /**
     * Called when the selection state of the associated component
     * changes.
     *
     * @param e the corresponding event.
     */
    public void stateChanged(ChangeEvent e) {
        if (e.getSource() instanceof Caret)
            updateButtonStates((Caret)e.getSource());
    }

    /**
     * Called when a property of the associated component changes.
     *
     * @param e the corresponding event.
     */
    public void propertyChange(PropertyChangeEvent e) {
        if (text != null) {
            readonly = ! text.isEditable();
            updateButtonStates(text.getCaret());
        }
    }

    /**
     * Correctly sets the enabled states of the individual menu items.
     * The cut and copy items will only be enabled if something is
     * selected in the associated component.  The cut and paste items
     * will only be enabled if the associated component is editable.
     *
     * @param caret the caret describing the current selection state in
     * the associated component.  This may be <tt>null</tt>, in which
     * case it will be assumed that nothing is selected.
     */
    private void updateButtonStates(Caret caret) {
        boolean sel = (caret == null ? false :
            (caret.getDot() != caret.getMark()));
        cut.setEnabled(sel && ! readonly);
        copy.setEnabled(sel);
        paste.setEnabled(! readonly);
    }
}

