/*
A library to allow applictions to provide simple indications of
information to be displayed to users of the application through the
interface shell.

Copyright 2009 Canonical Ltd.

Authors:
    Ted Gould <ted@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of either or both of the following licenses:

1) the GNU Lesser General Public License version 3, as published by the 
Free Software Foundation; and/or
2) the GNU Lesser General Public License version 2.1, as published by 
the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the applicable version of the GNU Lesser General Public 
License for more details.

You should have received a copy of both the GNU Lesser General Public 
License version 3 and version 2.1 along with this program.  If not, see 
<http://www.gnu.org/licenses/>
*/
 
#include "server.h"
#include "interests-priv.h"
#include "indicate-marshal.h"
#include <gio/gio.h>
#include <libdbusmenu-glib/server.h>
#include "dbus-shared.h"
#include "gen-indicate-interface.xml.h"

/* Errors */
enum {
	NO_GET_DESKTOP,
	NO_GET_INDICATOR_COUNT,
	NO_GET_INDICATOR_LIST,
	NO_GET_INDICATOR_PROPERTY,
	NO_GET_INDICATOR_PROPERTY_GROUP,
	NO_GET_INDICATOR_PROPERTIES,
	NO_SHOW_INDICATOR_TO_USER,
	NO_INDICATOR_DISPLAYED,
	INVALID_INDICATOR_ID,
	NO_SHOW_INTEREST,
	NO_REMOVE_INTEREST,
	SHOW_INTEREST_FAILED,
	REMOVE_INTEREST_FAILED,
	NO_MAX_INDICATORS_SET,
	MAX_INDICATORS_SET_FAILED,
	NO_SUCH_PROPERTY,
	NOT_IMPLEMENTED,
	LAST_ERROR
};

/* Signals */
enum {
	INDICATOR_ADDED,
	INDICATOR_REMOVED,
	INDICATOR_MODIFIED,
	SERVER_SHOW,
	SERVER_HIDE,
	SERVER_DISPLAY,
	INTEREST_ADDED,
	INTEREST_REMOVED,
	MAX_INDICATORS_CHANGED,
	INDICATOR_NOT_SHOWN,
	SERVER_COUNT_CHANGED,
	LAST_SIGNAL
};

/* Properties */
enum {
	PROP_0,
	PROP_DESKTOP,
	PROP_TYPE,
	PROP_COUNT,
	PROP_MENU, 
	PROP_PATH,
	PROP_ICON_THEME
};

static guint signals[LAST_SIGNAL] = { 0 };

/* Private area */
typedef struct _IndicateServerPrivate IndicateServerPrivate;
struct _IndicateServerPrivate
{
	GCancellable * connection_cancel;
	GDBusConnection *connection;
	guint broadcast_signal;

	gchar * path;
	GSList * indicators;
	gboolean visible;
	guint current_id;
	guint registered;

	gchar * desktop;
	gchar * type;
	gchar * icon_theme;

	guint count;

	DbusmenuServer * dbusmenu;

	// TODO: Should have a more robust way to track this, but this'll work for now
	guint num_hidden;

	/* Folks storage */
	GList * interestedfolks;

	/* Folks caches */
	gint max_indicators;
	gboolean interests[INDICATE_INTEREST_LAST];
	gulong interest_timer;
};

#define INDICATE_SERVER_GET_PRIVATE(o) \
          (G_TYPE_INSTANCE_GET_PRIVATE ((o), INDICATE_TYPE_SERVER, IndicateServerPrivate))

typedef struct _IndicateServerInterestedFolk IndicateServerInterestedFolk;
struct _IndicateServerInterestedFolk {
	gchar * sender;
	gboolean interests[INDICATE_INTEREST_LAST];
	gint max_indicators;
	GHashTable * indicators_displayed;
	GDBusProxy * proxy;
};

static const gint MAX_INDICATORS_INFINITE = -1;
static const gint MAX_INDICATORS_UNSET = -2;

/* Define Type */
G_DEFINE_TYPE (IndicateServer, indicate_server, G_TYPE_OBJECT);

/* Prototypes */
static void indicate_server_dispose (GObject * obj);
static void indicate_server_finalize (GObject * obj);
static gboolean get_indicator_count (IndicateServer * server, guint * count, GError **error);
static gboolean get_indicator_list (IndicateServer * server, GArray ** indicators, GError ** error);
static IndicateIndicator * get_indicator (IndicateServer * server, guint id, GError **error);
static gboolean get_indicator_property (IndicateServer * server, guint id, gchar * property, GVariant ** value, GError **error);
static gboolean get_indicator_property_group (IndicateServer * server, guint id, const gchar ** properties, GVariant ** output, GError **error);
static gboolean get_indicator_properties (IndicateServer * server, guint id, gchar *** properties, GError **error);
static gboolean show_indicator_to_user (IndicateServer * server, guint id, guint timestamp, GError ** error);
static gboolean indicator_displayed (IndicateServer * server, const gchar * sender, guint id, gboolean displayed, GError ** error);
static void indicator_display_check_recalc (gpointer key, gpointer value, gpointer userdata);
static void recalculate_indicator_displayed (IndicateServer * server, guint id);
static guint get_next_id (IndicateServer * server);
static void set_property (GObject * obj, guint id, const GValue * value, GParamSpec * pspec);
static void get_property (GObject * obj, guint id, GValue * value, GParamSpec * pspec);
static gboolean show_interest (IndicateServer * server, const gchar * sender, IndicateInterests interest);
static gboolean remove_interest (IndicateServer * server, const gchar * sender, IndicateInterests interest);
static gboolean check_interest (IndicateServer * server, IndicateInterests intrest);
static gint max_indicators_get (IndicateServer * server);
static gboolean max_indicators_set (IndicateServer * server, const gchar * sender, gint max);
static void recalculate_max_indicators (IndicateServer * server);
static gint indicate_server_interested_folks_equal (gconstpointer a, gconstpointer b);
static void indicate_server_interested_folks_init (IndicateServerInterestedFolk * folk, const gchar * sender, const gchar * path, GDBusConnection * connection);
static void indicate_server_interested_folks_set (IndicateServerInterestedFolk * folk, IndicateInterests interest, gboolean value);
static void indicate_server_interested_folks_copy (IndicateServerInterestedFolk * folk, gboolean * interests);
static void indicate_server_interested_folks_destroy(IndicateServerInterestedFolk * folk);
static void bus_connection_cb (GObject * obj, GAsyncResult * res, gpointer user_data);
static void bus_broadcast_cb (GDBusConnection * connection, const gchar * sender, const gchar * object_path, const gchar * interface_name, const gchar * signal_name, GVariant * parameters, gpointer user_data);
static void folk_proxy_cb (GObject * object, GAsyncResult * res, gpointer user_data);
static void bus_method_call (GDBusConnection * connection, const gchar * sender, const gchar * path, const gchar * interface, const gchar * method, GVariant * params, GDBusMethodInvocation * invocation, gpointer user_data);
static GVariant * bus_get_prop (GDBusConnection * connection, const gchar * sender, const gchar * path, const gchar * interface, const gchar * property, GError ** error, gpointer user_data);
static void bus_get_indicator_count (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_get_indicator_list (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_get_indicator_property (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_get_indicator_property_group (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_get_indicator_properties (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_show_indicator_to_user (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_indicator_displayed (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_show_interest (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_remove_interest (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static void bus_set_max_indicators (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);
static GQuark indicate_server_error_quark (void);
static gboolean interest_timer (gpointer user_data);

/* Method Table */
typedef void (*MethodTableFunc) (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation);

typedef struct _method_table_t method_table_t;
struct _method_table_t {
	const gchar * interned_name;
	MethodTableFunc func;
};

enum {
	METHOD_GET_INDICATOR_COUNT = 0,
	METHOD_GET_INDICATOR_LIST,
	METHOD_GET_INDICATOR_PROPERTY,
	METHOD_GET_INDICATOR_PROPERTY_GROUP,
	METHOD_GET_INDICATOR_PROPERTIES,
	METHOD_SHOW_INDICATOR_TO_USER,
	METHOD_INDICATOR_DISPLAYED,
	METHOD_SHOW_INTEREST,
	METHOD_REMOVE_INTEREST,
	METHOD_SET_MAX_INDICATORS,
	/* Counter, do not remove! */
	METHOD_COUNT
};

/* Bus Stuff */
static GDBusNodeInfo *            bus_node_info = NULL;
static GDBusInterfaceInfo *       bus_interface_info = NULL;
static const GDBusInterfaceVTable bus_interface_table = {
	method_call:    bus_method_call,
	get_property:   bus_get_prop,
	set_property:   NULL /* No properties that can be set */
};
static method_table_t             bus_method_table[METHOD_COUNT];



/* Code */
static void
indicate_server_class_init (IndicateServerClass * class)
{
	/* g_debug("Server Class Initialized"); */
	GObjectClass * gobj;
	gobj = G_OBJECT_CLASS(class);

	g_type_class_add_private (class, sizeof (IndicateServerPrivate));

	gobj->dispose = indicate_server_dispose;
	gobj->finalize = indicate_server_finalize;
	gobj->set_property = set_property;
	gobj->get_property = get_property;

	/**
		IndicateServer::indicator-added:
		@arg0: The #IndicateServer object
		@arg1: The #IndicateIndicator ID number

		Emitted every time that a new indicator is made visible to
		the world.  This results in a signal on DBus.
	*/
	signals[INDICATOR_ADDED] = g_signal_new(INDICATE_SERVER_SIGNAL_INDICATOR_ADDED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, indicator_added),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__UINT,
	                                        G_TYPE_NONE, 1, G_TYPE_UINT);
	/**
		IndicateServer::indicator-removed:
		@arg0: The #IndicateServer object
		@arg1: The #IndicateIndicator ID number

		Emitted every time that a new indicator is made invisible to
		the world.  This results in a signal on DBus.
	*/
	signals[INDICATOR_REMOVED] = g_signal_new(INDICATE_SERVER_SIGNAL_INDICATOR_REMOVED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, indicator_removed),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__UINT,
	                                        G_TYPE_NONE, 1, G_TYPE_UINT);
	/**
	 * IndicateServer::indicator-modified:
	 * @arg0: The #IndicateServer object
	 * @arg1: The #IndicateIndicator ID number
	 * @arg2: The name of the property modified
	 *
	 * Emitted every time that a property on an indicator changes
	 * and it is visible to the world.  This results in a signal on DBus.
	*/
	signals[INDICATOR_MODIFIED] = g_signal_new(INDICATE_SERVER_SIGNAL_INDICATOR_MODIFIED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, indicator_modified),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__UINT_STRING,
	                                        G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_STRING);
	/**
	  * IndicateServer::server-show:
	  * @arg0: The #IndicateServer object
	  * @arg1: The type of the server
	  *
	  * Emitted when a server comes onto DBus by being shown.  This
	  * is typically when listeners start reacting to the application's
	  * indicators.  This results in a signal on DBus.
 	  */
	signals[SERVER_SHOW] = g_signal_new(INDICATE_SERVER_SIGNAL_SERVER_SHOW,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, server_show),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__STRING,
	                                        G_TYPE_NONE, 1, G_TYPE_STRING);
	/**
	 * IndicateServer::server-hide:
	 * @arg0: The #IndicateServer object
	 * @arg1: The type of the server
	 *
	 * Emitted when a server removes itself from DBus.  This results
	 * in a signal on DBus.
	 */
	signals[SERVER_HIDE] = g_signal_new(INDICATE_SERVER_SIGNAL_SERVER_HIDE,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, server_hide),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__STRING,
	                                        G_TYPE_NONE, 1, G_TYPE_STRING);
	/**
	 * IndicateServer::server-display:
	 * @arg0: The #IndicateServer object
	 * @arg1: Timestamp of the show event
	 *
	 * Emitted when a listener signals that the server itself should be
	 * displayed.  This signal is caused by a user clicking on the application
	 * item in the Messaging Menu.  This signal is emitted by DBus.
	 */
	signals[SERVER_DISPLAY] = g_signal_new(INDICATE_SERVER_SIGNAL_SERVER_DISPLAY,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, server_display),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__UINT,
	                                        G_TYPE_NONE, 1, G_TYPE_UINT);
	/**
	 * IndicateServer::interest-added:
	 * @arg0: The #IndicateServer object
	 * @arg1: The interest that was added from #IndicateInterests
	 *
	 * Emitted when a listener signals that they are interested in
	 * this server for a particular reason.  This signal is emitted by DBus.
	 */
	signals[INTEREST_ADDED] = g_signal_new(INDICATE_SERVER_SIGNAL_INTEREST_ADDED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, interest_added),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__UINT,
	                                        G_TYPE_NONE, 1, G_TYPE_UINT);
	/**
	 * IndicateServer::interest-removed:
	 * @arg0: The #IndicateServer object
	 * @arg1: The interest that was removed from #IndicateInterests
	 *
	 * Emitted when a listener signals that they are no longer interested in
	 * this server for a particular reason.  This signal is emitted by DBus.
	 *
	 * @note This signal is also emitted after a timeout when the object
	 * is created with @arg1 set to #INDICATOR_INTREST_NONE if no one has
	 * shown any interest in the server.
	 */
	signals[INTEREST_REMOVED] = g_signal_new(INDICATE_SERVER_SIGNAL_INTEREST_REMOVED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, interest_removed),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__UINT,
	                                        G_TYPE_NONE, 1, G_TYPE_UINT);
	/**
	 * IndicateServer::max-indicators-changed:
	 * @arg0: The #IndicateServer object
	 * @arg1: The new max number of indicators
	 *
	 * Emitted when a listener either specifies their max number
	 * to be higher, or at all.  The default is -1 or infinite.
	 */
	signals[MAX_INDICATORS_CHANGED] = g_signal_new(INDICATE_SERVER_SIGNAL_MAX_INDICATORS_CHANGED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, max_indicators_changed),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__INT,
	                                        G_TYPE_NONE, 1, G_TYPE_INT);
	/**
	 * IndicateServer::server-count-changed:
	 * @arg0: The #IndicateServer object
	 * @arg1: The count variable on the server changed.
	 *
	 * Emitted when the count property of the server changes
	 * to a new value.
	 */
	signals[SERVER_COUNT_CHANGED] = g_signal_new(INDICATE_SERVER_SIGNAL_SERVER_COUNT_CHANGED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateServerClass, server_count_changed),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__UINT,
	                                        G_TYPE_NONE, 1, G_TYPE_UINT);

	g_object_class_install_property (gobj, PROP_DESKTOP,
	                                 g_param_spec_string("desktop", "Desktop File",
	                                              "The desktop file representing this server",
	                                              "",
	                                              G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
	g_object_class_install_property (gobj, PROP_TYPE,
	                                 g_param_spec_string("type", "Server Type",
	                                              "The type of indicators that this server will provide",
	                                              "",
	                                              G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
	g_object_class_install_property (gobj, PROP_COUNT,
	                                 g_param_spec_uint("count", "Server Count",
	                                              "A number reprsenting the number of items in a server",
												  0, G_MAXUINT, 0,
	                                              G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
	g_object_class_install_property (gobj, PROP_MENU,
	                                 g_param_spec_string("menu", "DBus Menu Object Path",
	                                              "The DBus Object path to an object with a dbusmenu interface on it.",
												  "",
	                                              G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));
	g_object_class_install_property (gobj, PROP_PATH,
					 g_param_spec_string("path", "DBus Path for server", "DBus path for the server object", "/com/canonical/indicate",
							     G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_STRINGS));
	g_object_class_install_property (gobj, PROP_ICON_THEME,
					 g_param_spec_string("icon-theme", "Icon Theme Name",
							     "The Custom Icon Theme Name to use when displaying this Server.",
							     "",
							     G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

	class->get_indicator_count = get_indicator_count;
	class->get_indicator_list = get_indicator_list;
	class->get_indicator_property = get_indicator_property;
	class->get_indicator_property_group = get_indicator_property_group;
	class->get_indicator_properties = get_indicator_properties;
	class->show_indicator_to_user = show_indicator_to_user;
	class->indicator_displayed = indicator_displayed;
	class->get_next_id = get_next_id;
	class->show_interest = show_interest;
	class->remove_interest = remove_interest;
	class->check_interest = check_interest;
	class->max_indicators_get = max_indicators_get;
	class->max_indicators_set = max_indicators_set;

	/* DBus interfaces */
	if (bus_node_info == NULL) {
		GError * error = NULL;

		bus_node_info = g_dbus_node_info_new_for_xml(_indicate_interface, &error);
		if (error != NULL) {
			g_error("Unable to parse Indicate Interface description: %s", error->message);
			g_error_free(error);
		}
	}

	if (bus_interface_info == NULL) {
		bus_interface_info = g_dbus_node_info_lookup_interface(bus_node_info, INDICATE_DBUS_IFACE);

		if (bus_interface_info == NULL) {
			g_error("Unable to find interface '" INDICATE_DBUS_IFACE "'");
		}
	}

	/* Building our Method table :( */
	bus_method_table[METHOD_GET_INDICATOR_COUNT].interned_name          = g_intern_static_string("GetIndicatorCount");
	bus_method_table[METHOD_GET_INDICATOR_COUNT].func                   = bus_get_indicator_count;

	bus_method_table[METHOD_GET_INDICATOR_LIST].interned_name           = g_intern_static_string("GetIndicatorList");
	bus_method_table[METHOD_GET_INDICATOR_LIST].func                    = bus_get_indicator_list;

	bus_method_table[METHOD_GET_INDICATOR_PROPERTY].interned_name       = g_intern_static_string("GetIndicatorProperty");
	bus_method_table[METHOD_GET_INDICATOR_PROPERTY].func                = bus_get_indicator_property;

	bus_method_table[METHOD_GET_INDICATOR_PROPERTY_GROUP].interned_name = g_intern_static_string("GetIndicatorPropertyGroup");
	bus_method_table[METHOD_GET_INDICATOR_PROPERTY_GROUP].func          = bus_get_indicator_property_group;

	bus_method_table[METHOD_GET_INDICATOR_PROPERTIES].interned_name     = g_intern_static_string("GetIndicatorProperties");
	bus_method_table[METHOD_GET_INDICATOR_PROPERTIES].func              = bus_get_indicator_properties;

	bus_method_table[METHOD_SHOW_INDICATOR_TO_USER].interned_name       = g_intern_static_string("ShowIndicatorToUser");
	bus_method_table[METHOD_SHOW_INDICATOR_TO_USER].func                = bus_show_indicator_to_user;

	bus_method_table[METHOD_INDICATOR_DISPLAYED].interned_name          = g_intern_static_string("IndicatorDisplayed");
	bus_method_table[METHOD_INDICATOR_DISPLAYED].func                   = bus_indicator_displayed;

	bus_method_table[METHOD_SHOW_INTEREST].interned_name                = g_intern_static_string("ShowInterest");
	bus_method_table[METHOD_SHOW_INTEREST].func                         = bus_show_interest;

	bus_method_table[METHOD_REMOVE_INTEREST].interned_name              = g_intern_static_string("RemoveInterest");
	bus_method_table[METHOD_REMOVE_INTEREST].func                       = bus_remove_interest;

	bus_method_table[METHOD_SET_MAX_INDICATORS].interned_name           = g_intern_static_string("SetMaxIndicators");
	bus_method_table[METHOD_SET_MAX_INDICATORS].func                    = bus_set_max_indicators;


	return;
}

static void
indicate_server_init (IndicateServer * server)
{
	/* g_debug("Server Object Initialized"); */

	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	priv->path = NULL;
	priv->indicators = NULL;
	priv->num_hidden = 0;
	priv->visible = FALSE;
	priv->registered = 0;
	priv->current_id = 0;
	priv->type = NULL;
	priv->desktop = NULL;
	priv->count = 0;
	priv->dbusmenu = NULL;

	guint i;
	for (i = INDICATE_INTEREST_NONE; i < INDICATE_INTEREST_LAST; i++) {
		priv->interests[i] = FALSE;
	}
	priv->interestedfolks = NULL;
	priv->max_indicators = MAX_INDICATORS_UNSET;

	priv->broadcast_signal = 0;
	priv->connection = NULL;
	priv->connection_cancel = g_cancellable_new();
	g_bus_get(G_BUS_TYPE_SESSION,
	          priv->connection_cancel, /* cancel */
	          bus_connection_cb,
	          server);

	priv->interest_timer = g_timeout_add(500, interest_timer, server);

	return;
}

static void
indicate_server_dispose (GObject * obj)
{
	IndicateServer * server = INDICATE_SERVER(obj);
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	if (priv->broadcast_signal != 0) {
		g_dbus_connection_signal_unsubscribe(priv->connection, priv->broadcast_signal);
		priv->broadcast_signal = 0;
	}

	if (priv->connection_cancel != NULL) {
		g_cancellable_cancel(priv->connection_cancel);
		g_object_unref(priv->connection_cancel);
		priv->connection_cancel = NULL;
	}

	if (priv->dbusmenu != NULL) {
		g_object_unref(priv->dbusmenu);
		priv->dbusmenu = NULL;
	}

	if (priv->visible) {
		if (priv->registered != 0) {
		g_dbus_connection_emit_signal(priv->connection,
		                              NULL, /* dest */
		                              priv->path,
		                              INDICATE_DBUS_IFACE,
		                              "ServerHide",
		                              g_variant_new("(s)", priv->type ? priv-> type : ""),
		                              NULL); /* error */
		}
		g_signal_emit(server, signals[SERVER_HIDE], 0, priv->type ? priv->type : "", TRUE);
	}

	if (priv->registered != 0) {
		g_dbus_connection_unregister_object(priv->connection, priv->registered);
	}

	if (priv->connection != NULL) {
		g_object_unref(priv->connection);
		priv->connection = NULL;
	}

	if (priv->interest_timer != 0) {
		g_source_remove(priv->interest_timer);
		priv->interest_timer = 0;
	}

	return;
}

static void
indicate_server_finalize (GObject * obj)
{
	IndicateServer * server = INDICATE_SERVER(obj);
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	if (priv->path) {
		g_free(priv->path);
	}
	if (priv->desktop) {
		g_free(priv->desktop);
	}
	if (priv->type) {
		g_free(priv->type);
	}

	G_OBJECT_CLASS (indicate_server_parent_class)->finalize (obj);

	return;
}

static void
set_property (GObject * obj, guint id, const GValue * value, GParamSpec * pspec)
{
	g_return_if_fail(G_VALUE_HOLDS_STRING(value) || G_VALUE_HOLDS_UINT(value));

	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(obj);
	switch (id) {
	case PROP_DESKTOP:
		if (priv->desktop != NULL) {
			g_free(priv->desktop);
		}
		priv->desktop = g_value_dup_string(value);
		break;
	case PROP_TYPE:
		if (priv->type != NULL) {
			g_free(priv->type);
		}
		priv->type = g_value_dup_string(value);
		break;
	case PROP_COUNT: {
		guint newval = g_value_get_uint(value);
		if (newval != priv->count) {
			priv->count = newval;
			if (priv->registered != 0) {
				g_dbus_connection_emit_signal(priv->connection,
				                              NULL, /* dest */
				                              priv->path,
				                              INDICATE_DBUS_IFACE,
				                              "ServerCountChanged",
				                              g_variant_new("(u)", newval),
				                              NULL); /* error */
			}
			g_signal_emit(obj, signals[SERVER_COUNT_CHANGED], 0, newval, TRUE);
		}
		break;
	}
	case PROP_PATH:
	        if (priv->path != NULL) {
	                g_free(priv->path);
	        }
	        priv->path = g_value_dup_string(value);
	        break;
	case PROP_ICON_THEME:
		if (priv->icon_theme != NULL) {
			g_free (priv->icon_theme);
		}
		priv->icon_theme = g_value_dup_string(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, id, pspec);
		break;
	}

	return;
}

/* Gets the Gobject properties for the IndicateServer.  Mostly
   just copies strings and a whole uint! */
static void
get_property (GObject * obj, guint id, GValue * value, GParamSpec * pspec)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(obj);
	switch (id) {
	case PROP_DESKTOP:
		if (priv->desktop == NULL) {
			g_value_set_string(value, "");
		} else {
			g_value_set_string(value, priv->desktop);
		}
		break;
	case PROP_TYPE:
		if (priv->type == NULL) {
			g_value_set_string(value, "");
		} else {
			g_value_set_string(value, priv->type);
		}
		break;
	case PROP_COUNT:
		g_value_set_uint(value, priv->count);
		break;
	case PROP_MENU:
		if (priv->dbusmenu != NULL) {
			GValue strvalue = {0};
			g_value_init(&strvalue, G_TYPE_STRING);
			g_object_get_property(G_OBJECT(priv->dbusmenu), DBUSMENU_SERVER_PROP_DBUS_OBJECT, &strvalue);
			if (g_value_get_string(&strvalue) != NULL) {
				g_value_set_boxed(value, g_value_dup_string(&strvalue));
			} else {
				g_value_set_boxed(value, g_strdup("/"));
			}
			g_value_unset(&strvalue);
		} else {
			g_value_set_boxed(value, g_strdup("/"));
		}
		break;
	case PROP_PATH:
		g_value_set_string(value, priv->path);
		break;
	case PROP_ICON_THEME:
		if (priv->icon_theme == NULL) {
			g_value_set_string(value, "");
		} else {
			g_value_set_string(value, priv->icon_theme);
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, id, pspec);
		break;
	}

	return;
}

/************************
  DBUS STUFF
 ************************/

/* Response to trying to get on the session bus */
static void
bus_connection_cb (GObject * obj, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;

	GDBusConnection * connection = g_bus_get_finish(res, &error);
	if (error != NULL) {
		g_error("Unable to get session bus: %s", error->message);
		g_error_free(error);
		return;
	}

	IndicateServer * server = INDICATE_SERVER(user_data);
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	if (priv->connection_cancel != NULL) {
		g_object_unref(priv->connection_cancel);
		priv->connection_cancel = NULL;
	}

	if (priv->connection != NULL) {
		g_warning("Getting a second connection?");
		g_object_unref(priv->connection);
		priv->connection = NULL;
	}
	priv->connection = connection;

	priv->broadcast_signal = g_dbus_connection_signal_subscribe(priv->connection,
	                                                            NULL, /* sender */
	                                                            INDICATE_LISTENER_DBUS_IFACE,
	                                                            "IndicatorServersReport",
	                                                            NULL, /* object */
	                                                            NULL, /* arg0 */
	                                                            G_DBUS_SIGNAL_FLAGS_NONE,
	                                                            bus_broadcast_cb,
	                                                            server,
	                                                            NULL); /* destroy notify */

	if (priv->visible) {
		priv->visible = FALSE;
		indicate_server_show(server);
	}

	return;
}

/* A small dbus filter that waits on the IndicatorServersReport
   signal and sends the Show signal if this server is not hidden. */
static void
bus_broadcast_cb (GDBusConnection * connection, const gchar * sender, const gchar * object_path, const gchar * interface_name, const gchar * signal_name, GVariant * parameters, gpointer user_data)
{
	g_return_if_fail(g_strcmp0(signal_name, "IndicatorServersReport") == 0);

	IndicateServer * server = INDICATE_SERVER(user_data);

	if (server != NULL) {
		IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);
		if (priv->visible) {
			if (priv->registered != 0) {
				g_dbus_connection_emit_signal(priv->connection,
				                              NULL, /* dest */
				                              priv->path,
				                              INDICATE_DBUS_IFACE,
				                              "ServerShow",
				                              g_variant_new("(s)", priv->type ? priv-> type : ""),
				                              NULL); /* error */
			}
			g_signal_emit(server, signals[SERVER_SHOW], 0, priv->type ? priv->type : "", TRUE);
		}
	}

	return;
}

/* A method call has come from DBus */
static void
bus_method_call (GDBusConnection * connection, const gchar * sender, const gchar * path, const gchar * interface, const gchar * method, GVariant * params, GDBusMethodInvocation * invocation, gpointer user_data)
{
	int i;
	const gchar * interned_method = g_intern_string(method);

	for (i = 0; i < METHOD_COUNT; i++) {
		if (bus_method_table[i].interned_name == interned_method) {
			if (bus_method_table[i].func != NULL) {
				return bus_method_table[i].func(INDICATE_SERVER(user_data), params, invocation);
			} else {
				/* If we have a null function we're responding but nothing else. */
				g_warning("Invalid function call for '%s' with parameters: %s", method, g_variant_print(params, TRUE));
				g_dbus_method_invocation_return_value(invocation, NULL);
				return;
			}
		}
	}

	/* We're here because there's an error */
	g_dbus_method_invocation_return_error(invocation,
	                                      indicate_server_error_quark(),
	                                      NOT_IMPLEMENTED,
	                                      "Unable to find method '%s'",
	                                      method);
	return;
}

/* Dbus have asked for properties, let's talk to it. */
static GVariant *
bus_get_prop (GDBusConnection * connection, const gchar * sender, const gchar * path, const gchar * interface, const gchar * property, GError ** error, gpointer user_data)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(user_data);
	GVariant * retvariant = NULL;
	
	if (g_strcmp0(property, "desktop") == 0) {
		if (priv->desktop != NULL) {
			retvariant = g_variant_new_string(priv->desktop);
		} else {
			retvariant = g_variant_new_string("");
		}
	} else if (g_strcmp0(property, "type") == 0) {
		if (priv->type != NULL) {
			retvariant = g_variant_new_string(priv->type);
		} else {
			retvariant = g_variant_new_string("");
		}
	} else if (g_strcmp0(property, "count") == 0) {
		retvariant = g_variant_new_uint32(priv->count);
	} else if (g_strcmp0(property, "menu") == 0) {
		if (priv->dbusmenu != NULL) {
			GValue strvalue = {0};
			g_value_init(&strvalue, G_TYPE_STRING);
			g_object_get_property(G_OBJECT(priv->dbusmenu), DBUSMENU_SERVER_PROP_DBUS_OBJECT, &strvalue);
			if (g_value_get_string(&strvalue) != NULL) {
				retvariant = g_variant_new_string(g_value_get_string(&strvalue));
			} else {
				retvariant = g_variant_new_string("/");
			}
			g_value_unset(&strvalue);
		} else {
			retvariant = g_variant_new_string("/");
		}

	} else if (g_strcmp0(property, "icontheme") == 0) {
		if (priv->icon_theme != NULL) {
			retvariant = g_variant_new_string (priv->icon_theme);
		} else {
			retvariant = g_variant_new_string ("");
		}
	} else {
		g_warning("Unknown property");
	}

	return retvariant;
}

/* Small little function to get an error quark for usage
   with the GError errors back across DBus */
static GQuark
indicate_server_error_quark (void)
{
	static GQuark quark = 0;
	if (quark == 0) {
		quark = g_quark_from_static_string (G_LOG_DOMAIN);

		/* Register our dbus error codes as well */
		g_dbus_error_register_error(quark, NO_GET_DESKTOP, "com.canonical.indicate.NO_GET_DESKTOP");
		g_dbus_error_register_error(quark, NO_GET_INDICATOR_COUNT, "com.canonical.indicate.NO_GET_INDICATOR_COUNT");
		g_dbus_error_register_error(quark, NO_GET_INDICATOR_LIST, "com.canonical.indicate.NO_GET_INDICATOR_LIST");
		g_dbus_error_register_error(quark, NO_GET_INDICATOR_PROPERTY, "com.canonical.indicate.NO_GET_INDICATOR_PROPERTY");
		g_dbus_error_register_error(quark, NO_GET_INDICATOR_PROPERTY_GROUP, "com.canonical.indicate.NO_GET_INDICATOR_PROPERTY_GROUP");
		g_dbus_error_register_error(quark, NO_GET_INDICATOR_PROPERTIES, "com.canonical.indicate.NO_GET_INDICATOR_PROPERTIES");
		g_dbus_error_register_error(quark, NO_SHOW_INDICATOR_TO_USER, "com.canonical.indicate.NO_SHOW_INDICATOR_TO_USER");
		g_dbus_error_register_error(quark, NO_INDICATOR_DISPLAYED, "com.canonical.indicate.NO_INDICATOR_DISPLAYED");
		g_dbus_error_register_error(quark, INVALID_INDICATOR_ID, "com.canonical.indicate.INVALID_INDICATOR_ID");
		g_dbus_error_register_error(quark, NO_SHOW_INTEREST, "com.canonical.indicate.NO_SHOW_INTEREST");
		g_dbus_error_register_error(quark, NO_REMOVE_INTEREST, "com.canonical.indicate.NO_REMOVE_INTEREST");
		g_dbus_error_register_error(quark, SHOW_INTEREST_FAILED, "com.canonical.indicate.SHOW_INTEREST_FAILED");
		g_dbus_error_register_error(quark, REMOVE_INTEREST_FAILED, "com.canonical.indicate.REMOVE_INTEREST_FAILED");
		g_dbus_error_register_error(quark, NO_MAX_INDICATORS_SET, "com.canonical.indicate.NO_MAX_INDICATORS_SET");
		g_dbus_error_register_error(quark, MAX_INDICATORS_SET_FAILED, "com.canonical.indicate.MAX_INDICATORS_SET_FAILED");
		g_dbus_error_register_error(quark, NO_SUCH_PROPERTY, "com.canonical.indicate.NO_SUCH_PROPERTY");
		g_dbus_error_register_error(quark, NOT_IMPLEMENTED, "com.canonical.indicate.NOT_IMPLEMENTED");
	}
	return quark;
}

/**
 * indicate_server_show:
 * @server: The #IndicateServer to be shown
 * 
 * This function exports the object onto DBus and shows it
 * to the world.  This will be the start of it receiving external
 * signals from DBus.  It is likely that, if there are listeners
 * running, there will several #IndicateServer::interest-added
 * signals coming shortly after this function.  This function
 * emits the #IndicateServer::server-added signal across the bus.
*/
void
indicate_server_show (IndicateServer * server)
{
	g_return_if_fail(INDICATE_IS_SERVER(server));
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	if (priv->visible) {
		return;
	}

	priv->visible = TRUE;

	if (priv->connection == NULL) {
		return;
	}

	if (priv->registered == 0) {
		GError * error = NULL;
		priv->registered = g_dbus_connection_register_object(priv->connection,
		                                                     priv->path,
		                                                     bus_interface_info,
		                                                     &bus_interface_table,
		                                                     server,
		                                                     NULL,
		                                                     &error);

		if (error != NULL) {
			g_warning("Unable to export object '%s' with interface '" INDICATE_DBUS_IFACE "' on dbus: %s", priv->path, error->message);
			g_error_free(error);
			priv->registered = 0; /* Just to be sure */
			return;
		}
	}

	if (priv->registered != 0) {
		g_dbus_connection_emit_signal(priv->connection,
		                              NULL, /* dest */
		                              priv->path,
		                              INDICATE_DBUS_IFACE,
		                              "ServerShow",
		                              g_variant_new("(s)", priv->type ? priv-> type : ""),
		                              NULL); /* error */
	}
	g_signal_emit(server, signals[SERVER_SHOW], 0, priv->type ? priv->type : "", TRUE);

	return;
}

/**
 * indicate_server_hide:
 * @server: The #IndicateServer to hide.
 * 
 * This function hides the server from DBus so that it does
 * not get signals anymore.  This causes the signal #IndicateServer::server-hide
 * to be sent across the bus for all listeners.  Also internally
 * it will signal #IndicateServer::interest-removed for all the
 * interests that were currently set for this server.
*/
void
indicate_server_hide (IndicateServer * server)
{
	g_return_if_fail(INDICATE_IS_SERVER(server));
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	if (!priv->visible)
		return;

	priv->visible = FALSE;

	/* Delete interested parties */
	g_list_foreach(priv->interestedfolks, (GFunc)indicate_server_interested_folks_destroy, NULL);
	g_list_free(priv->interestedfolks);
	priv->interestedfolks = NULL;

	/* Signal the lack of interest */
	guint i;
	for (i = INDICATE_INTEREST_NONE; i < INDICATE_INTEREST_LAST; i++) {
		if (priv->interests[i]) {
			g_signal_emit(G_OBJECT(server), signals[INTEREST_REMOVED], 0, i, TRUE);
		}
		priv->interests[i] = FALSE;
	}

	/* Signal that we don't have a max */
	priv->max_indicators = MAX_INDICATORS_UNSET;
	g_signal_emit(G_OBJECT(server), signals[MAX_INDICATORS_CHANGED], 0, MAX_INDICATORS_INFINITE, TRUE);

	if (priv->registered != 0) {
		g_dbus_connection_emit_signal(priv->connection,
		                              NULL, /* dest */
		                              priv->path,
		                              INDICATE_DBUS_IFACE,
		                              "ServerHide",
		                              g_variant_new("(s)", priv->type ? priv-> type : ""),
		                              NULL); /* error */
	}
	g_signal_emit(server, signals[SERVER_HIDE], 0, priv->type ? priv->type : "", TRUE);

	return;
}

/* Checks each value in the hash table to see if it's
   been set.  If it is, then we need to recalculate
   the value of displayed. */
static void
indicator_display_check_recalc (gpointer key, gpointer value, gpointer userdata)
{
	if (!GPOINTER_TO_UINT(value)) {
		return;
	}

	recalculate_indicator_displayed(INDICATE_SERVER(userdata), GPOINTER_TO_UINT(key));
	return;
}

/* Look to see if a specific indicator ID has anyone watching
   for it, and if not we need to set it to undisplayed. */
static void
recalculate_indicator_displayed (IndicateServer * server, guint id)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);
	GList * folkpointer;

	for (folkpointer = priv->interestedfolks; folkpointer != NULL; folkpointer = g_list_next(folkpointer)) {
		IndicateServerInterestedFolk * folk = (IndicateServerInterestedFolk *)folkpointer->data;
		if (g_hash_table_lookup(folk->indicators_displayed, GUINT_TO_POINTER(id))) {
			break;
		}
	}

	if (folkpointer == NULL) {
		/* We went through the list and no one cares about
		   this indicator.  It's not displayed anymore. */
		IndicateIndicator * indicator = get_indicator(server, id, NULL);
		if (indicator != NULL) {
			indicate_indicator_set_displayed(indicator, FALSE);
		} else {
			g_warning("I'm removing a displayed set from an indicator that doesn't seem to exist.");
		}
	}

	return;
}

/* Get the next ID from the server */
static guint
get_next_id (IndicateServer * server)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);
	priv->current_id++;
	return priv->current_id;
}

/* Allows a listener to show interest.  First we look to see
   if we've talked to this listener before and if so we just
   mark their folk listing.  If it changes the overall setting
   then we need to signal. */
static gboolean
show_interest (IndicateServer * server, const gchar * sender, IndicateInterests interest)
{
	if (!(interest > INDICATE_INTEREST_NONE && interest < INDICATE_INTEREST_LAST)) {
		return FALSE;
	}

	/* g_debug("Someone is showing interest.  %s in %d", sender, interest); */
	IndicateServerInterestedFolk localfolk;
	localfolk.sender = (gchar *)sender; /* Okay to drop the const as we're only using this for searching */

	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	if (priv->interest_timer != 0) {
		g_source_remove(priv->interest_timer);
		priv->interest_timer = 0;
	}

	GList * entry = g_list_find_custom(priv->interestedfolks, &localfolk, indicate_server_interested_folks_equal);
	IndicateServerInterestedFolk * folkpointer = NULL;
	if (entry == NULL) {
		folkpointer = g_new0(IndicateServerInterestedFolk, 1);
		indicate_server_interested_folks_init(folkpointer, sender, INDICATE_LISTENER_OBJ_PATH, priv->connection);
		priv->interestedfolks = g_list_append(priv->interestedfolks, folkpointer);
	} else {
		folkpointer = (IndicateServerInterestedFolk *)entry->data;
	}

	indicate_server_interested_folks_set(folkpointer, interest, TRUE);
	if (!priv->interests[interest]) {
		g_signal_emit(G_OBJECT(server), signals[INTEREST_ADDED], 0, interest, TRUE);
		priv->interests[interest] = TRUE;
	}

	return TRUE;
}

/* Removes an interest from a folk.  It removes it from the
   folk structure and then checks to see if that has global
   effect, and handles it.  */
static gboolean
remove_interest (IndicateServer * server, const gchar * sender, IndicateInterests interest)
{
	if (!(interest > INDICATE_INTEREST_NONE && interest < INDICATE_INTEREST_LAST)) {
		return FALSE;
	}

	IndicateServerInterestedFolk localfolk;
	localfolk.sender = (gchar *)sender; /* Okay to drop the const as we're only using this for searching */

	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	/* NOTE: We're not removing the timer here, because we know the timer
	   started in a state where there was no interest.  So if someone has
	   since shown interest, there'll be no timer, and if they haven't then
	   this function won't send a signal anyway. */

	/* Figure out the folk that we're talking to.  If we
	   have an entry for them, use it, otherwise we need
	   to create one. */
	GList * entry = g_list_find_custom(priv->interestedfolks, &localfolk, indicate_server_interested_folks_equal);
	IndicateServerInterestedFolk * folkpointer = NULL;
	if (entry == NULL) {
		folkpointer = g_new0(IndicateServerInterestedFolk, 1);
		indicate_server_interested_folks_init(folkpointer, sender, INDICATE_LISTENER_OBJ_PATH, priv->connection);
		priv->interestedfolks = g_list_append(priv->interestedfolks, folkpointer);
	} else {
		folkpointer = (IndicateServerInterestedFolk *)entry->data;
	}

	/* Set the interest for this guy */
	indicate_server_interested_folks_set(folkpointer, interest, FALSE);

	/* Check to see if the interest has changed as a result of
	   this an announce it to the world */
	if (priv->interests[interest]) {
		guint i;
		for (i = INDICATE_INTEREST_NONE; i < INDICATE_INTEREST_LAST; i++) {
			priv->interests[i] = FALSE;
		}

		GList * listi = NULL;
		for (listi = priv->interestedfolks ; listi != NULL ; listi = listi->next) {
			folkpointer = (IndicateServerInterestedFolk *)listi->data;
			indicate_server_interested_folks_copy(folkpointer, priv->interests);
		}

		if (!priv->interests[interest]) {
			g_signal_emit(G_OBJECT(server), signals[INTEREST_REMOVED], 0, interest, TRUE);
		}
	}

	return TRUE;
}

/* This little timer fires if no one shows any interest at startup
   and signals a removed of NONE saying that there is no interest. */
static gboolean
interest_timer (gpointer user_data)
{
	g_return_val_if_fail(INDICATE_IS_SERVER(user_data), FALSE);
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(user_data);

	g_signal_emit(G_OBJECT(user_data), signals[INTEREST_REMOVED], 0, INDICATE_INTEREST_NONE, TRUE);
	priv->interest_timer = 0;

	return FALSE;
}

/* Checks to see if a particular interest value is
   set.  Uses the interest cache. */
static gboolean
check_interest (IndicateServer * server, IndicateInterests interest)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);
	return priv->interests[interest];
}

/* Internal function that can be subclassed to get
   the value of the max number of indicators. */
static gint
max_indicators_get (IndicateServer * server)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);
	if (priv->max_indicators == MAX_INDICATORS_UNSET) {
		return MAX_INDICATORS_INFINITE;
	} else {
		return priv->max_indicators;
	}
}

/* Internal function to set the number of max indicators
   from a particular listener. */
static gboolean
max_indicators_set (IndicateServer * server, const gchar * sender, gint max)
{
	g_return_val_if_fail(max >= MAX_INDICATORS_INFINITE, FALSE);

	IndicateServerInterestedFolk localfolk;
	localfolk.sender = (gchar *)sender; /* Okay to drop the const as we're only using this for searching */

	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	/* Figure out the folk that we're talking to.  If we
	   have an entry for them, use it, otherwise we need
	   to create one. */
	GList * entry = g_list_find_custom(priv->interestedfolks, &localfolk, indicate_server_interested_folks_equal);
	IndicateServerInterestedFolk * folkpointer = NULL;
	if (entry == NULL) {
		folkpointer = g_new0(IndicateServerInterestedFolk, 1);
		indicate_server_interested_folks_init(folkpointer, sender, INDICATE_LISTENER_OBJ_PATH, priv->connection);
		priv->interestedfolks = g_list_append(priv->interestedfolks, folkpointer);
	} else {
		folkpointer = (IndicateServerInterestedFolk *)entry->data;
	}

	/* If there's a change going on... */
	if (max != folkpointer->max_indicators) {
		gboolean recalculate = FALSE;
		/* If this guy is setting it to infinite or is
                   increasing the number */
		if ((max == -1 && priv->max_indicators != -1) || max > priv->max_indicators) {
			recalculate = TRUE;
		}

		/* Or if potentially we were the ones setting the
		   value for everyone */
		if (priv->max_indicators == folkpointer->max_indicators) {
			recalculate = TRUE;
		}

		folkpointer->max_indicators = max;
		if (recalculate) {
			recalculate_max_indicators(server);
		}
	}

	return TRUE;
}

/* Look through all the folks and trying to figure out
   what the max should be.  Signal if it changes. */
static void
recalculate_max_indicators (IndicateServer * server)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	/* Let's talk to all the folks and see what they
	   think the max should be. */
	GList * folkitem;
	gint newmax = MAX_INDICATORS_UNSET;
	for (folkitem = priv->interestedfolks; folkitem != NULL; folkitem = g_list_next(folkitem)) {
		IndicateServerInterestedFolk * thisfolk = (IndicateServerInterestedFolk *)folkitem->data;
		if (thisfolk->max_indicators == MAX_INDICATORS_UNSET) {
			continue;
		}
		if (thisfolk->max_indicators == MAX_INDICATORS_INFINITE) {
			newmax = MAX_INDICATORS_INFINITE;
			break;
		}
		if (thisfolk->max_indicators > newmax) {
			newmax = thisfolk->max_indicators;
		}
	}

	/* Okay, what ever happened, now it's changing
	   things and we need to tell everyone. */
	if (priv->max_indicators != newmax) {
		priv->max_indicators = newmax;
		g_signal_emit(G_OBJECT(server), signals[MAX_INDICATORS_CHANGED], 0, newmax, TRUE);
	}

	return;
}

static void
indicator_show_cb (IndicateIndicator * indicator, IndicateServer * server)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);
	priv->num_hidden--;
	if (priv->registered != 0) {
		g_dbus_connection_emit_signal(priv->connection,
		                              NULL, /* dest */
		                              priv->path,
		                              INDICATE_DBUS_IFACE,
		                              "IndicatorNew",
		                              g_variant_new("(u)", indicate_indicator_get_id(indicator)),
		                              NULL); /* error */
	}
	g_signal_emit(server, signals[INDICATOR_ADDED], 0, indicate_indicator_get_id(indicator), TRUE);
	return;
}

static void
indicator_hide_cb (IndicateIndicator * indicator, IndicateServer * server)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);
	priv->num_hidden++;
	if (priv->registered != 0) {
		g_dbus_connection_emit_signal(priv->connection,
		                              NULL, /* dest */
		                              priv->path,
		                              INDICATE_DBUS_IFACE,
		                              "IndicatorDelete",
		                              g_variant_new("(u)", indicate_indicator_get_id(indicator)),
		                              NULL); /* error */
	}
	g_signal_emit(server, signals[INDICATOR_REMOVED], 0, indicate_indicator_get_id(indicator), TRUE);
	return;
}

static void
indicator_modified_cb (IndicateIndicator * indicator, gchar * property, IndicateServer * server)
{
	/* g_debug("Indicator Modified: %d %s", indicate_indicator_get_id(indicator), property); */
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);
	if (priv->registered != 0) {
		g_dbus_connection_emit_signal(priv->connection,
		                              NULL, /* dest */
		                              priv->path,
		                              INDICATE_DBUS_IFACE,
		                              "IndicatorModified",
		                              g_variant_new("(us)", indicate_indicator_get_id(indicator), property),
		                              NULL); /* error */
	}
	g_signal_emit(server, signals[INDICATOR_MODIFIED], 0, indicate_indicator_get_id(indicator), property, TRUE);
}

/**
 * indicate_server_add_indicator:
 * @server: The #IndicateServer to add the #IndicateIndictor to.
 * @indicator: The #IndicateIndicator to add.
 * 
 * This function adds an indicator @indicator to the list that are
 * watched by the server @server.  This means that signals that are
 * emitted by the indicator will be picked up and passed via DBus onto
 * listeners of the application.
*/
void
indicate_server_add_indicator (IndicateServer * server, IndicateIndicator * indicator)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

    if (g_slist_find (priv->indicators, indicator) != NULL)
            return;

    priv->indicators = g_slist_prepend(priv->indicators, indicator);

	if (!indicate_indicator_is_visible(indicator)) {
		priv->num_hidden++;
	} else {
		if (priv->registered != 0) {
			g_dbus_connection_emit_signal(priv->connection,
			                              NULL, /* dest */
			                              priv->path,
			                              INDICATE_DBUS_IFACE,
			                              "IndicatorNew",
			                              g_variant_new("(u)", indicate_indicator_get_id(indicator)),
			                              NULL); /* error */
		}
		g_signal_emit(server, signals[INDICATOR_ADDED], 0, indicate_indicator_get_id(indicator), TRUE);
	}

	g_signal_connect(indicator, INDICATE_INDICATOR_SIGNAL_SHOW, G_CALLBACK(indicator_show_cb), server);
	g_signal_connect(indicator, INDICATE_INDICATOR_SIGNAL_HIDE, G_CALLBACK(indicator_hide_cb), server);
	g_signal_connect(indicator, INDICATE_INDICATOR_SIGNAL_MODIFIED, G_CALLBACK(indicator_modified_cb), server);

	return;
}

/**
 * indicate_server_remove_indicator:
 * @server: The #IndicateServer to remove the #IndicateIndictor from.
 * @indicator: The #IndicateIndicator to remove.
 * 
 * Removes an indicator @indicator from being watched by the server @server
 * so it's signals are no longer watched and set over DBus.
*/
void
indicate_server_remove_indicator (IndicateServer * server, IndicateIndicator * indicator)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

    if (g_slist_find (priv->indicators, indicator) == NULL)
            return;

	priv->indicators = g_slist_remove(priv->indicators, indicator);
	if (indicate_indicator_is_visible(indicator)) {
		if (priv->registered != 0) {
			g_dbus_connection_emit_signal(priv->connection,
			                              NULL, /* dest */
			                              priv->path,
			                              INDICATE_DBUS_IFACE,
			                              "IndicatorDelete",
			                              g_variant_new("(u)", indicate_indicator_get_id(indicator)),
			                              NULL); /* error */
		}
		g_signal_emit(server, signals[INDICATOR_REMOVED], 0, indicate_indicator_get_id(indicator), TRUE);
	} else {
		priv->num_hidden--;
	}

	g_signal_handlers_disconnect_by_func(indicator, indicator_show_cb, server);
	g_signal_handlers_disconnect_by_func(indicator, indicator_hide_cb, server);
	g_signal_handlers_disconnect_by_func(indicator, indicator_modified_cb, server);

	return;
}

void
indicate_server_set_dbus_object (const gchar * obj)
{
	/* TODO */

	return;
}

/**
 * indicate_server_set_desktop_file:
 * @server: The #IndicateServer to set the type of
 * @path: The new desktop file representing the server
 * 
 * This is a convience function to set the #IndicateServer:desktop
 * property of the @server object.  The property can also be set
 * via traditional means, but this one is easier to read.
*/
void
indicate_server_set_desktop_file (IndicateServer * server, const gchar * path)
{
	GValue value = {0};
	g_value_init(&value, G_TYPE_STRING);
	g_value_set_string(&value, path);
	g_object_set_property(G_OBJECT(server), "desktop", &value);
	return;
}

/**
 * indicate_server_set_type:
 * @server: The #IndicateServer to set the type of
 * @type: The new type of the server
 * 
 * This is a convience function to set the #IndicateServer:type
 * property of the @server object.  The property can also be set
 * via traditional means, but this one is easier to read.
*/
void
indicate_server_set_type (IndicateServer * server, const gchar * type)
{
	GValue value = {0};
	g_value_init(&value, G_TYPE_STRING);
	g_value_set_string(&value, type);
	g_object_set_property(G_OBJECT(server), "type", &value);
	g_value_unset(&value);
	return;
}

/**
 * indicate_server_set_count:
 * @server: The #IndicateServer to set the type of
 * @count: The number of items that the server believes the user
 *         would be interested in.
 * 
 * A small convience function to set the #IndicateServer:count
 * property on the server.  This should represent a number of messages
 * on a particular server.  This should not be used at the same time
 * as individual indicators to show information to the users.  They
 * sound be used independently.
*/
void
indicate_server_set_count (IndicateServer * server, guint count)
{
	GValue value = {0};
	g_value_init(&value, G_TYPE_UINT);
	g_value_set_uint(&value, count);
	g_object_set_property(G_OBJECT(server), "count", &value);
	return;
}

/**
 * indicate_server_set_icon_theme:
 * @server: The #IndicateServer to set the type of
 * @name: The name of an Icon Theme (according to the Icon Naming 
 *        Specification) to request renderers of the server to use. 
 * 
 * This is a convience function to set the #IndicateServer:icon-theme
 * property of the @server object.  The property can also be set
 * via traditional means, but this one is easier to read.
*/
void
indicate_server_set_icon_theme (IndicateServer * server, const gchar * name)
{
	GValue value = {0};
	g_value_init(&value, G_TYPE_STRING);
	g_value_set_string(&value, name);
	g_object_set_property(G_OBJECT(server), "icon-theme", &value);
	return;
}

static IndicateServer * default_indicate_interface_server = NULL;

/**
 * indicate_server_ref_default:
 * 
 * This function will return a reference to the default #IndicateServer
 * reference if there is one, or it will create one if one had not
 * previously been created.  It is recommended that all applications
 * use this function to create a #IndicateServer as it ensure that there
 * is only one per application.
 * 
 * Return value: (transfer full): A reference to the default #IndicateServer instance.
*/
IndicateServer *
indicate_server_ref_default (void)
{
	if (default_indicate_interface_server != NULL) {
		g_object_ref(default_indicate_interface_server);
	} else {
		default_indicate_interface_server = g_object_new(INDICATE_TYPE_SERVER, NULL);
		g_object_add_weak_pointer(G_OBJECT(default_indicate_interface_server),
		                          (gpointer *)&default_indicate_interface_server);
	}

	return default_indicate_interface_server;
}

/**
 * indicate_server_set_default:
 * @server: The #IndicateServer that should be used
 * 
 * This function is used to set the default #IndicateServer that will
 * be used when creating #IndicateIndicators or for anyone else that
 * calls indicate_server_ref_default().  Typically this is just an
 * instance of #IndicateServer but applications that create a subclass
 * of #IndicateServer should set this as well.
*/
void
indicate_server_set_default (IndicateServer * server)
{
	if (default_indicate_interface_server != NULL) {
		g_warning("Setting a default Indicator Server when one has already been created.  I'm not going to destroy that one, but let it live.  This may create some odd results if you don't know what you're doing.");
	}

	if (server != NULL) {
		default_indicate_interface_server = server;
		g_object_add_weak_pointer(G_OBJECT(default_indicate_interface_server),
		                          (gpointer *)&default_indicate_interface_server);
	}

	return;
}

static gboolean
get_indicator_count (IndicateServer * server, guint * count, GError **error)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	guint lstcnt = g_slist_length(priv->indicators);

	g_return_val_if_fail(priv->num_hidden < lstcnt, TRUE);
	
	*count = lstcnt - priv->num_hidden;

	return TRUE;
}

static gboolean
get_indicator_list (IndicateServer * server, GArray ** indicators, GError ** error)
{
	g_return_val_if_fail(INDICATE_IS_SERVER(server), TRUE);

	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);
	g_return_val_if_fail(class->get_indicator_count != NULL, TRUE);

	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	*indicators = g_array_sized_new(FALSE, FALSE, sizeof(guint), g_slist_length(priv->indicators) - priv->num_hidden);

	GSList * iter;
	int i;
	for (iter = priv->indicators, i = 0; iter != NULL; iter = iter->next) {
		IndicateIndicator * indicator = INDICATE_INDICATOR(iter->data);
		if (indicate_indicator_is_visible(indicator)) {
			guint id = indicate_indicator_get_id(indicator);
			g_array_insert_val(*indicators, i++, id);
		}
	}

	return TRUE;
}

static IndicateIndicator *
get_indicator (IndicateServer * server, guint id, GError **error)
{
	g_return_val_if_fail(INDICATE_IS_SERVER(server), NULL);
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	GSList * iter;
	for (iter = priv->indicators; iter != NULL; iter = iter->next) {
		IndicateIndicator * indicator = INDICATE_INDICATOR(iter->data);
		if (indicate_indicator_get_id(indicator) == id) {
			return indicator;
		}
	}

	if (error) {
		g_set_error(error,
		            indicate_server_error_quark(),
		            INVALID_INDICATOR_ID,
		            "Invalid Indicator ID: %d",
		            id);
	}
	return NULL;
}

static gboolean
get_indicator_property (IndicateServer * server, guint id, gchar * property, GVariant ** variant, GError **error)
{
	IndicateIndicator * indicator = get_indicator(server, id, error);
	if (indicator == NULL) {
		return FALSE;
	}

	GVariant * ind_property = indicate_indicator_get_property_variant(indicator, property);
	if (ind_property == NULL) {
		if (error != NULL) {
			g_set_error(error,
			            indicate_server_error_quark(),
			            NO_SUCH_PROPERTY,
			            "Indicator %d has no property named '%s'",
			            id, property);
		}
		return FALSE;
	}

	*variant = ind_property;
	return TRUE;
}

static gboolean
get_indicator_property_group (IndicateServer * server, guint id, const gchar ** properties, GVariant ** output, GError **error)
{
	IndicateIndicator * indicator = get_indicator(server, id, error);
	if (indicator == NULL) {
		return FALSE;
	}

	GVariantBuilder builder;
	g_variant_builder_init(&builder, G_VARIANT_TYPE_DICTIONARY);
	int i;
	for (i = 0; properties[i] != NULL; i++) {
		GVariant * ind_property = indicate_indicator_get_property_variant(indicator, properties[i]);

		if (ind_property == NULL) {
			continue;
		}

		GVariant * dictentry = g_variant_new_dict_entry(g_variant_new_string(properties[i]), ind_property);
		g_variant_builder_add_value(&builder, dictentry);
	}

	*output = g_variant_builder_end(&builder);

	return TRUE;
}

static gboolean
get_indicator_properties (IndicateServer * server, guint id, gchar *** properties, GError **error)
{
	IndicateIndicator * indicator = get_indicator(server, id, error);
	if (indicator == NULL) {
		return FALSE;
	}

	GPtrArray * array = indicate_indicator_list_properties(indicator);
	g_ptr_array_add(array, NULL);

	*properties = (gchar **)g_ptr_array_free(array, FALSE);

	return TRUE;
}

static gboolean
show_indicator_to_user (IndicateServer * server, guint id, guint timestamp, GError ** error)
{
	if (id == INDICATE_SERVER_INDICATOR_NULL) {
		g_signal_emit(server, signals[SERVER_DISPLAY], 0, timestamp, TRUE);
		return TRUE;
	}

	IndicateIndicator * indicator = get_indicator(server, id, error);
	if (indicator == NULL) {
		return FALSE;
	}

	indicate_indicator_user_display(indicator, timestamp);
	return TRUE;
}

/* A function representing when an indicator is shown
   to the user in some form.  First it sees if we've talked
   to this listener before and uses that entry.  Otherwise, we
   build one.  If we're displaying it, things are good, we can
   just set that.  If we're not, then we need to check to see
   if anyone else is before continuing.  */
static gboolean
indicator_displayed (IndicateServer * server, const gchar * sender, guint id, gboolean displayed, GError ** error)
{
	IndicateServerInterestedFolk localfolk;
	localfolk.sender = (gchar *)sender; /* Okay to drop the const as we're only using this for searching */

	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	GList * entry = g_list_find_custom(priv->interestedfolks, &localfolk, indicate_server_interested_folks_equal);
	IndicateServerInterestedFolk * folkpointer = NULL;
	if (entry == NULL) {
		folkpointer = g_new0(IndicateServerInterestedFolk, 1);
		indicate_server_interested_folks_init(folkpointer, sender, INDICATE_LISTENER_OBJ_PATH, priv->connection);
		priv->interestedfolks = g_list_append(priv->interestedfolks, folkpointer);
	} else {
		folkpointer = (IndicateServerInterestedFolk *)entry->data;
	}

	g_hash_table_insert(folkpointer->indicators_displayed, GUINT_TO_POINTER(id), GUINT_TO_POINTER(displayed));

	if (displayed) {
		IndicateIndicator * indicator = get_indicator(server, id, error);
		if (indicator == NULL) {
			return FALSE;
		}

		indicate_indicator_set_displayed(indicator, displayed);
	} else {
		recalculate_indicator_displayed(server, id);
	}

	return TRUE;
}

/* Virtual Functions */
static void
bus_get_indicator_count (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->get_indicator_count != NULL) {
		guint count = 0;
		GError * error = NULL;

		class->get_indicator_count (server, &count, &error);

		if (error != NULL) {
			g_dbus_method_invocation_return_gerror(invocation, error);
			g_error_free(error);
		} else {
			g_dbus_method_invocation_return_value(invocation,
			                                      g_variant_new("(u)", count));
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
		            indicate_server_error_quark(),
		            NO_GET_INDICATOR_COUNT,
		            "get_indicator_count function doesn't exist for this server class: %s",
		            G_OBJECT_TYPE_NAME(server));
	return;
}

static void
bus_get_indicator_list (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->get_indicator_list != NULL) {
		GArray * array = NULL;
		GError * error = NULL;

		class->get_indicator_list (server, &array, &error);

		if (error != NULL) {
			g_dbus_method_invocation_return_gerror(invocation, error);
			g_error_free(error);
		} else {
			GVariant * retval = NULL;
			if (array->len != 0) {
				GVariantBuilder builder;
				g_variant_builder_init(&builder, G_VARIANT_TYPE_ARRAY);

				int i;
				for (i = 0; array != NULL && i < array->len; i++) {
					g_variant_builder_add_value(&builder, g_variant_new_int32(g_array_index(array, gint, i)));
				}

				retval = g_variant_builder_end(&builder);
			} else {
				retval = g_variant_parse(g_variant_type_new("ai"), "[]", NULL, NULL, NULL);
			}
			g_dbus_method_invocation_return_value(invocation, g_variant_new_tuple(&retval, 1));

			g_array_free(array, TRUE);
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
		            indicate_server_error_quark(),
		            NO_GET_INDICATOR_LIST,
		            "get_indicator_list function doesn't exist for this server class: %s",
		            G_OBJECT_TYPE_NAME(server));
	return;
}

static void
bus_get_indicator_property (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->get_indicator_property != NULL) {
		guint id;
		gchar * property;
		GVariant * variant;
		GError * error = NULL;

		g_variant_get(params, "(us)", &id, &property);

		class->get_indicator_property (server, id, property, &variant, &error);

		g_free(property);
		if (error != NULL) {
			g_dbus_method_invocation_return_gerror(invocation, error);
		} else {
			g_dbus_method_invocation_return_value(invocation, g_variant_new("(v)", variant));
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
		            indicate_server_error_quark(),
		            NO_GET_INDICATOR_PROPERTY,
		            "get_indicator_property function doesn't exist for this server class: %s",
		            G_OBJECT_TYPE_NAME(server));
	return;
}

static void
bus_get_indicator_property_group (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->get_indicator_property_group != NULL) {
		guint id = g_variant_get_uint32(g_variant_get_child_value(params, 0));;
		const gchar ** props = g_variant_get_strv(g_variant_get_child_value(params, 1), NULL);
		GVariant * output;
		GError * error = NULL;

		class->get_indicator_property_group (server, id, props, &output, &error);

		if (error != NULL) {
			g_dbus_method_invocation_return_gerror(invocation, error);
			g_error_free(error);
		} else {
			if (output != NULL) {
				output = g_variant_new_tuple(&output, 1);
			}
			g_dbus_method_invocation_return_value(invocation, output);
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
		            indicate_server_error_quark(),
		            NO_GET_INDICATOR_PROPERTY_GROUP,
		            "get_indicator_property_group function doesn't exist for this server class: %s",
		            G_OBJECT_TYPE_NAME(server));
	return;
}

static void
bus_get_indicator_properties (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->get_indicator_properties != NULL) {
		guint id = g_variant_get_uint32(g_variant_get_child_value(params, 0));
		gchar ** props = NULL;
		GError * error = NULL;

		class->get_indicator_properties (server, id, &props, &error);

		if (error != NULL) {
			g_dbus_method_invocation_return_gerror(invocation, error);
			g_error_free(error);
		} else {
			GVariant * retvals = g_variant_new_strv((const gchar * const *)props, -1);
			if (retvals != NULL) {
				retvals = g_variant_new_tuple(&retvals, 1);
			}
			g_dbus_method_invocation_return_value(invocation, retvals);
			g_strfreev(props);
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
		            indicate_server_error_quark(),
		            NO_GET_INDICATOR_PROPERTIES,
		            "get_indicator_properties function doesn't exist for this server class: %s",
		            G_OBJECT_TYPE_NAME(server));
	return;
}

static void
bus_show_indicator_to_user (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->show_indicator_to_user != NULL) {
		GError * error = NULL;
		guint id, timestamp;
		g_variant_get(params, "(uu)", &id, &timestamp);

		class->show_indicator_to_user (server, id, timestamp, &error);

		if (error != NULL) {
			g_dbus_method_invocation_return_gerror(invocation, error);
			g_error_free(error);
		} else {
			g_dbus_method_invocation_return_value(invocation, NULL);
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
		            indicate_server_error_quark(),
		            NO_SHOW_INDICATOR_TO_USER,
		            "show_indicator_to_user function doesn't exist for this server class: %s",
		            G_OBJECT_TYPE_NAME(server));
	return;
}

/* DBus function to wrap a virtual function call so that
   it can be subclassed if someone so chooses */
static void
bus_indicator_displayed (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->indicator_displayed != NULL) {
		GError * error = NULL;
		guint id;
		gboolean displayed;

		g_variant_get(params, "(ub)", &id, &displayed);

		if (class->indicator_displayed (server, g_dbus_method_invocation_get_sender(invocation), id, displayed, &error)) {
			g_dbus_method_invocation_return_value(invocation, NULL);
		} else {
			g_dbus_method_invocation_return_gerror(invocation, error);
			g_error_free(error);
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
	            indicate_server_error_quark(),
	            NO_INDICATOR_DISPLAYED,
	            "indicator_displayed function doesn't exist for this server class: %s",
	            G_OBJECT_TYPE_NAME(server));
	return;
}

/**
 * indicate_server_get_next_id:
 * @server: The #IndicateServer the ID will be on
 * 
 * Returns the next available unused ID that an indicator
 * can have.
 * 
 * Return value: A valid indicator ID.
*/
guint 
indicate_server_get_next_id (IndicateServer * server)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->get_next_id != NULL) {
		return class->get_next_id (server);
	}

	return 0;
}

/**
 * @indicate_server_get_path:
 * @server: The #IndicateServer to get the path of.
 *
 * Gets DBus object path for the exported server object.
 * 
 * Return value: The servers DBus path.
 */
const gchar *
indicate_server_get_path (IndicateServer *server)
{
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

        g_return_val_if_fail(INDICATE_IS_SERVER(server), NULL);

        return priv->path;
}

static IndicateInterests
interest_string_to_enum (const gchar * interest_string)
{
	if (!g_strcmp0(interest_string, INDICATE_INTEREST_STRING_SERVER_DISPLAY)) {
		return INDICATE_INTEREST_SERVER_DISPLAY;
	}

	if (!g_strcmp0(interest_string, INDICATE_INTEREST_STRING_SERVER_SIGNAL)) {
		return INDICATE_INTEREST_SERVER_SIGNAL;
	}

	if (!g_strcmp0(interest_string, INDICATE_INTEREST_STRING_INDICATOR_DISPLAY)) {
		return INDICATE_INTEREST_INDICATOR_DISPLAY;
	}

	if (!g_strcmp0(interest_string, INDICATE_INTEREST_STRING_INDICATOR_SIGNAL)) {
		return INDICATE_INTEREST_INDICATOR_SIGNAL;
	}

	if (!g_strcmp0(interest_string, INDICATE_INTEREST_STRING_INDICATOR_COUNT)) {
		return INDICATE_INTEREST_INDICATOR_COUNT;
	}

	return INDICATE_INTEREST_NONE;
}

static void
bus_show_interest (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->show_interest != NULL) {
		const gchar * interest = NULL;

		interest = g_variant_get_string(g_variant_get_child_value(params, 0), NULL);

		if (class->show_interest (server, g_dbus_method_invocation_get_sender(invocation), interest_string_to_enum(interest))){
			g_dbus_method_invocation_return_value(invocation, NULL);
		} else {
			g_dbus_method_invocation_return_error(invocation,
			            indicate_server_error_quark(),
			            SHOW_INTEREST_FAILED,
			            "Unable to show interest: %s",
			            interest);
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
	            indicate_server_error_quark(),
	            NO_SHOW_INTEREST,
	            "show_interest function doesn't exist for this server class: %s",
	            G_OBJECT_TYPE_NAME(server));
	return;
}

static void
bus_remove_interest (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->remove_interest != NULL) {
		const gchar * interest = NULL;

		interest = g_variant_get_string(g_variant_get_child_value(params, 0), NULL);

		if (class->remove_interest (server, g_dbus_method_invocation_get_sender(invocation), interest_string_to_enum(interest))){
			g_dbus_method_invocation_return_value(invocation, NULL);
		} else {
			g_dbus_method_invocation_return_error(invocation,
						indicate_server_error_quark(),
						REMOVE_INTEREST_FAILED,
						"Unable to remove interest: %s",
						interest);
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
				indicate_server_error_quark(),
				NO_REMOVE_INTEREST,
				"remove_interest function doesn't exist for this server class: %s",
				G_OBJECT_TYPE_NAME(server));
	return;
}

static void
bus_set_max_indicators (IndicateServer * server, GVariant * params, GDBusMethodInvocation * invocation)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->max_indicators_set != NULL) {
		gint max = g_variant_get_int32(g_variant_get_child_value(params, 0));

		if (class->max_indicators_set (server, g_dbus_method_invocation_get_sender(invocation), max)){
			g_dbus_method_invocation_return_value(invocation, NULL);
		} else {
			g_dbus_method_invocation_return_error(invocation,
			            indicate_server_error_quark(),
			            MAX_INDICATORS_SET_FAILED,
			            "Unable to set max indicators: %d",
			            max);
		}

		return;
	}

	g_dbus_method_invocation_return_error(invocation,
	            indicate_server_error_quark(),
	            NO_MAX_INDICATORS_SET,
	            "max_indicators_set function doesn't exist for this server class: %s",
	            G_OBJECT_TYPE_NAME(server));
	return;
}

/**
 * indicate_server_check_interest:
 * @server: The #IndicateServer being checked
 * @interest: Which interest type we're checking for
 * 
 * This function looks at all the interest that various listeners
 * have specified that they have for this server and returns whether
 * there is a listener that has the interest specified in @interest.
 * 
 * Return value: %TRUE if a listener as the interest otherwise %FALSE
*/
gboolean
indicate_server_check_interest (IndicateServer * server, IndicateInterests interest)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->check_interest != NULL) {
		return class->check_interest (server, interest);
	}

	g_warning("check_interest function not implemented in this server class: %s", G_OBJECT_TYPE_NAME(server));
	return FALSE;
}

/**
 * indicate_server_get_max_indicators:
 * @server: The #IndicateServer being checked
 * 
 * This function looks at everyone listening and determines
 * the max number of indicators that can be shown to anyone.
 * 
 * Return value: A max number or -1 for unlimited.
*/
gint
indicate_server_get_max_indicators (IndicateServer * server)
{
	IndicateServerClass * class = INDICATE_SERVER_GET_CLASS(server);

	if (class != NULL && class->max_indicators_get != NULL) {
		return class->max_indicators_get (server);
	}

	g_warning("get_max_indicators function not implemented in this server class: %s", G_OBJECT_TYPE_NAME(server));
	return -1;
}

/**
 * indicate_server_set_menu:
 * @server: The #IndicateServer to use
 * @menu: A #DbusmenuServer object represting the menu
 * 
 * This function sets the internal menu representation for this
 * indicator.  It makes it so that when clients ask for information
 * on the menus it'll repospond with the address of this menu.
 * 
 * It is important to note that there is not signal about menus
 * changing.  So if the indicator is already visible there is no
 * way that a listener would know about the change, and thus is
 * likely to have the wrong menu (or no menu).
 * 
 * Return value: None.
*/
void
indicate_server_set_menu (IndicateServer * server, DbusmenuServer * menu)
{
	g_return_if_fail(INDICATE_IS_SERVER(server));
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(server);

	if (priv->visible) {
		g_warning("Menu being changed when the indicator is visible.  Listeners will NOT be notified of this change.");
	}

	if (priv->dbusmenu != NULL) {
		g_object_unref(priv->dbusmenu);
		priv->dbusmenu = NULL;
	}

	priv->dbusmenu = menu;
	g_object_ref(priv->dbusmenu);

	return;
}

/* *** Folks stuff *** */

/* This checks for folk by looking at the sender value
   and sees if we can find any that have the same sender. */
static gint
indicate_server_interested_folks_equal (gconstpointer a, gconstpointer b)
{
	return g_strcmp0(((IndicateServerInterestedFolk *)a)->sender,((IndicateServerInterestedFolk *)b)->sender);
}

/* Creates a IndicateServerInterestedFolk structure and
   initializes the default values to reasonable defaults. */
static void
indicate_server_interested_folks_init (IndicateServerInterestedFolk * folk, const gchar * sender, const gchar * path, GDBusConnection * connection)
{
	folk->sender = g_strdup(sender);

	guint i;
	for (i = INDICATE_INTEREST_NONE; i < INDICATE_INTEREST_LAST; i++) {
		folk->interests[i] = FALSE;
	}

	folk->max_indicators = MAX_INDICATORS_UNSET;

	folk->indicators_displayed = g_hash_table_new(g_direct_hash, g_direct_equal);

	g_dbus_proxy_new(connection,
	                 G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES | G_DBUS_PROXY_FLAGS_DO_NOT_CONNECT_SIGNALS,
	                 bus_interface_info,
	                 sender,
	                 path,
	                 INDICATE_LISTENER_DBUS_IFACE,
	                 NULL, /* TODO: cancelable */
	                 folk_proxy_cb,
	                 folk);

	return;
}

/* Look at when the name owner changes so that we can figure out
   if we need to delete the folks item */
static void
folks_name_owner_change (GObject * object, GParamSpec * pspec, gpointer user_data)
{
	gchar * name = g_dbus_proxy_get_name_owner(G_DBUS_PROXY(object));
	if (name != NULL) {
		/* What?  Well, no matter, we don't care if it's there, we'll
		   just ignore this signal and move on. */
		g_free(name);
		return;
	}

	/* g_debug("\tBeing removed, interesting"); */
	IndicateServerPrivate * priv = INDICATE_SERVER_GET_PRIVATE(user_data);

	IndicateServerInterestedFolk searchitem;
	searchitem.sender = (gchar *)name;
	GList * entry = g_list_find_custom(priv->interestedfolks, &searchitem, indicate_server_interested_folks_equal);

	if (entry == NULL) {
		/* g_debug("\tWe don't have it, not interesting"); */
		return;
	}

	IndicateServerInterestedFolk * folk = (IndicateServerInterestedFolk *)entry->data;
	priv->interestedfolks = g_list_remove(priv->interestedfolks, entry->data);

	guint i;
	for (i = INDICATE_INTEREST_NONE; i < INDICATE_INTEREST_LAST; i++) {
		priv->interests[i] = FALSE;
	}

	GList * listi = NULL;
	for (listi = priv->interestedfolks ; listi != NULL ; listi = listi->next) {
		IndicateServerInterestedFolk * folkpointer = (IndicateServerInterestedFolk *)listi->data;
		/* g_debug("\tRebuild list from folk: %s", folkpointer->sender); */
		indicate_server_interested_folks_copy(folkpointer, priv->interests);
	}

	for (i = INDICATE_INTEREST_NONE; i < INDICATE_INTEREST_LAST; i++) {
		/* g_debug("\tComparing interests.  Interest: %d  Folk: %d  Everyone: %d", i, folk->interests[i], priv->interests[i]); */
		if (folk->interests[i] && !priv->interests[i]) {
			/* We can only remove interest here.  Think about it for a
			   moment and I think you'll be cool with it. */
			/* g_debug("\tOh, and it was interested in %d.  Not anymore.", i); */
			g_signal_emit(G_OBJECT(user_data), signals[INTEREST_REMOVED], 0, i, TRUE);
		}
	}

	/* If the retired folk has set it's max indicators
	   and it's the value we're using, we need to recalculate */
	if (folk->max_indicators != MAX_INDICATORS_UNSET && folk->max_indicators == priv->max_indicators) {
		recalculate_max_indicators(user_data);
	}

	g_hash_table_foreach(folk->indicators_displayed, indicator_display_check_recalc, INDICATE_SERVER(user_data));

	/* Finally destory everything */
	indicate_server_interested_folks_destroy(folk);
	return;

}

/* Callback for creating the proxy on the folks object */
static void
folk_proxy_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;

	GDBusProxy * proxy = g_dbus_proxy_new_finish(res, &error);
	if (error != NULL) {
		g_warning("Unable to get folks proxy!");
		g_error_free(error);
		return;
	}

	IndicateServerInterestedFolk * folk = (IndicateServerInterestedFolk *)user_data;
	folk->proxy = proxy;

	g_signal_connect(G_OBJECT(proxy), "notify::g-name-owner", G_CALLBACK(folks_name_owner_change), folk);

	return;
}

/* Identifies an interest in the folk. */
static void
indicate_server_interested_folks_set (IndicateServerInterestedFolk * folk, IndicateInterests interest, gboolean value)
{
	folk->interests[interest] = value;
	return;
}

/* Copies the interest table from a folk structure into a 
   generic gboolean array of length INDICATE_INTEREST_LAST */
static void
indicate_server_interested_folks_copy (IndicateServerInterestedFolk * folk, gboolean * interests)
{
	guint i;
	for (i = INDICATE_INTEREST_NONE; i < INDICATE_INTEREST_LAST; i++) {
		if (folk->interests[i]) {
			interests[i] = TRUE;
		}
	}

	return;
}

/* Destroys all the internal parts of the IndicateServerInterestedFolk
   and then destroys the structure itself */
static void
indicate_server_interested_folks_destroy(IndicateServerInterestedFolk * folk)
{
	g_free(folk->sender);
	g_hash_table_destroy(folk->indicators_displayed);

	if (folk->proxy != NULL) {
		g_object_unref(folk->proxy);
	}

	g_free(folk);
	return;
}

/* *** End Folks *** */
