#!/usr/local/bin/perl
# Work out the bandwidth usage for all virtual servers.
# For those that are over their limit, send a warning.

$no_acl_check++;
require './virtual-server-lib.pl';
use Time::Local;

# Work out the start of the monitoring period
$now = time();
$day = int($now / (24*60*60));
@tm = localtime(time());
if ($config{'bw_past'} eq 'week') {
	# Start on last sunday
	$start_day = $day - $tm[6];
	}
elsif ($config{'bw_past'} eq 'month') {
	# Start at 1st of month
	$start_day = int(timelocal(0, 0, 0, 1, $tm[4], $tm[5]) / (24*60*60));
	}
elsif ($config{'bw_past'} eq 'year') {
	# Start at start of year
	$start_day = int(timelocal(0, 0, 0, 1, 0, $tm[5]) / (24*60*60));
	}
else {
	# Start N days ago
	$start_day = $day - $config{'bw_period'};
	}

# For each server, scan it's log files for all usage since the last check, and
# update the count for each day.
foreach $d (&list_domains()) {
	# Add bandwidth for all features
	$bwinfo = &get_bandwidth($d);
	foreach $f (@features) {
		local $bwfunc = "bandwidth_$f";
		if (defined(&$bwfunc)) {
			$bwinfo->{"last_$f"} =
				&$bwfunc($d, $bwinfo->{"last_$f"}, $bwinfo);
			}
		}

	# Prune days more than 1 year old
	foreach $k (keys %$bwinfo) {
		if ($k =~ /^\d+$/ && $k < $day - 366) {
			delete($bwinfo->{$k});
			}
		}
	&save_bandwidth($d, $bwinfo);
	}

# For each server, sum up usage over the monitoring period to find those
# that are over their limit
foreach $d (&list_domains()) {
	next if ($d->{'parent'});

	# Sum up usage for domain and sub-domains
	$usage = 0;
	foreach $dd ($d, &get_domain_by("parent", $d->{'id'})) {
		$bwinfo = &get_bandwidth($dd);
		local $usage_only = 0;
		foreach $k (keys %$bwinfo) {
			if ($k =~ /^\d+$/ && $k >= $start_day) {
				$usage += $bwinfo->{$k};
				$usage_only += $bwinfo->{$k};
				}
			}
		$dd->{'bw_usage_only'} = $usage_only;
		$dd->{'bw_start'} = $start_day;
		if ($d ne $dd) {
			&save_domain($dd);
			}
		}
	$d->{'bw_usage'} = $usage;
	if ($d->{'bw_limit'} && $usage > $d->{'bw_limit'}) {
		# Over the limit! But check limit on how often to notify
		$etime = $now - $d->{'bw_notify'} > $config{'bw_notify'}*60*60;
		if ($etime) {
			# Time to email ..
			$tmpl = $config{'bw_template'} eq 'default' ?
				"$module_config_directory/bw-template" :
				$config{'bw_template'};
			%tkeys = %$d;
			$tkeys{'bw_limit'} = &nice_size($tkeys{'bw_limit'});
			$tkeys{'bw_usage'} = &nice_size($tkeys{'bw_usage'});
			$tkeys{'bw_period'} = $config{'bw_period'};
			local @addrs;
			push(@addrs, $d->{'email'} ||
				   $d->{'user'}.'@'.&get_system_hostname() )
				if ($config{'bw_owner'});
			push(@addrs, split(/\s+,\s+/, $config{'bw_email'}));
			@erv = &send_template_email(
				&cat_file($tmpl), join(", ", @addrs),
				\%tkeys, &text('newbw_subject', $d->{'dom'}));
			if ($erv[0]) {
				$d->{'bw_notify'} = $now;
				}
			else {
				print STDERR "Failed to send email : $erv[1]\n";
				}
			}
		if (!$d->{'disabled'} && $etime && $config{'bw_disable'}) {
			# Time to disable
			&set_all_null_print();
			@disable = grep { $d->{$_} && $config{$_} }
				        split(/,/, $config{'disable'});
			push(@disable, "ssl")
			    if (&indexof("web", @disable) >= 0 && $d->{'ssl'});

			%disable = map { $_, 1 } @disable;

			# Run the before command
			&set_domain_envs($d, "DISABLE_DOMAIN");
			$merr = &making_changes();
			next if ($merr);

			# Disable all configured features
			my $f;
			foreach $f (@features) {
				if ($d->{$f} && $disable{$f}) {
					local $dfunc = "disable_$f";
					&$dfunc($d);
					push(@disabled, $f);
					}
				}

			# Save new domain details
			$d->{'disabled'} = join(",", @disabled);
			$d->{'disabled_reason'} = 'bw';

			# Run the after command
			&run_post_actions();
			&set_domain_envs($d, "DISABLE_DOMAIN");
			&made_changes();
			&webmin_log("disable", "domain", $d->{'dom'}, $d);
			}
		}
	elsif ($d->{'bw_limit'} && $config{'bw_warn'} &&
	       $usage > $d->{'bw_limit'}*$config{'bw_warn'}/100) {
		# Reached the warning limit! But check limit on how often warn
		if ($now - $d->{'bw_warnnotify'} > $config{'bw_notify'}*60*60) {
			# Time to email ..
			$tmpl = $config{'warnbw_template'} eq 'default' ?
				"$module_config_directory/warnbw-template" :
				$config{'warnbw_template'};
			%tkeys = %$d;
			$tkeys{'bw_limit'} = &nice_size($tkeys{'bw_limit'});
			$tkeys{'bw_usage'} = &nice_size($tkeys{'bw_usage'});
			$tkeys{'bw_period'} = $config{'bw_period'};
			$tkeys{'bw_percent'} = int(100*$usage/$d->{'bw_limit'});
			$tkeys{'bw_warn'} = $config{'bw_warn'};
			local @addrs;
			push(@addrs, $d->{'email'} ||
				   $d->{'user'}.'@'.&get_system_hostname() )
				if ($config{'bw_owner'});
			push(@addrs, split(/\s+,\s+/, $config{'bw_email'}));
			@erv = &send_template_email(
				&cat_file($tmpl), join(", ", @addrs),
				\%tkeys,
				&text('newbw_warnsubject', $d->{'dom'}));
			if ($erv[0]) {
				$d->{'bw_warnnotify'} = $now;
				}
			else {
				print STDERR "Failed to send email : $erv[1]\n";
				}
			}
		}
	&save_domain($d);
	}

