//<copyright>
//
// Copyright (c) 1995,96
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        writevrml.C
//
// Purpose:     implementation of SDFScene::writeVRML (conversion SDF to VRML)
//
// Created:      5 May 95   Michael Pichler (with some assistance by Peter Mikl in May 95)
//
// Changed:     11 Jun 96   Michael Pichler
//
// $Id: writevrml.C,v 1.4 1997/02/25 17:03:58 mpichler Exp $
//
//</file>


#include "sdfscene.h"
#include "geomobj.h"
#include "material.h"
#include "light.h"

#include <ge3d/ge3d.h>
#include <iostream.h>


const char* const vrml_header = "#VRML V1.0 ascii\n";


// writeVRML
// write the current SDF scene in VRML format

// assumes that transformation matrices have been built (for objects
// with parents) and makes use of ge3d calls for matrix calculations.
// Therefore it can only be used interactive and not in batch mode.

void SDFScene::writeVRML (ostream& os)
{
  os << vrml_header << endl;
  os << "# written with VRweb (converted from SDF)" << endl;

  os << "\nSeparator {" << endl;

  os << "\nShapeHints { vertexOrdering COUNTERCLOCKWISE shapeType SOLID faceType CONVEX }\n" << endl;

  // TODO: camera

  // material definitions
  // enclosed into switch (used on demand)
  os << "Switch {" << endl;
  for (const Material* matptr = (Material*) materlist_.first ();  matptr;
                       matptr = (Material*) materlist_.next ())
  {
    const colorRGB* color = matptr->natural ();
    const char* texture = matptr->texture ();  // non-NULL
    os << "DEF Mat_" << matptr->name () << ' ';
    if (*texture)  // Material + Texture2
      os << "Group { ";
    os << "Material { ";
    os << "diffuseColor " << color->R << ' ' << color->G << ' ' << color->B;
    if (*texture)  // Texture2
      os << " } Texture2 { filename textures/" << texture << " }";
    os << " }\n";
  }
  os << "} # materials\n" << endl;

  static point3D deflgtpos = { 0.0, 0.0, 0.0 };
  static vector3D deflgtdir = { -1.0, 0.0, 0.0 };

  // light sources
  for (const Light* lgtptr = (Light*) lightlist_.first ();  lgtptr;
                    lgtptr = (Light*) lightlist_.next ())
  {
    // transforming position/orientation (SDF and VRML conventions differ)
    // TODO: light sources relative to camera

    ge3d_push_new_matrix ((const float (*) [4]) lgtptr->getTrfMat ());

    const colorRGB* intens = lgtptr->intensity ();
    if (lgtptr->positional ())
    {
      point3D pos;
      ge3dTransformMcWc (&deflgtpos, &pos);

      os << "PointLight {\n"
         << "\tcolor " << intens->R << ' ' << intens->G << ' ' << intens->B << '\n'
         << "\tlocation " << pos.x << ' ' << pos.y << ' ' << pos.z << '\n'
         << "}\n" << endl;
    }
    else // directional light
    { // SDF (WF): vector towards light source, default (-1, 0, 0) ("from left")
      // WRL (IV): vector parallel to light emission, default (0, 0, -1) ("to the back")
      // TODO (construct sample scene first)
      os << "DirectionalLight {\n"
         << "}\n" << endl;
    }

    ge3d_pop_matrix ();
  } // lights

  // geometric objects
  for (const GeometricObject* gobjptr = (GeometricObject*) gobjlist_.first ();  gobjptr;
                              gobjptr = (GeometricObject*) gobjlist_.next ())
  {
    // mpichler, 19960611:
    // changed from TransformSeparator to Separator to avoid textures leaking out
    os << "Separator {\n";
    gobjptr->writeVRMLTransformation (os);

    gobjptr->writeVRML (os);  // also handles shared objects

    os << "}\n" << endl;

  }  // for gobjptr

  os << "} # EOF" << endl;  // outermost Seperator

} // writeVRML
