/* -*- mode: c; c-file-style: "gnu" -*-
 * mime.c -- MIME-type detection routines
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file mime.c
 * MIME-type detection routines.
 *
 * MIME detection can be done in two very different ways: using an
 * postfix-based approach, and using the libmagic. The first one
 * is way faster and more lightweight and secure, but probably less
 * accurate.
 *
 * The postfix-based approach simply compares the filename's end with
 * given string, and if they match, the MIME type is found. There is
 * no limitation, the postfix can be as long as one wants. It might
 * even be a full path name, not only an extension.
 *
 * The other approach - as said above - is using the libmagic
 * library. This is highly dependant on the contents of the magic
 * file, and involves lots of lookups. But the postfix approach has
 * that problem too, so...
 */

#include "system.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif

#include "compat/compat.h"
#include "bh-libs/list.h"

#include "config.h"
#include "mime.h"
#include "mime-types.h"
#include "options.h"
#include "types.h"

#if THY_OPTION_FILE_MAGIC
#include <magic.h>
#endif

/** @internal Global variable to hold the MIME type.
 * This one will hold the mime type of the last looked up filename.
 */
static const char *mime_type_str = NULL;

#if THY_OPTION_FILE_MAGIC
/** @internal Dummy variable.
 * We need this to link to libmagic.
 */
int nomgc = 0;

/** @internal Magic cookie.
 * Used by the functions in libmagic.
 */
static magic_t _mime_magic_cookie = NULL;
/** @internal libmagic initialisation flag.
 * If set to non-zero, it means that Thy already tried to initialise
 * libmagic.
 */
static int _mime_magic_initialised = 0;

/** @internal Initialise libmagic.
 * Sets up the global magic cookie and loads the default magic
 * databases.
 *
 * Sets #_mime_magic_cookie to non-NULL on success, leaves it at NULL
 * on error.
 */
static void
_mime_file_magic_init (void)
{
  _mime_magic_cookie = magic_open (MAGIC_SYMLINK | MAGIC_MIME);
  if (_mime_magic_cookie)
    {
      if (magic_load (_mime_magic_cookie, NULL) == -1)
	{
	  magic_close (_mime_magic_cookie);
	  _mime_magic_cookie = NULL;
	}
    }
  _mime_magic_initialised = 1;
}
#endif

/** @internal Determine if a file is of a given mime-type.
 * Check if FILENAME's extension matches the extension part of MIME
 * (case sensitivity determined by the value of CASEMIME).
 *
 * @param mime is the MIME type to check against.
 * @param filename is the file name to perform the check on.
 * @param flen is its length
 * @param casemime controls whether the check is case sensitive or
 * not.
 *
 * @returns MIME's type member, or NULL if they do not match.
 */
static const char *
_mime_type_check (const pair_t *mime, const char *filename,
		  size_t flen, int casemime)
{
  if (casemime)
    {
      if (!strcmp (&filename[flen - strlen (mime->field)],
		   mime->field))
	return mime->value;
    }
  else
    {
      if (!strcasecmp (&filename[flen - strlen (mime->field)],
		       mime->field))
	return mime->value;
    }
  return NULL;
}

/** @internal Determine a file's MIME type from the internal list.
 * Iterate through the internal and user specified MIME type lists and
 * check if the given filename matches any of them.
 *
 * @param filename is who's type should be determined.
 * @param absuri is the absolute URL of the file.
 *
 * @returns Zero on success, -1 on error.
 *
 * @note Modifies #mime_type_str in-place!
 */
static int
_mime_type_lookup (const char *filename, const char *absuri)
{
  size_t i = 0, flen = strlen (filename);
  thy_mappable_config_t *config = config_get_mapped (absuri, filename);

  for (i = 0; i < bhl_list_size (config->mime_types); i++)
    {
      pair_t *t;

      bhl_list_get (config->mime_types, i, (void **)&t);
      mime_type_str = _mime_type_check
	(t, filename, flen, (config->options.casemime == THY_BOOL_TRUE));
      free (t);
      if (mime_type_str)
	{
	  free (config);
	  return 0;
	}
    }

  i = 0;
  while (mime_types[i].value)
    {
      mime_type_str = _mime_type_check
	(&mime_types[i], filename, flen,
	 (config->options.casemime == THY_BOOL_TRUE));
      if (mime_type_str)
	{
	  free (config);
	  return 0;
	}
      i++;
    }

  free (config);
  return -1;
}

#if THY_OPTION_FILE_MAGIC
/** Determine a file's MIME type via libmagic.
 * Pass the filename down to libmagic, then frob the result so only
 * the MIME types remain.
 *
 * @param filename is who's type should be determined.
 *
 * @returns Zero on success, -1 on error.
 *
 * @note Modified #mime_type_str in-place!
 */
static int
_mime_type_magic (const char *filename)
{
  char *tmp;

  if (!_mime_magic_initialised)
    _mime_file_magic_init ();

  if (!_mime_magic_cookie)
    return -1;

  mime_type_str = magic_file (_mime_magic_cookie, filename);
  if (!mime_type_str)
    return -1;

  tmp = strchr (mime_type_str, ',');
  if (tmp && tmp != mime_type_str)
    tmp[0] = '\0';
  return 0;
}
#endif

/** Determine the MIME type of a file.
 * Calls various functions to determine the MIME-type of the given
 * filename.
 *
 * @param filename is the file whose MIME type we're intersted in.
 * @param default_type is the MIME type to return if nothing
 * appropriate was found.
 * @param absuri is the absolute URL of the file.
 *
 * @returns A pointer to a static buffer containing either the MIME
 * type, or default_type.
 */
const char *
mime_type (const char *filename, const char *default_type,
	   const char *absuri)
{
  int i = -1;

  mime_type_str = NULL;

  i = _mime_type_lookup (filename, absuri);

#if THY_OPTION_FILE_MAGIC
  if (i == 0)
    i = _mime_type_magic (filename);
#endif

  if (i == 0)
    return mime_type_str;
  else
    return default_type;
}
