/* -*- mode: c; c-file-style: "gnu" -*-
 * src/methods/passwd.c -- passwd authentication routines
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy-Auth.
 *
 * Thy-Auth is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Thy-Auth is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file passwd.c
 * /etc/passwd-style authentication. This is used by all the
 * Authorisers who wish to authenticate against a user:password file.
 */

#include "compat/compat.h"
#include "methods/passwd.h"

#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

/** Parse an AuthGroupFile.
 * Find the relevant group in AuthGroupFile, and split the members
 * into a NULL terminated char * array.
 *
 * @param fn is the file to read.
 * @param group is the group we're searching for.
 *
 * @returns A NULL-terminated list of members, or NULL on error.
 *
 * @note All members of the array, and the array itself must be freed
 * by the caller!
 */
char **
thy_auth_passwd_group_get (const char *fn, const char *group)
{
  FILE *f;
  char *line = NULL;
  char **members = NULL, *member;
  size_t n = 0, alloc = 10, count = 0;
  int l, gl;

  f = fopen (fn, "r");
  if (f == NULL)
    return NULL;

  gl = strlen (group);

  while ((l = getline (&line, &n, f)) != -1)
    {
      int p = 0;

      if (l < gl)
	continue;
      if (memcmp (line, group, gl) != 0 || line[gl] != ':')
	continue;

      members = (char **)bhc_calloc (alloc, sizeof (char *));

      /* Remove the trailing \r or \n. */
      while (p < l)
	{
	  if (line[p] == '\r' || line[p] == '\n')
	    line[p] = '\0';
	  p++;
	}

      member = strtok (&line[gl+1], " ");
      do
	{
	  if (!member)
	    break;

	  members[count++] = strdup (member);
	  if (count > alloc)
	    {
	      alloc *= 2;
	      XSREALLOC (members, char *, alloc);
	    }
	} while ((member = strtok (NULL, " ")) != NULL);

      XSREALLOC (members, char *, (alloc > count) ? alloc : alloc + 1);
      members[count] = NULL;
      break;
    }
  fclose (f);

  return members;
}

/** Authenticate a user against a passwd-style file.
 * @param file is the file to authenticate against.
 * @param user is the name of the user to authenticate.
 * @param pw is his password.
 *
 * @returns Zero on sucess, -1 if authentication failed.
 */
int
thy_auth_passwd_auth (const char *file, const char *user, const char *pw)
{
  FILE *f;
  char *line = NULL, *pw_pass, *salt, *cpw;
  int ul, n = 0, i, l;

  if (!user || !pw || !file)
    return -1;

  ul = strlen (user);

  f = fopen (file, "r");
  if (f == NULL)
    return -1;

  while ((l = getline (&line, &n, f)) != -1)
    {
      if (l < ul)
	continue;
      if (memcmp (line, user, ul) != 0 || line[ul] != ':')
	continue;

      pw_pass = strndup (&line[ul + 1], l - ul - 2);
      for (i = 0; pw_pass[i] != ':' && i < l - ul - 2; i++) ;
      pw_pass[i] = '\0';

      if (memcmp (pw_pass, "$1$", 3) == 0)
	{
	  i = 4;
	  while (pw_pass[i] != '$' && i < l - ul - 1)
	    i++;
	  salt = strndup (pw_pass, i);
	}
      else
	salt = strndup (pw_pass, 2);
      cpw = crypt (pw, salt);

      free (salt);
      if (!strcmp (cpw, pw_pass))
	{
	  free (pw_pass);
	  return 0;
	}
      else
	{
	  free (pw_pass);
	  return -1;
	}
    }
  return -1;
}

/* arch-tag: 919a8a38-889b-4937-8cc4-66c620d78224 */
