/*

    File: parti386.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <ctype.h>      /* toupper, tolower */
#include "types.h"
#include "common.h"
#include "testdisk.h"
#include "fnctdsk.h"
#include "lang.h"
#include "intrf.h"
#include "chgtype.h"
#include "bfs.h"
#include "swap.h"
#include "bsd.h"
#include "fat.h"
#include "ntfs.h"
#include "ext2.h"
#include "netware.h"
#include "rfs.h"
#include "lvm.h"
#include "md.h"
#include "jfs_superblock.h"
#include "jfs.h"
#include "savehdr.h"
#include "cramfs.h"
#include "xfs.h"
#include "ufs.h"
#include "sysv.h"

static int is_extended(const unsigned int part_type);
static int can_be_ext(const t_param_disk *disk_car, t_partition *partition);
static int test_structure_i386(t_list_part *list_part);

#define s_cyl(p) (((p)->cyl & (unsigned)0xff) | (((p)->sector << 2) & (unsigned)0x300))
#define s_head(p) ((p)->head)
#define s_sect(p) ((p)->sector & (unsigned)0x3f)
#define e_cyl(p) (((p)->end_cyl & (unsigned)0xff) | (((p)->end_sector << 2) & (unsigned)0x300))
#define e_head(p) ((p)->end_head)
#define e_sect(p) ((p)->end_sector & (unsigned)0x3f)
static uint64_t get_start_sect(const struct partition_dos *p);
static uint64_t get_nr_sects(const struct partition_dos *p);
static void aff_entry_rapport(const t_param_disk *disk_car, const struct partition_dos*);
static int get_geometry_from_i386mbr(const unsigned char *buffer, const int debug, t_CHS *geometry);
static t_list_part *get_ext_data_i386(t_param_disk *disk_car, t_list_part *list_part, const int debug, const int saveheader);
static void test_MBR_data(t_list_part *list_part);
static int test_MBR_over(t_param_disk *disk_car,t_list_part *list_part);
static int write_mbr_i386(t_param_disk *disk_car, const t_list_part *list_part, const int ro, const int debug);
static int write_all_log_i386(t_param_disk *disk_car, const t_list_part *list_part, const int ro , const int debug, const int align);
static int diff(const t_param_disk *disk_car,const unsigned char buffer[DEFAULT_SECTOR_SIZE], const unsigned char buffer_org[DEFAULT_SECTOR_SIZE]);

static t_list_part *read_part_i386(t_param_disk *disk_car, const int debug, const int saveheader);
static int write_part_i386(t_param_disk *disk_car, const t_list_part *list_part, const int ro , const int debug, const int align);
static t_list_part *init_part_order_i386(const t_param_disk *disk_car, t_list_part *list_part);
static int write_MBR_code_i386(unsigned char *buffer);
static t_list_part *add_partition_i386(t_param_disk *disk_car,t_list_part *list_part, const int debug);
static void set_prev_status_i386(const t_param_disk *disk_car, t_partition *partition);
static void set_next_status_i386(const t_param_disk *disk_car, t_partition *partition);
static int is_part_type_ok_i386(unsigned int part_type);
static void init_structure_i386(const t_param_disk *disk_car,t_list_part *list_part, const int debug);
static void erase_list_part_i386(unsigned char *buffer);
static int check_part_i386(t_param_disk *disk_car, const int debug,t_partition *partition,const int saveheader);

static void partition2_i386_entry(const t_param_disk *disk_car,const uint64_t pos,const t_partition *partition, struct partition_dos *p);
static int i386_entry2partition(t_param_disk *disk_car, const uint64_t offset, t_partition *partition, const struct partition_dos *p, const status_type_t status,const unsigned int order,const int debug, const int saveheader);
static const char* errmsg_i386_entry2partition(const errcode_type_t errcode);
static const struct systypes i386_sys_types[] = {
  {"\x00" "empty"},
  {"\x01" "FAT12"},
  {"\x02" "XENIX root"},
  {"\x03" "XENIX /usr"},
  {"\x04" "FAT16 <32M"},
  {"\x05" "extended"},
  {"\x06" "FAT16 >32M"},
  {"\x07" "HPFS - NTFS"},
  {"\x09" "AIX data"},
  {"\x0a" "OS/2 Boot Manager"},
  {"\x0b" "FAT32"},
  {"\x0c" "FAT32 LBA"},
  {"\x0E" "FAT16 LBA"},
  {"\x0F" "extended LBA"},
  {"\x10" "OPUS"},
  {"\x11" "hid. FAT12"},
  {"\x12" "Compaq Diagnostics"},
  {"\x14" "hid. FAT16 <32M"},
  {"\x16" "hid. FAT16 >32M"},
  {"\x17" "hid. HPFS/NTFS"},
  {"\x18" "AST swap"},
  {"\x19" "Willowtech Photon"},
  {"\x1b" "hid. FAT32"},
  {"\x1c" "hid. FAT32 LBA"},
  {"\x1e" "hid. FAT16 LBA"},
  {"\x20" "Willowsoft OFS1"},
  {"\x24" "NEC MS-DOS 3.x"},
  {"\x38" "Theos"},
  {"\x3c" "PMagic recovery"},
  {"\x40" "VENIX 80286"},
  {"\x41" "PPC PReP Boot"},
  {"\x42" "W2K Dynamic/SFS"},
  {"\x50" "OnTrack DM RO"},
  {"\x51" "OnTrack DM RW-NOVEL"},
  {"\x52" "CP/M-Microport V/386"},
  {"\x53" "OnTrack DM WO ???"},
  {"\x54" "OnTrack DM DDO"},
  {"\x55" "EZ-Drive"},
  {"\x56" "GoldenBow VFeature"},
  {"\x61" "SpeedStor"},
  {"\x63" "Unixware, HURD, SCO"},
  {"\x64" "NetWare 286"},
  {"\x65" "NetWare 3.11+"},
  {"\x67" "Novell"},
  {"\x68" "Novell"},
  {"\x69" "Novell"},
  {"\x70" "DiskSecure MB"},
  {"\x75" "PC/IX"},
  {"\x80" "Minix v1.1-1.4a"},
  {"\x81" "Minix / old Linux"},
  {"\x82" "Linux Swap"},
  {"\x83" "Linux"},
  {"\x85" "Linux extended"},
  {"\x86" "NT FAT16 V/S set"},
  {"\x87" "HPFS FT mirror-V/S set"},
  {"\x8e" "Linux LVM"},
  {"\x93" "Amoeba"},
  {"\x94" "Amoeba bad block"},
  {"\xa0" "NoteBIOS save2disk"},
  {"\xa5" "FreeBSD"},
  {"\xa6" "OpenBSD"},
  {"\xa8" "Darwin UFS"},
  {"\xa9" "NetBSD"},
  {"\xab" "Darwin boot"},
  {"\xb7" "BSDI"},
  {"\xb8" "BSDI swap"},
  {"\xbe" "Solaris boot"},
  {"\xc1" "secured FAT12"},
  {"\xc4" "secured FAT16"},
  {"\xc6" "sec. Huge-bad FAT16"},
  {"\xc7" "Syrinx Boot-bad NTFS"},
  {"\xd8" "CP/M-86"},
  {"\xdb" "CP/M"},
  {"\xde" "Dell Utility"},
  {"\xe1" "SpeedStor FAT12 ext"},
  {"\xe3" "DOS RO"},
  {"\xe4" "SpeedStor FAT16 ext"},
  {"\xeb" "BeFS"},
  {"\xee" "EFI GPT"},          /* Intel EFI GUID Partition Table */
  {"\xef" "EFI (FAT-12/16/32)"},/* Intel EFI System Partition */
  {"\xf0" "Linux/PA-RISC boot"},/* Linux/PA-RISC boot loader */
  {"\xf1" "Storage Dimensions"},
  {"\xf2" "DOS secondary"},
  {"\xf4" "SpeedStor"},
  {"\xfd" "Linux RAID"},
  {"\xfe" "LANstep"},
  {"\xff" "Xenix bad block"},
  { NULL }
};

t_arch_fnct arch_i386= {
  .part_name="Partition table type : Intel",
  .parttype_name_table=i386_sys_types,
  .read_part=read_part_i386,
  .write_part=write_part_i386,
  .init_part_order=init_part_order_i386,
  .get_geometry_from_mbr=get_geometry_from_i386mbr,
  .check_part=check_part_i386,
  .write_MBR_code=write_MBR_code_i386,
  .add_partition=add_partition_i386,
  .set_prev_status=set_prev_status_i386,
  .set_next_status=set_next_status_i386,
  .test_structure=test_structure_i386,
  .is_part_type_ok=is_part_type_ok_i386,
  .init_structure=init_structure_i386,
  .erase_list_part=erase_list_part_i386
};

static void store4_little_endian(unsigned char *cp, unsigned int val)
{
  cp[0] = (val & 0xff);
  cp[1] = ((val >> 8) & 0xff);
  cp[2] = ((val >> 16) & 0xff);
  cp[3] = ((val >> 24) & 0xff);
}

static unsigned int read4_little_endian(const unsigned char *cp)
{
  return (unsigned int)(cp[0]) + ((unsigned int)(cp[1]) << 8) + ((unsigned int)(cp[2]) << 16) + ((unsigned int)(cp[3]) << 24);
}

static uint64_t get_start_sect(const struct partition_dos *p)
{
  return read4_little_endian(p->start4);
}

uint64_t get_nr_sects(const struct partition_dos *p)
{
  return read4_little_endian(p->size4);
}

static void set_nr_sects(struct partition_dos *p, unsigned int nr_sects)
{
  store4_little_endian(p->size4, nr_sects);
}

static void set_start_sect(struct partition_dos *p, unsigned int start_sect)
{
  store4_little_endian(p->start4, start_sect);
}


int get_geometry_from_i386mbr(const unsigned char *buffer, const int debug, t_CHS *geometry)
{
  unsigned int i;
  geometry->cylinder=0;
  geometry->head=0;
  geometry->sector=0;
  if(debug>1)
  {
    ecrit_rapport("get_geometry_from_i386mbr\n");
  }
  if((buffer[0x1FE]!=(unsigned char)0x55)||(buffer[0x1FF]!=(unsigned char)0xAA))
  {
    return 1;
  }
  for(i=0;i<4;i++)
  {
    const struct partition_dos *p=pt_offset_const(buffer,i);
    if(p->sys_ind!=0)
    {
      if(geometry->cylinder<e_cyl(p))
	geometry->cylinder=e_cyl(p);
      if(geometry->head<e_head(p))
	geometry->head=e_head(p);
      if(geometry->sector<e_sect(p))
	geometry->sector=e_sect(p);
    }
  }
  if(geometry->sector>0)
  {
    ecrit_rapport("Geometry from i386 MBR: head=%u sector=%u\n",geometry->head+1,geometry->sector);
  }
  return 0;
}

static t_list_part *init_part_order_i386(const t_param_disk *disk_car, t_list_part *list_part)
{
  int nbr_log=0;
  int nbr_prim=0;
  t_list_part *element;
  for(element=list_part;element!=NULL;element=element->next)
  {
    switch(element->part->status)
    {
      case STATUS_PRIM:
      case STATUS_PRIM_BOOT:
      case STATUS_EXT:
	element->part->order=++nbr_prim;
	break;
      case STATUS_LOG:
	element->part->order=(++nbr_log)+4;
	break;
      default:
	ecrit_rapport("init_part_order_i386: severe error\n");
	break;
    }
  }
  return list_part;
}


static t_list_part *read_part_i386(t_param_disk *disk_car, const int debug, const int saveheader)
{
  unsigned char buffer[disk_car->sector_size];
  t_list_part *new_list_part=NULL;
  unsigned int i;
  int res=0;
  t_CHS geometry;
  aff_buffer(BUFFER_RESET,"Q");
  if(read_MBR(disk_car,&buffer)!=0)
    return NULL;
  if(get_geometry_from_i386mbr(buffer,debug,&geometry)!=0)
  {
    aff_buffer(BUFFER_ADD,msg_TBL_NMARK);
    return NULL;
  }
  for(i=0;i<4;i++)
  {
    const struct partition_dos *p=pt_offset(buffer,i);
    status_type_t status=STATUS_PRIM;
    switch(p->sys_ind)
    {
      case P_NO_OS:
	break;
      case P_EXTENDX:
      case P_EXTENDED:
      case P_LINUXEXTENDX:
	status=STATUS_EXT;
	/* don't put a break */
      default:
	{
	  t_partition *new_partition=partition_new();
	  i386_entry2partition(disk_car, (uint64_t)0, new_partition, p, status,i+1,debug,saveheader);
	  if(debug>1)
	    aff_entry_rapport(disk_car,p);
	  aff_part_buffer(AFF_PART_ORDER,disk_car,new_partition);
	  if(new_partition->errcode!=BAD_NOERR)
	  {
	    aff_buffer(BUFFER_ADD,"%s\n",errmsg_i386_entry2partition(new_partition->errcode));
	    res|=1;
	  }
	  new_list_part=insert_new_partition(new_list_part,new_partition);
	}
	break;
    }
  }
/*  if(res==0) */
  {
    test_MBR_data(new_list_part);
    if(test_MBR_over(disk_car,new_list_part)==0)
      new_list_part=get_ext_data_i386(disk_car,new_list_part,debug,saveheader);
  }
  get_geometry_from_list_part(disk_car, new_list_part, debug);
  return new_list_part;
}

static void test_MBR_data(t_list_part *list_part)
{
  t_list_part *element;
  unsigned int nb_dos=0, nb_hidden=0, nb_mb=0, nb_ext=0, nb_boot=0;
  wdoprintf(stdscr,msg_PART_HEADER_LONG);
  for(element=list_part;element!=NULL;element=element->next)
  {
    const t_partition *partition=element->part;
    switch(partition->status)
    {
      case STATUS_PRIM_BOOT:
	nb_boot++;
      case STATUS_PRIM:
	switch(partition->part_type)
	{
	  case P_12FAT:
	  case P_16FAT:
	  case P_16FATBD:
	    nb_dos++;
	    break;
	  case P_16FATBDH:
	  case P_16FATH:
	  case P_NTFSH:
	    nb_hidden++;
	    break;
	  case P_OS2MB:
	    nb_mb++;
	    break;
	}
	break;
      case STATUS_EXT:
	nb_ext++;
	break;
      default:
	ecrit_rapport("test_MBR_data: severe error\n");
	break;
    }
  }
  if(nb_dos>1)
    aff_buffer(BUFFER_ADD,msg_ONLY_ONE_DOS);
  if(nb_ext>1)
    aff_buffer(BUFFER_ADD,msg_ONLY_ONE_EXT);
  /* S'il y a des partitions caches, il faut un MB */
  /* Obsolete
  if(nb_hidden>0 && nb_mb==0)
    aff_buffer(BUFFER_ADD,msg_NO_OS2MB);
    */
  /* Nombre de partition bootable */
  if(nb_boot==0)
    aff_buffer(BUFFER_ADD,msg_NO_BOOTABLE);
  else
    if(nb_boot>1)
      aff_buffer(BUFFER_ADD,msg_ONLY1MUSTBOOT);
}

static t_list_part *get_ext_data_i386(t_param_disk *disk_car, t_list_part *list_part, const int debug, const int saveheader)
{
  t_list_part *element;
  int res=0;
  t_partition *partition_main_ext=NULL;
  for(element=list_part;element!=NULL;element=element->next)
  {
    if(element->part->status==STATUS_EXT)
      partition_main_ext=element->part;
  }
  if(partition_main_ext!=NULL)
  {
    t_partition *partition_ext=partition_main_ext;
    t_partition *partition_next_ext=NULL;
    unsigned int i;
    unsigned int order=5;
    do
    {
      unsigned char buffer[DEFAULT_SECTOR_SIZE];
      int nb_hidden=0, nb_mb=0, nb_part=0, nb_ext=0, nb_boot=0;
      if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition_ext->part_offset)!=0)
	return list_part;
      if((buffer[0x1FE]!=(unsigned char)0x55)||(buffer[0x1FF]!=(unsigned char)0xAA))
      {
	aff_buffer(BUFFER_ADD,"\ntest_logical: " msg_TBL2_NMARK);
	return list_part;
      }
      for(i=0;i<4;i++)
      {
	const struct partition_dos *p=pt_offset(buffer,i);
	if(p->boot_ind==(unsigned char)0x80)
	  nb_boot++;
	switch(p->sys_ind)
	{
	  case P_16FATBDH:
	  case P_16FATH:
	  case P_NTFSH:
	    nb_hidden++;
	    break;
	  case P_OS2MB:
	    nb_mb++;
	    break;
	  case P_EXTENDX:
	  case P_EXTENDED:
	  case P_LINUXEXTENDX:
	    nb_ext++;
	    break;
	  case P_NO_OS:
	    break;
	  default:
	    nb_part++;
	}
      }
      if(nb_hidden>0)
	aff_buffer(BUFFER_ADD,"Partition musn't be hidden\n");
      if(nb_mb>0)
	aff_buffer(BUFFER_ADD,"Multiboot must be a primary partition, not a logical\n");
      if(nb_ext>1)
	aff_buffer(BUFFER_ADD,"A logical partition musn't have more than one link to another logical partition\n");
      if(nb_part>1)
	aff_buffer(BUFFER_ADD,"A logical partition must contain only one partition\n");
      if(nb_boot>0)
	aff_buffer(BUFFER_ADD,"Logical partition musn't be bootable\n");
      partition_next_ext=NULL;
      for(i=0;i<4;i++)
      {
	const struct partition_dos *p=pt_offset(buffer,i);
	if(p->sys_ind!=0)
	{
	  t_partition *new_partition=partition_new();
	  new_partition->order=order;
	  if(debug>1)
	    aff_entry_rapport(disk_car,p);
	  if(is_extended(p->sys_ind))
	  {
	    partition_next_ext=new_partition;
	    i386_entry2partition(disk_car, partition_main_ext->part_offset, new_partition, p, STATUS_EXT_IN_EXT,order,debug,saveheader);
	    aff_part_buffer(AFF_PART_ORDER,disk_car,new_partition);
	    if(new_partition->errcode!=BAD_NOERR)
	    {
	      aff_buffer(BUFFER_ADD,"%s\n",errmsg_i386_entry2partition(new_partition->errcode));
	      res=1;
	    }
	    {
	      if((new_partition->part_offset<=partition_main_ext->part_offset) ||
		  (new_partition->part_offset+new_partition->part_size-1 > partition_main_ext->part_offset+partition_main_ext->part_size-1))
	      {	/* Must be IN partition_main_ext */
		res=1;
		aff_buffer(BUFFER_ADD,"Must be in extended partition\n");
		aff_part_buffer(AFF_PART_ORDER,disk_car,partition_main_ext);
		aff_part_buffer(AFF_PART_ORDER,disk_car,new_partition);
	      }
	      else
	      {
		for(element=list_part;element!=NULL;element=element->next)
		{
		  t_partition *partition=element->part;
		  if(partition->status==STATUS_EXT_IN_EXT)
		  {
		    if(((partition->part_offset>=new_partition->part_offset) && (partition->part_offset<=new_partition->part_offset+new_partition->part_size-1)) ||
			((partition->part_offset+partition->part_size-1>=new_partition->part_offset) && (partition->part_offset+partition->part_size-1<=new_partition->part_offset+partition->part_size-1)))
		    { /* New Partition start or end musn't been in partition */
		      res=1;
		      aff_buffer(BUFFER_ADD, "Logical partition must be in its own extended partition\n");
		      aff_part_buffer(AFF_PART_ORDER,disk_car,partition);
		      aff_part_buffer(AFF_PART_ORDER,disk_car,new_partition);
		    }
		  }
		}
	      }
	    }
	  }
	  else
	  {
	    i386_entry2partition(disk_car,partition_ext->part_offset, new_partition, p, STATUS_LOG,order,debug,saveheader);
	    order++;
	    if(debug>1)
	      aff_entry_rapport(disk_car,p);
	    aff_part_buffer(AFF_PART_ORDER,disk_car,new_partition);
	    if(new_partition->errcode!=BAD_NOERR)
	    {
	      aff_buffer(BUFFER_ADD,"%s\n",errmsg_i386_entry2partition(new_partition->errcode));
	      res=1;
	    }
	    {
	      if((new_partition->part_offset<=partition_main_ext->part_offset) ||
		  (new_partition->part_offset+new_partition->part_size-1 > partition_main_ext->part_offset+partition_main_ext->part_size-1))
	      {	/* Must be IN partition_main_ext */
		res=1;
		aff_buffer(BUFFER_ADD, msg_SAME_SPACE);
		aff_part_buffer(AFF_PART_ORDER,disk_car,partition_main_ext);
		aff_part_buffer(AFF_PART_ORDER,disk_car,new_partition);
	      }
	    }
	  }
	  list_part=insert_new_partition(list_part,new_partition);
	}
      }
      partition_ext=partition_next_ext;
    } while ((partition_ext!=NULL) && (order<30));
  }
  return list_part;
}

static int test_MBR_over(t_param_disk *disk_car,t_list_part *list_part)
{/* Tester le overlapping des partitions */
  int res=0;
  t_list_part *element;
  for(element=list_part;element!=NULL;element=element->next)
    if((element->next!=NULL) && (element->part->part_offset + element->part->part_size - 1 >= element->next->part->part_offset))
    {
      res=1;
      aff_buffer(BUFFER_ADD, msg_SAME_SPACE);
      aff_part_buffer(AFF_PART_ORDER,disk_car,element->part);
      aff_part_buffer(AFF_PART_ORDER,disk_car,element->next->part);
    }
  return res;
}

int write_part_i386(t_param_disk *disk_car, const t_list_part *list_part, const int ro, const int debug, const int align)
{
  write_mbr_i386(disk_car,list_part,ro,debug);
  return write_all_log_i386(disk_car,list_part,ro,debug,align);
}

static int write_mbr_i386(t_param_disk *disk_car, const t_list_part *list_part, const int ro, const int debug)
{
  const t_list_part *element;
  unsigned char buffer[disk_car->sector_size];
  unsigned char buffer_org[disk_car->sector_size];
  if(debug>0)
  {
    ecrit_rapport("\nwrite_mbr_i386: starting...\n");
  }
  if(read_MBR(disk_car,&buffer_org))
  {
    memset((void *)&buffer_org,0,DEFAULT_SECTOR_SIZE);
  }
  memset((void *)&buffer,0,DEFAULT_SECTOR_SIZE);
  if((buffer_org[0x1FE]==0x55) && (buffer_org[0x1FF]==0xAA))
  {
    memcpy(buffer,buffer_org,TAB_PART);
    buffer[0x1FE]=0x55;
    buffer[0x1FF]=0xAA;
  } else {
    write_MBR_code_i386(buffer);
  }
  for(element=list_part;element!=NULL;element=element->next)
  {
    switch(element->part->status)
    {
      case STATUS_PRIM:
      case STATUS_PRIM_BOOT:
      case STATUS_EXT:
	if((element->part->order>=1) && (element->part->order<=4))
	{
	  partition2_i386_entry(disk_car,(uint64_t)0,element->part, 
	      pt_offset(buffer,element->part->order-1));
	}
	break;
      case STATUS_LOG:
	break;
      default:
	ecrit_rapport("write_mbr_i386: severe error\n");
	break;
    }
  }
  if(debug>1)
  {
    int i;
    for(i=0;i<4;i++)
    {
      const struct partition_dos *p=pt_offset(buffer,i);
      aff_entry_rapport(disk_car,p);
    }
    diff(disk_car,buffer,buffer_org);
  }
  if(ro==0)
    return write_MBR(disk_car,&buffer);
  return 0;
}

static int write_all_log_i386(t_param_disk *disk_car, const t_list_part *list_part, const int ro, const int debug, const int align)
{
  const t_list_part *element;
  const t_list_part *pos_ext=NULL;
  uint64_t current_pos;
  t_partition *bloc_nextext;
  if(debug)
    ecrit_rapport("write_all_log_i386: starting...\n");
  for(element=list_part;element!=NULL;element=element->next)
  {
    if(element->part->status==STATUS_EXT)
    {
      if(is_extended(element->part->part_type))
      {
	if(pos_ext!=NULL)
	  ecrit_rapport("write_all_log_i386: pos_ext already defined\n");
	pos_ext=element;
      }
      else
      {
	ecrit_rapport("write_all_log_i386: STATUS_EXT with bad part_type\n");
      }
    }
  }
  if(pos_ext==NULL)
  {
    wdoprintf(stdscr,msg_NO_EXT_PART);
    return 1;
  }
  current_pos=pos_ext->part->part_offset;
  bloc_nextext=(t_partition *)MALLOC(sizeof(*bloc_nextext));
  bloc_nextext->part_type=P_EXTENDED;       /* Never P_EXTENDX */
  if(pos_ext->next==NULL || (pos_ext->next->part->status!=STATUS_LOG))
  {
    unsigned char buffer[DEFAULT_SECTOR_SIZE];
    unsigned char buffer_org[DEFAULT_SECTOR_SIZE];
    if(debug>0)
    {
      ecrit_rapport("write_all_log_i386: CHS: %u/%u/%u,lba=%lu\n", offset2cylinder(disk_car,current_pos), offset2head(disk_car,current_pos), offset2sector(disk_car,current_pos),(long unsigned)(current_pos/disk_car->sector_size));
    }
    if(disk_car->read(disk_car,sizeof(buffer_org), &buffer_org, current_pos))
    {
      memset(buffer_org,0,DEFAULT_SECTOR_SIZE);
    }
    memset(buffer,0,DEFAULT_SECTOR_SIZE);
    memcpy(buffer,buffer_org,TAB_PART);
    buffer[0x1FE]=0x55;
    buffer[0x1FF]=0xAA;
    if(ro)
    {
      if(debug>1)
	diff(disk_car,buffer,buffer_org);
    }
    else
    {
      if(disk_car->write(disk_car,sizeof(buffer), &buffer, current_pos))
      {
	ecrit_rapport(msg_PART_WR_ERR);
	display_message(msg_PART_WR_ERR);
      }
    }
  }
  else
  {
    for(element=pos_ext->next;(element!=NULL) && (element->part->status==STATUS_LOG);element=element->next)
    {
      int j;
      unsigned char buffer[DEFAULT_SECTOR_SIZE];
      unsigned char buffer_org[DEFAULT_SECTOR_SIZE];
      if(debug>0)
      {
	ecrit_rapport("write_all_log_i386: CHS: %u/%u/%u,lba=%lu\n", offset2cylinder(disk_car,current_pos), offset2head(disk_car,current_pos), offset2sector(disk_car,current_pos),(long unsigned)(current_pos/disk_car->sector_size));
      }
      if(disk_car->read(disk_car,sizeof(buffer_org), &buffer_org, current_pos))
      {
	memset(buffer_org,0,DEFAULT_SECTOR_SIZE);
      }
      memset(buffer,0,DEFAULT_SECTOR_SIZE);
      memcpy(buffer,buffer_org,TAB_PART);
      buffer[0x1FE]=0x55;
      buffer[0x1FF]=0xAA;
      partition2_i386_entry(disk_car,current_pos,element->part, pt_offset(buffer,0));
      if(element->next!=NULL && (element->next->part->status==STATUS_LOG))
      { /* Construit le pointeur vers la prochaine partition logique */
	t_CHS nextext_start;
	bloc_nextext->part_offset=element->next->part->part_offset-disk_car->sector_size;
	offset2CHS(disk_car,bloc_nextext->part_offset,&nextext_start);
	if(nextext_start.sector!=disk_car->CHS.sector)
	{
	  if(nextext_start.head>0)
	    nextext_start.head--;
	  else
	  {
	    nextext_start.head=disk_car->CHS.head;
	    nextext_start.cylinder--;
	  }
	}
	nextext_start.sector=1;
	if(debug>1)
	  ecrit_rapport("nextext_start %u/%u/%u %lu ? %lu\n", nextext_start.cylinder,nextext_start.head,nextext_start.sector,
	    (long unsigned)(CHS2offset(disk_car,&nextext_start)/disk_car->sector_size),
	    (long unsigned)((element->part->part_offset+element->part->part_size-1)/disk_car->sector_size));
	if(CHS2offset(disk_car,&nextext_start)<=element->part->part_offset+element->part->part_size-1)
	{
	  offset2CHS(disk_car,bloc_nextext->part_offset,&nextext_start);
	  nextext_start.sector=1;
	  if(debug>1)
	    ecrit_rapport("nextext_start %u/%u/%u %lu ? %lu\n", nextext_start.cylinder,nextext_start.head,nextext_start.sector,
	    (long unsigned)(CHS2offset(disk_car,&nextext_start)/disk_car->sector_size),
	    (long unsigned)((element->part->part_offset+element->part->part_size-1)/disk_car->sector_size));
	  if(CHS2offset(disk_car,&nextext_start)<=element->part->part_offset+element->part->part_size-1)
	  {
	    offset2CHS(disk_car,bloc_nextext->part_offset,&nextext_start);
	  }
	}
	if(debug>1)
	  ecrit_rapport("nextext_start %u/%u/%u %lu ? %lu\n", nextext_start.cylinder,nextext_start.head,nextext_start.sector,
	    (long unsigned)(CHS2offset(disk_car,&nextext_start)/disk_car->sector_size),
	    (long unsigned)((element->part->part_offset+element->part->part_size-1)/disk_car->sector_size));
	bloc_nextext->part_offset=CHS2offset(disk_car,&nextext_start);
	/*      ecrit_rapport("table[i]->next=%p table[i+1]=%p\n",table[i]->next,table[i+1]); */
	bloc_nextext->part_size=(uint64_t)element->next->part->part_offset+element->next->part->part_size-bloc_nextext->part_offset;
	partition2_i386_entry(disk_car,pos_ext->part->part_offset,bloc_nextext, pt_offset(buffer,1));
      }
      if(ro)
      {
	if(debug>1)
	{
	  for(j=0;j<4;j++)
	  {
	    const struct partition_dos *p=pt_offset(buffer,j);
	    if(p->sys_ind!=0)
	      aff_entry_rapport(disk_car,p);
	  }
	  diff(disk_car,buffer,buffer_org);
	}
      }
      else
      {
	if(disk_car->write(disk_car,sizeof(buffer), &buffer, current_pos))
	{
	  ecrit_rapport(msg_PART_WR_ERR);
	  display_message(msg_PART_WR_ERR);
	}
      }
      current_pos=bloc_nextext->part_offset;
    }
  }
  FREE(bloc_nextext);
  return 0;
}

static int diff(const t_param_disk *disk_car, const unsigned char buffer[DEFAULT_SECTOR_SIZE], const unsigned char buffer_org[DEFAULT_SECTOR_SIZE])
{
  int j;
  if(memcmp(buffer,buffer_org,DEFAULT_SECTOR_SIZE))
  {
    ecrit_rapport("\nSectors are different.\n");
    ecrit_rapport("buffer_org\n");
    for(j=0;j<4;j++)
    {
      const struct partition_dos *p=pt_offset_const(buffer_org,j);
      if(p->sys_ind!=0)
	aff_entry_rapport(disk_car,p);
    }
    ecrit_rapport("buffer\n");
    for(j=0;j<4;j++)
    {
      const struct partition_dos *p=pt_offset_const(buffer,j);
      if(p->sys_ind!=0)
	aff_entry_rapport(disk_car,p);
    }
    for(j=0;j<DEFAULT_SECTOR_SIZE;j++)
      if(buffer_org[j]!=buffer[j])
	ecrit_rapport("%02X %02X %02X\n", j, buffer_org[j], buffer[j]);
    ecrit_rapport("\n");
  }
  return 0;
}

static int write_MBR_code_i386(unsigned char *buffer)
{
  /* od -t x1 -v ~/Std_mbr.bin */
  const unsigned char mbr_code[DEFAULT_SECTOR_SIZE]={
    0xfa, 0x33, 0xc0, 0x8e, 0xd0, 0xbc, 0x00, 0x7c, 0x8b, 0xf4, 0x50, 0x07, 0x50, 0x1f, 0xfb, 0xfc,
    0xbf, 0x00, 0x06, 0xb9, 0x00, 0x01, 0xf2, 0xa5, 0xea, 0x1d, 0x06, 0x00, 0x00, 0xbe, 0xbe, 0x07,
    0xb3, 0x04, 0x80, 0x3c, 0x80, 0x74, 0x0e, 0x80, 0x3c, 0x00, 0x75, 0x1c, 0x83, 0xc6, 0x10, 0xfe,
    0xcb, 0x75, 0xef, 0xcd, 0x18, 0x8b, 0x14, 0x8b, 0x4c, 0x02, 0x8b, 0xee, 0x83, 0xc6, 0x10, 0xfe,
    0xcb, 0x74, 0x1a, 0x80, 0x3c, 0x00, 0x74, 0xf4, 0xbe, 0x8b, 0x06, 0xac, 0x3c, 0x00, 0x74, 0x0b,
    0x56, 0xbb, 0x07, 0x00, 0xb4, 0x0e, 0xcd, 0x10, 0x5e, 0xeb, 0xf0, 0xeb, 0xfe, 0xbf, 0x05, 0x00,
    0xbb, 0x00, 0x7c, 0xb8, 0x01, 0x02, 0x57, 0xcd, 0x13, 0x5f, 0x73, 0x0c, 0x33, 0xc0, 0xcd, 0x13,
    0x4f, 0x75, 0xed, 0xbe, 0xa3, 0x06, 0xeb, 0xd3, 0xbe, 0xc2, 0x06, 0xbf, 0xfe, 0x7d, 0x81, 0x3d,
    0x55, 0xaa, 0x75, 0xc7, 0x8b, 0xf5, 0xea, 0x00, 0x7c, 0x00, 0x00, 0x49, 0x6e, 0x76, 0x61, 0x6c,
    0x69, 0x64, 0x20, 0x70, 0x61, 0x72, 0x74, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x61, 0x62,
    0x6c, 0x65, 0x00, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x6c, 0x6f, 0x61, 0x64, 0x69, 0x6e, 0x67,
    0x20, 0x6f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65,
    0x6d, 0x00, 0x4d, 0x69, 0x73, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x6f, 0x70, 0x65, 0x72, 0x61, 0x74,
    0x69, 0x6e, 0x67, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0xaa };

    memcpy(buffer,&mbr_code,TAB_PART);
    buffer[0x1FE]=(unsigned char)0x55;
    buffer[0x1FF]=(unsigned char)0xAA;
    return 0;
}

static void partition2_i386_entry(const t_param_disk *disk_car, const uint64_t pos, const t_partition *partition, struct partition_dos *p)
{
  t_CHS start,end;
  offset2CHS(disk_car,partition->part_offset,&start);
  offset2CHS(disk_car,partition->part_offset+partition->part_size-disk_car->sector_size,&end);
  if(partition->status==STATUS_PRIM_BOOT)
    p->boot_ind=0x80;
  else
    p->boot_ind=0;             /* Non bootable */
  p->sys_ind=partition->part_type;
  set_start_sect(p,(partition->part_offset-pos)/disk_car->sector_size);
  if(start.cylinder>1023)
  { /* Partition Magic 5 uses CHS=(1023,0,1) if extended or last logical *
     * Linux fdisk and TestDisk use CHS=(1023,lastH,lastS)               */
    p->head=(unsigned char)disk_car->CHS.head;
    p->sector=(unsigned char)(disk_car->CHS.sector|((1023>>8)<<6));
    p->cyl=(unsigned char)1023;
  }
  else
  {
    p->head=(unsigned char)start.head;
    p->sector=(unsigned char)(start.sector|((start.cylinder>>8)<<6));
    p->cyl=(unsigned char)(start.cylinder);
  }
  if(end.cylinder>1023)
  {
    p->end_head=(unsigned char)disk_car->CHS.head;
    p->end_sector=(unsigned char)(disk_car->CHS.sector|((1023>>8)<<6));
    p->end_cyl=(unsigned char)1023;
  }
  else
  {
    p->end_head=(unsigned char)end.head;
    p->end_sector=(unsigned char)(end.sector|((end.cylinder>>8)<<6));
    p->end_cyl=(unsigned char)end.cylinder;
  }
  set_nr_sects(p,partition->part_size/disk_car->sector_size);
}

static int i386_entry2partition(t_param_disk *disk_car, const uint64_t offset, t_partition *partition, const struct partition_dos *p, const status_type_t status,const unsigned int order,const int debug, const int saveheader)
{
  t_CHS start,end;
  t_CHS start_calculated,end_calculated;
  partition_reset(partition);
  partition->part_type=p->sys_ind;
  partition->part_offset=offset+(uint64_t)get_start_sect(p)*disk_car->sector_size;
  partition->order=order;
  partition->part_size=(uint64_t)get_nr_sects(p)*disk_car->sector_size;

  offset2CHS(disk_car,partition->part_offset,&start_calculated);
  offset2CHS(disk_car,partition->part_offset+partition->part_size-disk_car->sector_size,&end_calculated);


  start.cylinder=s_cyl(p);
  start.head=s_head(p);
  start.sector=s_sect(p);
  end.cylinder=e_cyl(p);
  end.head=e_head(p);
  end.sector=e_sect(p);
  switch(status)
  {
    case STATUS_PRIM:
      if(is_extended(partition->part_type))
      {
	partition->status=STATUS_EXT;
	partition->upart_type=UP_EXTENDED;
      }
      else
	if(p->boot_ind!=0)
	  partition->status=STATUS_PRIM_BOOT;
	else
	  partition->status=status;
      break;
    default:
      partition->status=status;
      break;
  }
  /* Check CHS */
  if((start.sector==0)||(start.sector>disk_car->CHS.sector))
  {
    if(partition->errcode==BAD_NOERR)
      partition->errcode=BAD_SS;
  }
  if((end.sector==0)||(end.sector>disk_car->CHS.sector))
  {
    if(partition->errcode==BAD_NOERR)
      partition->errcode=BAD_ES;
  }
  if(start.head>disk_car->CHS.head)
  {
    if(partition->errcode==BAD_NOERR)
      partition->errcode=BAD_SH;
  }
  if(start.cylinder>disk_car->CHS.cylinder)
  {
    if(partition->errcode==BAD_NOERR)
      partition->errcode=BAD_SC;
  }
  if(end.head>disk_car->CHS.head)
  {
    if(partition->errcode==BAD_NOERR)
      partition->errcode=BAD_EH;
  }
  if(end.cylinder>disk_car->CHS.cylinder)
  {
    if(partition->errcode==BAD_NOERR)
      partition->errcode=BAD_EC;
  }
  if(((start_calculated.cylinder<=1023)&& (C_H_S2offset(disk_car,start.cylinder,start.head,start.sector)!=partition->part_offset))
    || ((start_calculated.cylinder>1023)&&(start.cylinder!=1023)&&(start.cylinder!=(start_calculated.cylinder&1023))))
  {
    ecrit_rapport("BAD_RS LBA=%lu %lu\n",(long unsigned)(partition->part_offset/disk_car->sector_size),C_H_S2LBA(disk_car,start.cylinder,start.head,start.sector));
    if(partition->errcode==BAD_NOERR)
      partition->errcode=BAD_RS;
  }
  if(((end_calculated.cylinder<=1023)&& (C_H_S2offset(disk_car,end.cylinder,end.head,end.sector)!=partition->part_offset+partition->part_size-disk_car->sector_size))
     || ((end_calculated.cylinder>1023)&&(end.cylinder!=1023)&&(end.cylinder!=(end_calculated.cylinder&1023))))
  {
    if(partition->errcode==BAD_NOERR)
      partition->errcode=BAD_SCOUNT;
  }
  /* Check partition and load partition name */
  disk_car->arch->check_part(disk_car,debug,partition,saveheader);
  return 0;
}

static const char* errmsg_i386_entry2partition(const errcode_type_t errcode)
{
  switch(errcode)
  {
    case BAD_SS: return msg_BAD_S_SECT;
    case BAD_ES: return msg_BAD_E_SECT;
    case BAD_SH: return msg_BAD_S_HEAD;
    case BAD_EH: return msg_BAD_E_HEAD;
    case BAD_EBS: return msg_END_BFR_START;
    case BAD_RS: return msg_BAD_RS;
    case BAD_SC: return msg_BAD_S_CYL;
    case BAD_EC: return msg_BAD_E_CYL;
    case BAD_SCOUNT: return msg_BAD_SCOUNT;
    case BAD_NOERR: return "";
  }
  ecrit_rapport("errmsg_i386_entry2partition: unhandled error\n");
  return "";
}

static void aff_entry_rapport(const t_param_disk *disk_car, const struct partition_dos *entree)
{
  if(partition_type(entree->sys_ind,disk_car->arch->parttype_name_table)!=NULL)
    ecrit_rapport(" %-20s ",partition_type(entree->sys_ind,disk_car->arch->parttype_name_table));
  else
    ecrit_rapport(" Sys=%02X               ", entree->sys_ind);
  ecrit_rapport("%4u %3u %2u"
	 " %4u %3u %2u"
	 " %10lu"
	 " %10lu\n",
	  s_cyl(entree),s_head(entree),s_sect(entree),
	  e_cyl(entree),e_head(entree),e_sect(entree),
	  (long unsigned)get_start_sect(entree),(long unsigned)get_nr_sects(entree));
}

static int can_be_ext(const t_param_disk *disk_car, t_partition *partition)
{
  return((offset2head(disk_car,partition->part_offset)>0)&&
      (offset2cylinder(disk_car,partition->part_offset)!=0 ||
       offset2head(disk_car,partition->part_offset)!=1 ||
       offset2sector(disk_car,partition->part_offset)!=1));
}

static int test_structure_i386(t_list_part *list_part)
{ /* Return 1 if bad*/
  int nbr_prim=0, nbr_prim_boot=0, nbr_log_block=0;
  t_list_part *first_log=NULL;
  t_list_part *new_list_part=NULL;
  t_list_part *element;
  t_list_part *new_element;
  int res=0;
  for(element=list_part;element!=NULL;element=element->next)
  {
    switch(element->part->status)
    {
      case STATUS_LOG:
	if(first_log==NULL)
	{
	  first_log=element;
	  nbr_log_block++;
	}
	if(is_extended(element->part->part_type))
	{
          return 1;
	}
	break;
      case STATUS_PRIM_BOOT:
	if(nbr_prim_boot++)
	  return 1;
      case STATUS_PRIM:
	nbr_prim++;
	first_log=NULL;
	break;
      case STATUS_DELETED:
	break;
      default:
	ecrit_rapport("test_structure_i386: severe error\n");
	break;
    }
  }
  if(nbr_log_block>1 || nbr_log_block+nbr_prim>4)
    return 1;
  /* Sort list_part in new_list_part */
  for(element=list_part;element!=NULL;element=element->next)
  {
    if(element->part->status!=STATUS_DELETED)
      new_list_part=insert_new_partition(new_list_part,element->part);
  }
  /* Test overlapping */
  for(element=new_list_part;element!=NULL;element=element->next)
  {
    if(((element->prev!=NULL) && (element->part->part_offset<=element->prev->part->part_offset+element->prev->part->part_size-1)) ||
	((element->next!=NULL) && (element->part->part_offset+element->part->part_size-1>=element->next->part->part_offset)))
    {
      res=1;
    }
  }
  for(element=new_list_part;element!=NULL;element=new_element)
  {
    new_element=element->next;
    FREE(element);
  }
  return res;
}

static int is_extended(const unsigned int part_type)
{
  return (part_type==(const unsigned char)P_EXTENDX || part_type==(const unsigned char)P_EXTENDED || part_type==(const unsigned char)P_LINUXEXTENDX);
}

static t_list_part *add_partition_i386(t_param_disk *disk_car,t_list_part *list_part, const int debug)
{
  t_CHS start,end;
  t_partition *new_partition=partition_new();
  int position=0;
  start.cylinder=0;
  start.head=0;
  start.sector=1;
  end.cylinder=disk_car->CHS.cylinder;
  end.head=disk_car->CHS.head;
  end.sector=disk_car->CHS.sector;
  {
    int done = FALSE;
    while (done==FALSE) {
      int command;
      static struct MenuItem menuGeometry[]=
      {
	{ 'c', "Cylinders", 	"Change starting cylinder" },
	{ 'h', "Heads", 	"Change starting head" },
	{ 's', "Sectors", 	"Change starting sector" },
	{ 'C', "Cylinders", 	"Change ending cylinder" },
	{ 'H', "Heads", 	"Change ending head" },
	{ 'S', "Sectors", 	"Change ending sector" },
	{ 'T' ,"Type",		"Change partition type"},
	{ 'd', "Done", "" },
	{ 0, NULL, NULL }
      };
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      new_partition->part_offset=CHS2offset(disk_car,&start);
      new_partition->part_size=CHS2offset(disk_car,&end) - new_partition->part_offset + disk_car->sector_size;
      wmove(stdscr,10, 0);
      wclrtoeol(stdscr);
      aff_part(stdscr,AFF_PART_SHORT,disk_car,new_partition);
      wmove(stdscr,COMMAND_LINE_Y, COMMAND_LINE_X);
      wclrtoeol(stdscr);
      wrefresh(stdscr);
      command=wmenuSimple(stdscr,menuGeometry, position);
      switch (command) {
	case 'c':
	  start.cylinder=ask_number(start.cylinder,0,disk_car->CHS.cylinder,"Enter the starting cylinder ");
	  position=1;
	  break;
	case 'h':
	  start.head=ask_number(start.head,0,disk_car->CHS.head,"Enter the starting head ");
	  position=2;
	  break;
	case 's':
	  start.sector=ask_number(start.sector,1,disk_car->CHS.sector,"Enter the starting sector ");
	  position=3;
	  break;
	case 'C':
	  end.cylinder=ask_number(end.cylinder,start.cylinder,disk_car->CHS.cylinder,"Enter the ending cylinder ");
	  position=4;
	  break;
	case 'H':
	  end.head=ask_number(end.head,0,disk_car->CHS.head,"Enter the ending head ");
	  position=5;
	  break;
	case 'S':
	  end.sector=ask_number(end.sector,1,disk_car->CHS.sector,"Enter the ending sector ");
	  position=6;
	  break;
	case 'T':
	case 't':
	  change_part_type(disk_car,new_partition);
	  position=7;
	  break;
	case key_ESC:
	case 'd':
	case 'D':
	case 'q':
	case 'Q':
	  done = TRUE;
	  break;
      }
    }
  }
  if((CHS2offset(disk_car,&end)>new_partition->part_offset)&&(new_partition->part_offset>0)&& new_partition->part_type>0)
  {
    t_list_part *new_list_part=insert_new_partition(list_part, new_partition);
    if(test_structure_i386(list_part)==0)
    { /* Determine if the partition can be Logical, Bootable or Primary */
      if(can_be_ext(disk_car,new_partition)!=0)
      {
	new_partition->status=STATUS_LOG;
	if(test_structure_i386(new_list_part)==0)
	  return new_list_part;
      }
      new_partition->status=STATUS_PRIM_BOOT;
      if(test_structure_i386(new_list_part)==0)
	return new_list_part;
      new_partition->status=STATUS_PRIM;
      if(test_structure_i386(new_list_part)==0)
	return new_list_part;
      new_partition->status=STATUS_DELETED;
    }
    return new_list_part;
  }
  FREE(new_partition);
  return list_part;
}

static void set_next_status_i386(const t_param_disk *disk_car, t_partition *partition)
{
  if(partition->status==STATUS_LOG)
    partition->status=STATUS_DELETED;
  else
    partition->status++;
  if(partition->status==STATUS_LOG && can_be_ext(disk_car,partition)==0)
    partition->status=STATUS_DELETED;
}

static void set_prev_status_i386(const t_param_disk *disk_car, t_partition *partition)
{
  if(partition->status==STATUS_DELETED)
    partition->status=STATUS_LOG;
  else
    partition->status--;
  if(partition->status==STATUS_LOG && can_be_ext(disk_car,partition)==0)
    partition->status--;
}

static int is_part_type_ok_i386(unsigned int part_type)
{
  return (part_type>0 && part_type <= 255 && is_extended(part_type)==0);
}

static void init_structure_i386(const t_param_disk *disk_car,t_list_part *list_part, const int debug)
{
  /* Reconstruit une structure 
   * nbr_prim nombre de partition primaire
   * end_log_block : partition etendu du block etendu
   * nbr_log_block : nombre de block de partition etendu
   *                    devrait etre 0 ou 1 */
  unsigned int nbr_prim=0, nbr_log_block=0;
  t_list_part *element;
  t_list_part *end_log_block=NULL;
  t_list_part *end_biggest_log_block=NULL;
  unsigned int log_block_size=0,biggest_log_block_size=0;
  t_list_part *new_list_part=NULL;
  /* Create new list */
  for(element=list_part;element!=NULL;element=element->next)
    element->to_be_removed=0;
  for(element=list_part;element!=NULL;element=element->next)
  {
    int to_be_removed=0;
    t_list_part *element2;
    for(element2=element->next;element2!=NULL;element2=element2->next)
      if(element->part->part_offset+element->part->part_size-1 >= element2->part->part_offset)
      {
	to_be_removed=1;
	element2->to_be_removed=1;
      }
    if(to_be_removed)
      element->to_be_removed=1;
    if(element->to_be_removed==0)
      new_list_part=insert_new_partition(new_list_part,element->part);
  }
  /* Verify */
  for(element=new_list_part;element!=NULL;element=element->next)
  {
    if(can_be_ext(disk_car,element->part)==0)
    {
      nbr_prim++;
      if((end_log_block!=NULL) && (end_log_block->next==element))
      {
	if(log_block_size>biggest_log_block_size)
	{
	  biggest_log_block_size=log_block_size;
	  end_biggest_log_block=end_log_block;
	}
	nbr_log_block++;
	end_log_block=NULL;
      }
    }
    else
    {
      log_block_size++;
      end_log_block=element;
    }
  }
  /* Verification */
  if((end_log_block!=NULL) && (end_log_block->next==NULL))
  {
    if(log_block_size>biggest_log_block_size)
    {
      biggest_log_block_size=log_block_size;
      end_biggest_log_block=end_log_block;
    }
    nbr_log_block++;
  }
  if(debug>1)
    ecrit_rapport("\nRes: nbr_prim %u, nbr_log_block %u\n", nbr_prim, nbr_log_block);
  /* Set primary, extended, logical */
  if(nbr_prim+nbr_log_block<=4)
  {
    int set_prim_bootable_done=0;
    for(element=end_biggest_log_block;element&&can_be_ext(disk_car,element->part);element=element->prev)
    {
/*     ecrit_rapport("%p %p %p %d\n",element,element->prev,element->next,element->part->status); */
      element->part->status=STATUS_LOG;
    }
    for(element=new_list_part;element!=NULL;element=element->next)
    {
      if(element->part->status!=STATUS_LOG)
      {
	if(set_prim_bootable_done)
	  element->part->status=STATUS_PRIM;
	else
	{
	  /* The first primary partition is bootable unless it's a swap */
	  if(element->part->upart_type!=UP_LINSWAP)
	  {
	    element->part->status=STATUS_PRIM_BOOT;
	    set_prim_bootable_done=1;
	  }
	}
      }
    }
  }
#ifdef DEBUG
  check_list_part(new_list_part);
#endif
  if(disk_car->arch->test_structure(new_list_part))
  {
    for(element=new_list_part;element!=NULL;element=element->next)
      element->part->status=STATUS_DELETED;
  }
  {
    /* free */
    t_list_part *new_element;
    for(element=new_list_part;element!=NULL;element=new_element)
    {
      new_element=element->next;
      FREE(element);
    }
  }
#ifdef DEBUG
  check_list_part(list_part);
#endif
}

static void erase_list_part_i386(unsigned char *buffer)
{
  memset(buffer+TAB_PART,0,0x40);
}

static int check_part_i386(t_param_disk *disk_car,const int debug,t_partition *partition, const int saveheader)
{
  int ret=0;
  switch(partition->part_type)
  {
    case P_12FAT:
    case P_16FAT:
    case P_16FATBD:
    case P_32FAT:
    case P_32FAT_LBA:
    case P_16FATBD_LBA:
    case P_12FATH:
    case P_16FATH:
    case P_16FATBDH:
    case P_32FATH:
    case P_32FAT_LBAH:
    case P_16FATBD_LBAH:
      ret=check_FAT(disk_car,partition,debug);
      break;
    case P_LINUX:
      ret=check_JFS(disk_car,partition,debug);
      if(ret!=0)
      {
	ret=check_rfs(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_EXT2(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_cramfs(disk_car,partition,debug);
      }
      if(ret!=0)
      {
	ret=check_xfs(disk_car,partition,debug);
      }
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"No EXT2, JFS, Reiser, cramfs or XFS marker\n"); }
      break;
    case P_NTFS:
    case P_NTFSH:
      ret=check_NTFS(disk_car,partition,debug,0);
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"Invalid NTFS boot\n"); }
      break;
    case P_LVM:
      ret=check_LVM(disk_car,partition,debug);
      break;
    case P_BEOS:
      ret=check_BeFS(disk_car,partition,debug);
      break;
    case P_FREEBSD:
      ret=check_BSD(disk_car,partition,debug,BSD_MAXPARTITIONS);
      break;
    case P_OPENBSD:
      ret=check_BSD(disk_car,partition,debug,OPENBSD_MAXPARTITIONS);
      break;
    case P_NETBSD:
      ret=check_BSD(disk_car,partition,debug,BSD_MAXPARTITIONS);
      break;
    case P_LINSWAP:
      ret=check_Linux_SWAP(disk_car,partition,debug);
      break;
    case P_RAID:
      ret=check_MD(disk_car,partition,debug);
      if(ret!=0)
      { aff_buffer(BUFFER_ADD,"Invalid RAID superblock\n"); }
      break;
    case P_EXTENDED:
    case P_EXTENDX:
    case P_LINUXEXTENDX:
    case P_NETWARE:
      break;
    default:
      if(debug>0)
      {
	ecrit_rapport("check_part_i386 %u type %02X: no test\n",partition->order,partition->part_type);
      }
      break;
  }
  if(ret!=0)
  {
    ecrit_rapport("check_part_i386 failed for partition type %02X\n", partition->part_type);
    aff_part_buffer(AFF_PART_ORDER,disk_car,partition);
    if(saveheader>0)
    {
      save_header(disk_car,partition,debug);
    }
  }
  return ret;
}

