// iofunction.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// An InputOutputFunction takes samples as input, and produces samples
// as output, with input and output usually representing different objects.
// All classes that have a non-unity input-to-output ratio (i.e., they consume
// or produce more samples than they produce or consume) are subclasses of
// this.

#ifndef IOFUNCTION_H
#ifdef __GNUG__
#pragma interface
#endif
#define IOFUNCTION_H

#include "localdefs.h"
#include "outputfunction.h"

#define Q_MORE_INPUT (1)
#define Q_MORE_OUTPUT (2)
#define Q_FLUSH_OUTPUT (4)

// An assisting class that keeps track of pointer locations

class Index {
public:
	Index(int location=0, int channel=0) : loc(location), chan(channel) {}
	void reset() { loc = 0; }
private:
	friend class DataRep;
	int loc;
	int chan;
};

class Data;

class InputOutputFunction : public QueuedOutputFunction {
	typedef QueuedOutputFunction Super;
public:
	InputOutputFunction(Data* input, Data* output,
		int inQSiz=1024, int outQSiz=1024);
	virtual ~InputOutputFunction();
	redefined void restoreState();
	void setInput(Data* newinput);
protected:
	virtual int newLength(int oldLength) { return oldLength; }
	virtual boolean areCompatibleLengths(int, int) { return true; }
	redefined void initialize();
	redefined RunStatus runProcess();
	redefined int doProcessing();
	virtual int processInput(boolean flushing) = 0;
	virtual void readFromInput();
	int addToInQueue(double *values, int count);
	int takeFromInQueue(double *values, int count);
	int inQueueFull() { return inputQueue.full(); }
	int inQueueEmpty() { return inputQueue.empty(); }
	boolean moreInput() { return queueStatus & Q_MORE_INPUT; }
	boolean moreOutput() { return queueStatus & Q_MORE_OUTPUT; }
	boolean needToFlush() { return queueStatus & Q_FLUSH_OUTPUT; }
	void moreInputIf(boolean b);
	void moreOutputIf(boolean b);
	void needToFlushIf(boolean b);
	const Data* getInput() { return myInput; }
private:
	Index& getIndex() { return myIndex; }
	Data* myInput;
	Queue inputQueue;
	Index myIndex;
	unsigned queueStatus;
	friend class DataRep;
};

inline void
InputOutputFunction::moreInputIf(boolean b) {
	if(b) queueStatus |= Q_MORE_INPUT;
	else queueStatus &= ~Q_MORE_INPUT;
}

inline void
InputOutputFunction::moreOutputIf(boolean b) {
	if(b) queueStatus |=  Q_MORE_OUTPUT;
	else queueStatus  &= ~Q_MORE_OUTPUT;
}

inline void
InputOutputFunction::needToFlushIf(boolean b) {
	if(b) queueStatus |= Q_FLUSH_OUTPUT;
	else queueStatus &= ~Q_FLUSH_OUTPUT;
}

// SimpleInputOutputFunction is for the special case when
// the input to output ratio is always one-to-one

class SimpleInputOutputFunction : public InputOutputFunction {
	typedef InputOutputFunction Super;
public:
	SimpleInputOutputFunction(Data* input, Data* output,
		int inQSiz=1024, int outQSiz=1024)
		 : InputOutputFunction(input, output, inQSiz, outQSiz) {}
protected:
	redefined int processInput(boolean flushing);
	virtual int doSimpleProcess(double *buf, int count) = 0;
};
#endif
