#line 2 "file.c"
/*-
 * C-SaCzech
 * Copyright (c) 1996-2002 Jaromir Dolecek <dolecek@ics.muni.cz>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Jaromir Dolecek
 *	for the CSacek project.
 * 4. The name of Jaromir Dolecek may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JAROMIR DOLECEK ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL JAROMIR DOLECEK BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: file.c,v 1.50 2002/02/03 11:13:41 dolecek Exp $ */

#include "csacek.h"

/* Local functions */
static const char *x_mimetype __P((const char *fname));
static int x_file_make_headers __P((csa_params_t *p, const char *filename));

static const struct {
	const char *suffix;
	int suffix_len;
	const char *type;
} x_mimetypes[] =
{
	{ "cz",	  2,	"text/html" },
	{ "html", 4,	"text/html" },
        { "htm",  3,	"text/html" },
        { "txt",  3,	"text/plain" },
        { NULL,	  0,	NULL },
};

/*
 * returns MIME type of filename given as parameter 
 */
static const char * 
x_mimetype( fname )
  const char *fname;
{
   const char *endp, *pp;
   int i;

   if ( fname == NULL ) return 0;

   endp = strchr(fname, '\0'); /* points to the end of ``fname'' */
   for(i=0; x_mimetypes[i].suffix; i++)
   {
	pp = endp - x_mimetypes[i].suffix_len;
	if (pp>fname && *(pp-1)=='.' && !strcasecmp(pp, x_mimetypes[i].suffix)) 
		return x_mimetypes[i].type;
   }

   return "text/plain";
}

/*
 * makes necessary headers out of informations got by stat() and friends
 */
static int 
x_file_make_headers(p, filename)
  csa_params_t *p;
  const char *filename;
{
  struct stat statbuf;
#ifndef CSA_DO_NOT_CACHE
  struct tm *tmi;
  char buf[2048];
#endif

#ifdef CSA_DEBUG
  csa_debug(p->dbg, "x_file_make_headers: called for %s", filename);
#endif /* CSA_DEBUG */

  if (csa_stat(filename, &statbuf) == -1)
	return CSA_FAILED;

  /* it's common practice to mark HTML files, which use SSI, by turning */
  /* execute bit on */
  if (statbuf.st_mode & (S_IXUSR|S_IXGRP|S_IXOTH)) {
#ifdef CSA_DEBUG
	csa_debug(p->dbg, "x_file_make_headers: executable bit set, declining");
#endif
	return CSA_FAILED;
  }

  /* finally, the headers can be sent */
  csa_setheaderout(p, "Status", "200 OK Document follows", CSA_I_OVERWRITE);

  /* csa_get_ct() would output proper Content-Type header, including */
  /* charset specification, if needed and configured so */
  csa_setheaderout(p, "Content-Type",
    csa_get_ct(p->pool_req, p->outcharset, x_mimetype(filename)),
    CSA_I_OVERWRITE);

#ifndef CSA_DO_NOT_CACHE
  tmi = gmtime(&statbuf.st_mtime);
#ifdef HAVE_STRFTIME
  strftime(buf, sizeof(buf), "%a, %d %b %Y %H:%M:%S GMT", tmi );
#else
  sprintf(buf, "%s, %02d %s %4d %02d:%02d:%02d GMT",
	weekdays[tmi->tm_wday],
	tmi->tm_mday, months[tmi->tm_mon], tmi->tm_year + 1900,
	tmi->tm_hour, tmi->tm_min, tmi->tm_sec );
#endif /* HAVE_STRFTIME */
  csa_setheaderout(p, "Last-Modified", buf, CSA_I_COPYVALUE|CSA_I_OVERWRITE);
#endif

  /* set input content_length */
  p->available_in = statbuf.st_size;

#ifdef CSA_DEBUG
  csa_debug(p->dbg, "x_file_make_headers: end");
#endif /* CSA_DEBUG */

  return CSA_OK;
} /* x_file_make_headers */

/* -----------------------------------------------------------------------*/
/* common MD staff */

/*
 * this is called from csa_alloc_fail() when some memory allocation
 * function fails
 */
void
csa_md_alloc_fail()
{
	const char *message = "<TITLE>C-SaCzech error</TITLE><H1>Internal server error: C-SaCzech error</H1>Cannot malloc()ate required memory.\n";

	printf("Status: 500 Internal server error\n");
	printf("Content-Type: text/html\n");
	printf("Content-Length: %lu\n",
		(unsigned long) strlen(message) * sizeof(char));
	printf("\n");
	printf("%s", message);
}

/*
 * return page (generated from external template page), on which it's possible
 * to explicitly choose preferred encoding
 */
int
csa_md_call_whichcode(p, filename)
  csa_params_t *p;
  const char *filename;
{
	/* Not supported */
	return CSA_FATAL;
}

/*
 * returns value of variable ``var'' (one of "standard" CGI variables)
 */
/* ARGSUSED */
const char *
csa_md_getvalueof(p, var)
  csa_params_t *p;
  const char *var;
{
	if (strcmp(var, "PATH_INFO") == 0)
		return "/baz/index.html";
	else if (strcmp(var, "PATH_TRANSLATED") == 0)
		return (const char *) p->m_cookie;
	else if (strcmp(var, "SCRIPT_NAME") == 0)
		return "/toASCII";
	else if (strcmp(var, "SERVER_NAME") == 0)
		return "localhost";
	else if (strcmp(var, "SERVER_PORT") == 0)
		return "80";
	else if (strcmp(var, "SERVER_PROTOCOL") == 0)
		return "HTTP/1.0";	/* avoid chunking */
	else if (strcmp(var, "REMOTE_ADDR") == 0)
		return "localhost";
	else if (strcmp(var, "REQUEST_METHOD") == 0)
		return "GET";

	return (NULL);
}

/*
 * logs an error into server's error log
 * returns 1 if no other output should be written (i.e. the 
 * rest of csa_http_error() should not be executed)
 */
/* ARGSUSED */
int
csa_md_log_error(p, log)
  csa_params_t *p;
  const char *log;
{
	time_t tt;
	char *time_str;

	tt = time(NULL);
	time_str = ctime(&tt);
	time_str[strlen(time_str) - 1] = '\0';	/* strip newline */
	fprintf(stderr, "[%s] %s\n", time_str, log);
	return 0;
}

/*
 * read's len bytes from input (i.e. result of request) and places
 * it in buf
 * returns number of characters written
 */
int
csa_md_read_response(p, buf, len)
  csa_params_t *p;
  char *buf;
  size_t len;
{
	return csa_fread(buf, 1, len, p->resp);
}

/*
 * get all input headers (headers sent by client) and store them
 * in ``p''
 */
int 
csa_md_set_headersin(p)
  csa_params_t *p;
{
	/* Nothing to do */
	return (0);
}

/*
 * separates headers and data; for FILE, just send a newline (headers
 * were sent already by csa_send_headersout() )
 */
/* ARGSUSED */
int
csa_md_send_separator(p)
  csa_params_t *p;
{
	printf("\r\n");
	return 0;
}

/*
 * send given header to output
 */
/* ARGSUSED */
void
csa_md_send_header(p, header_name, header_value)
  csa_params_t *p;
  const char *header_name, *header_value;
{
	printf("%s: %s\r\n", header_name, header_value);
}

/* 
 * sends ``len'' bytes from the place pointed to by ``ptr''
 * into output
 */
/* ARGSUSED */
int
csa_md_send_output(p, ptr, len)
  csa_params_t *p;
  const char *ptr;
  size_t len;
{
	size_t retval;
	retval = fwrite(ptr, 1, len, stdout);
	return (len == retval);
}

/* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- */

/*
 * main
 */
int 
main( argc, argv)
  int argc;
  const char *argv[];
{
   csa_params_t params, *p = &params;
   int csa_retval=CSA_OK;
   struct pool *main_pool;
   FILE *dbg=NULL;

   if (argc < 2) {
	fprintf(stderr, "%s: incorrect number of parameters\n",
		argv[0]);
	fprintf(stderr, "Usage: %s FILE_NAME\n", argv[0]);
	return 1;
  }
  
  /* time need to be expressed in GMT */
#ifdef HAVE_SETENV
   setenv("TZ", "", 1);
#elif defined(HAVE_PUTENV)
   putenv("TZ=");
#endif /* HAVE_SETENV || HAVE_PUTENV */
   tzset();

   main_pool = ap_make_sub_pool(NULL);

   /* initialize global list of CSacek servers */
   csa_cs_slist_init(main_pool);

#ifdef CSA_DEBUG
   dbg = csa_debug_start();
#endif

   memset((void *)p, '\0', sizeof(csa_params_t));

   csa_retval = csa_init_params(p, main_pool, (void *) argv[1], NULL, dbg);
   if (csa_retval == CSA_OK) {
	p->resp = fopen(argv[1], "r");
	if (!p->resp) {
		fprintf(stderr, "%s: cannot open '%s': %s\n",
			argv[0], argv[1], strerror(errno));
		return 2;
	}

	CSA_SET(p->flags, CSA_FL_CONVERT);

	if (strcmp(x_mimetype(argv[1]), "text/html") == 0)
		CSA_SET(p->flags, CSA_FL_ISHTML);

	/* make necessary headers */
	(void) x_file_make_headers(p, argv[1]);

   	(void) csa_process_body(p);
	csa_output(p);

	/* fflush(stdout); */
   }

#ifdef CSA_DEBUG
    /* close debug log */
    csa_debug_end(p->dbg);
#endif

    return (0);
} /* main */
