
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/time.h>
#include <stdint.h>
#include <stdlib.h>

#include <abz/error.h>
#include <debug/memory.h>

#include "stats.h"

/*
 * Initialize stats structure. Returns 0 if successful, -1 if
 * some error occurred. You can check which error occurred by
 * calling abz_get_error().
 */
void stats_create (struct stats *stats,time_t period)
{
   stats->avail = 0;
   stats->total = period + 1;
   stats->sample = NULL;
}

static void destroy (struct sample **sample)
{
   if (*sample != NULL)
	 {
		struct sample *head = *sample,*tmp;

		while ((tmp = (*sample)->tail) != head)
		  {
			 mem_free (*sample);
			 *sample = tmp;
		  }

		mem_free (*sample);
		*sample = NULL;
	 }
}

/*
 * Destroy stats structure.
 */
void stats_destroy (struct stats *stats)
{
   stats->avail = 0;
   destroy (&stats->sample);
}

static inline int add_to_tail (struct sample **tail,const struct sample *sample)
{
   struct sample *tmp;

   if ((tmp = mem_alloc (sizeof (struct sample))) == NULL)
	 {
		abz_set_error ("out of memory");
		return (-1);
	 }

   *tmp = *sample;

   if (*tail != NULL)
	 {
		(*tail)->head->tail = tmp;
		tmp->head = (*tail)->head;
		(*tail)->head = tmp;
		tmp->tail = *tail;
	 }
   else *tail = tmp->head = tmp->tail = tmp;

   return (0);
}

static inline void remove_from_head (struct sample **head)
{
   if (*head != NULL)
	 {
		struct sample *tmp = *head;

		if (*head != (*head)->tail)
		  {
			 (*head)->tail->head = (*head)->head;
			 (*head)->head->tail = (*head)->tail;
			 *head = (*head)->tail;
		  }
		else *head = NULL;

		mem_free (tmp);
	 }
}

/*
 * Update stats counters. Returns 0 if successful, -1 if some
 * occurred. You can check which error occurred by calling
 * abz_get_error().
 */
int stats_update (struct stats *stats,const struct sample *sample)
{
   abz_clear_error ();

   if (stats->sample == NULL || time (NULL) != stats->sample->head->timestamp)
	 {
		if (add_to_tail (&stats->sample,sample) < 0)
		  return (-1);

		if (stats->avail == stats->total)
		  remove_from_head (&stats->sample);
		else
		  stats->avail++;
	 }

   return (0);
}

/*
 * Calculate average rate over period (or less if there are
 * not enough samples available) and store the result in sample.
 * Return 0 if successful, -1 if no samples are available.
 */
int stats_calc_rate (struct rate *avg,const struct stats *stats,time_t period)
{
   struct sample *tmp,*head,*tail;
   uint64_t input = 0,output = 0;
   double elapsed;

   abz_clear_error ();

   if (stats->avail < 2)
	 {
		fail:
		abz_set_error ("not enough samples available");
		return (-1);
	 }

   for (head = stats->sample, tmp = tail = stats->sample->head;
		(elapsed = difftime (tail->timestamp,tmp->timestamp)) < period && tmp != head;
		tmp = tmp->head)
	 {
		input += tmp->input - tmp->head->input;
		output += tmp->output - tmp->head->output;
	 }

   if (!elapsed) goto fail;

   avg->input = input / elapsed;
   avg->output = output / elapsed;

   return (0);
}

