/*
 *   (C) Copyright IBM Corp. 2001, 2005
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: linear_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Linear MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "md.h"
#include "linear_mgr.h"

#define my_plugin_record linear_plugin

static boolean linear_can_change_region_configuration( storage_object_t *region )
{
	boolean rc = TRUE;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	if (volume->region_mgr_flags & MD_LINEAR_CONFIG_CHANGE_PENDING) {
		rc = FALSE;
	}
	
	LOG_EXIT_BOOL(rc);
	return rc;
}

/*
 * Look for this object in the list, return TRUE if found
 */
static boolean linear_find_object_in_list( list_anchor_t list, storage_object_t *obj )
{
	if (EngFncs->find_in_list(list, obj, NULL, NULL))
		return TRUE;
	else
		return FALSE;
}

static int linear_add_new_disk(md_volume_t *vol, storage_object_t *new_disk)
{
	int rc = 0;
	md_member_t *member = NULL;

	LOG_ENTRY();
	
	member = md_allocate_member(new_disk);
	if (!member) {
		rc = ENOMEM;
		goto out;
	}

	member->data_size = md_object_usable_size(new_disk, &vol->sb_ver, 0);
	member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
	rc = md_volume_add_new_member(vol, member);
	if (rc) {
		goto out_error_free;
	}

	md_append_region_to_object(vol->region, new_disk);
out_error_free:
	if (rc && member) {
		md_free_member(member);
	}
out:
	LOG_EXIT_INT(rc);
	return rc;
}


static int linear_remove_last_disk(md_volume_t *vol, storage_object_t *child, boolean kill_sects)
{

	int rc = 0;
	md_member_t *member;
	list_element_t li;

	LOG_ENTRY();
	member = EngFncs->last_thing(vol->members, &li);
	if (member->obj == child) {

		rc = md_volume_remove_member(member, TRUE);
		if (!rc) {
			md_remove_region_from_object(vol->region, child);

			/* If we are unwinding expansion, don't clear the superblock */
			if (kill_sects) {
				vol->sb_func->zero_superblock(member, FALSE);
			}
			md_free_member(member);
		}

	} else {
		LOG_ERROR(" %s is not the last disk of the %s region\n",
			  child->name, vol->region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_setup_evms_plugin
 *
 *	This function gets called shortly after the plugin is loaded by the
 *	Engine. It performs all tasks that are necessary before the initial
 *	discovery pass.
 */
static int linear_setup_evms_plugin(engine_functions_t	* functions)
{
	int rc = 0;

	// Parameter check
	if (!functions) {
		return EINVAL;
	}

	EngFncs = functions;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Region Checking Functions ******/


/* All of the following md_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: linear_can_delete
 *
 *	Can we remove the specified MD logical volume, and consolidate the
 *	space with the freespace volume?
 */
static int linear_can_delete( storage_object_t * region )
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/* Function: linear_can_last_child_expand
 *
 * Notes: This function returns error code not boolean
 */
static int linear_can_last_child_expand(
	storage_object_t *region,
	u_int64_t        expand_limit,
	list_anchor_t    expansion_points )
{
	int rc = 0;
	md_member_t *member;
	list_element_t li;
	storage_object_t *child;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	
	LOG_ENTRY();
	if (vol->nr_disks < 1) {
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}
	member = EngFncs->last_thing(vol->members, &li);
	child = member->obj;
	if (!child) {
		LOG_MD_BUG();
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}
	rc = child->plugin->functions.plugin->can_expand(child, expand_limit, expansion_points);
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_can_last_child_shrink
 *
 * Notes: This function returns error code not boolean
 */
static int linear_can_last_child_shrink(
	storage_object_t *region,
	u_int64_t        shrink_limit,
	list_anchor_t    shrink_points )
{
	int rc = 0;
	md_member_t *member;
	list_element_t li;
	storage_object_t *child;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	
	LOG_ENTRY();
	if (vol->nr_disks < 1) {
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}
	member = EngFncs->last_thing(vol->members, &li);
	child = member->obj;
	if (!child) {
		LOG_MD_BUG();
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}
	rc = child->plugin->functions.plugin->can_shrink(child, shrink_limit, shrink_points);
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_can_expand
 *
 * This region can be expanded if:
 *	The last child can be expanded -OR-
 *	There is at least 1 available object
 */
static int linear_can_expand(storage_object_t *region,
			     u_int64_t expand_limit,
			     list_anchor_t expansion_points)
{
	int rc = 0;
	uint count;
	list_anchor_t tmp_list;
	list_element_t li=NULL;
	expand_object_info_t * expand_object;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (region->flags & SOFLAG_CORRUPT) {
		LOG_EXIT_INT(EPERM);
		return EPERM;
	}

	if (!linear_can_change_region_configuration(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* Allow the last child to add expansion point */
	linear_can_last_child_expand(region, expand_limit, expansion_points);

	/* get a list of all valid input disks, segments, and regions. */
	EngFncs->get_object_list(DISK | SEGMENT | REGION,
				DATA_TYPE,
				NULL,
				region->disk_group,
				VALID_INPUT_OBJECT | NO_DISK_GROUP,
				&tmp_list);
	
	/* Remove this MD object from the list */
	EngFncs->remove_thing(tmp_list, region);

	/* Remove all parents of this MD region from list */
	remove_parent_regions_from_list(tmp_list, region);

	count = EngFncs->list_count(tmp_list);

	EngFncs->destroy_list(tmp_list);

	if (count) {
		expand_object = (expand_object_info_t *) EngFncs->engine_alloc( sizeof(expand_object_info_t) );
		if (expand_object) {
			expand_object->object          = region;
			expand_object->max_expand_size = -1;

			li = EngFncs->insert_thing(expansion_points,
						   expand_object,
						   INSERT_AFTER,
						   NULL);
			if (!li) {
				EngFncs->engine_free( expand_object );
				rc = ENOMEM;
			}
		} else {
			rc = ENOMEM;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_can_expand_by
 *
 *	Can we expand the specified region, by the specified size, using space
 *	from the freespace volume in the same group?
 */

static int linear_can_expand_by(storage_object_t	* child_object,
				u_int64_t		* size )
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: linear_can_shrink
 *
 * This region can be expanded if:
 *	The last child can be shrunk -OR-
 *	This region has more than 1 disk
 */
static int linear_can_shrink(storage_object_t *region,
			     u_int64_t shrink_limit,
			     list_anchor_t shrink_points )
{
	int rc = 0;
	list_element_t li = NULL;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	shrink_object_info_t * shrink_object;
	md_member_t *first;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	/*
	 * Note: In the future, we may consider shrinking a corrupt linear region.
	 * Especially the case of the missing child(ren) is the last one.
	 */
	if (region->flags & SOFLAG_CORRUPT) {
		LOG_EXIT_INT(EPERM);
		return EPERM;
	}

	if (!linear_can_change_region_configuration(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* Allow the last child to add shrink point */
	linear_can_last_child_shrink(region, shrink_limit, shrink_points);

	if (vol->nr_disks > 1) {
		first = EngFncs->first_thing(vol->members, &li);
		shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
		if (shrink_object) {
			shrink_object->object          = region;
			shrink_object->max_shrink_size = region->size - first->data_size;
			li = EngFncs->insert_thing(shrink_points,
						   shrink_object,
						   INSERT_AFTER,
						   NULL);
			if (!li) {
				EngFncs->engine_free( shrink_object );
				rc = ENOMEM;
			}
		} else {
			rc = ENOMEM;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_can_shrink_by
 *
 *	Can we shrink the specified logical volume and consolidate the removed
 *	space into the freespace volume? If it can be shrunk, but not to
 *	exactly new_size, reset new_size to the next lower possible size.
 */
static int linear_can_shrink_by(storage_object_t	* region,
				u_int64_t		* size )
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


static int linear_can_replace_child(storage_object_t *region,
				    storage_object_t *child,
				    storage_object_t *new_child)
{
	int rc;
	my_plugin = linear_plugin;
	LOG_ENTRY();
	rc = md_can_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


static int linear_compare_targets(md_volume_t * vol)
{
	dm_target_t *target, *targets = NULL;
	dm_device_t *dev;
	u_int64_t offset = 0;
	md_member_t *member;
	list_element_t iter;
	int rc;

	LOG_ENTRY();

	rc = EngFncs->dm_get_targets(vol->region, &targets);
	if (rc) {
		goto out;
	}

	target = targets;
	LIST_FOR_EACH(vol->members, iter, member) {
		if (!target) {
			rc = EINVAL;
			break;
		}
		if ( (target->start != offset) ||
		     (target->length != member->data_size) ||
		     (target->type != DM_TARGET_LINEAR) ) {
			rc = EINVAL;
			break;
		}
		dev = target->data.linear;
		if ( (dev->major != member->obj->dev_major) ||
		     (dev->minor != member->obj->dev_minor) ||
		     (dev->start != member->data_offset) ) {
			rc = EINVAL;
			break;
		}
		target = target->next;
		offset += member->data_size;
	}

	if (!rc) {
		if (target) {
			rc = EINVAL;
		}
	}

out:
	EngFncs->dm_deallocate_targets(targets);
	LOG_EXIT_INT(rc);		 	
	return rc;
}

static int linear_create_region(md_volume_t * vol, list_anchor_t output_list, boolean final_call)
{
	int rc = 0;
	storage_object_t * region = NULL;
	md_member_t *member;
	int i;
	int length;
	mdu_array_info_t info;

	LOG_ENTRY();

	if (!vol->sb && !final_call) {
		LOG_WARNING("Volume %s does not have superblock, delaying discovery.\n",
			    vol->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if ((vol->nr_disks != vol->raid_disks) && !final_call) {
       		LOG_DETAILS("Region %s. missing %d members, delaying discovery.\n",
			    vol->name, vol->raid_disks - vol->nr_disks);
		LOG_EXIT_INT(0);
		return 0;
	}

	rc = EngFncs->allocate_region(vol->name, &region);
	if (rc) {
		LOG_WARNING("Region %s is already created, try new name.\n", vol->name);
		rc = md_volume_get_alternative_name(vol, 255);
		if (!rc) {
			LOG_WARNING("Trying new region name: %s...\n", vol->name);
			rc = EngFncs->allocate_region(vol->name, &region);
			if (!rc) {
				LOG_WARNING("OK. got it.\n");
			} else {
				LOG_CRITICAL("Give up.\n");
			}
		} else {
			LOG_CRITICAL("The MD region name space has been exausted.\n");
			LOG_EXIT_INT(rc);
			return rc;
		}
	}

	region->data_type = DATA_TYPE;
	region->plugin = linear_plugin;
	region->private_data = (void *)vol;
	vol->flags |= MD_DISCOVERED;
	vol->region = region;

	md_analyze_volume(vol);

	for (i=0; i<vol->raid_disks; i++) {
		member = md_volume_find_member(vol, i);
		if (member) {
			md_append_region_to_object(region, member->obj);
		} else {
			length = sprintf(message_buffer, _("  The disk indexed %d is missing.\n"), i);
			md_queue_corrupt_message(vol->personality, message_buffer, length);
			vol->flags |= MD_CORRUPT;
		}
	}

	region->size = md_volume_calc_size(vol);

	if (vol->flags & MD_CORRUPT) {
		goto out;
	}

	/*
	 * Query device-mapper for the status of this MD object.
	 * if this MD object is active, it's already activated as
	 * an DM device.  Otherwise, check with the MD kernel driver.
	 */
	rc = EngFncs->dm_update_status(region);
	if (!rc && (region->flags & SOFLAG_ACTIVE)) {
		rc = linear_compare_targets(vol);
		if (rc) {
			region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
		} else {
			LOG_DEBUG("Region %s is an active DM device (%d:%d)\n",
				  region->name, region->dev_major, region->dev_minor);
		}
	} else {
		rc = 0;
		region->dev_major = MD_MAJOR;
		region->dev_minor = vol->md_minor;
		md_get_kernel_info(region, &info);
	}

out:
	if (vol->flags & MD_CORRUPT) {
		region->flags |= SOFLAG_CORRUPT;
	}

	if (region) {
		md_add_object_to_list(region, output_list);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int linear_discover_regions( list_anchor_t output_list, int *count, boolean final_call )
{
	int rc = 0;
	md_volume_t * volume = volume_list_head;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	while (volume != NULL) {
		if ((!(volume->flags & MD_DISCOVERED)) && (volume->personality == LINEAR)) {
			rc = linear_create_region(volume, output_list, final_call);
			if (volume->flags & MD_DISCOVERED) {
				*count = *count + 1;
			}
		}
		volume = volume->next;
	}
	if (final_call) {
		md_display_corrupt_messages(LINEAR);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_discover
 *
 *	Examine all disk segments and find MD PVs. Assemble volume groups
 *	and export all MD logical volumes as EVMS regions.
 *
 *	All newly created regions must be added to the output list, and all
 *	segments from the input list must either be claimed or moved to the
 *	output list.
 */
static int linear_discover(list_anchor_t	input_list,
           		   list_anchor_t	output_list,
			   boolean	final_call )
{
	int count = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!input_list || !output_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (final_call) {
		md_discover_final_call(input_list, output_list, &count);
	} else {
		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("Object discovery complete.\n");

		// LV discovery and exporting
		linear_discover_regions(output_list, &count, final_call);
		LOG_DETAILS("Object creation complete.\n");
	}

	LOG_EXIT_INT(count);
	return count;
}



/****** Region Functions ******/

static int linear_get_create_options( option_array_t *options, md_sb_ver_t *sb_ver)
{
	int i;
	int rc = 0;
	boolean version_1 = FALSE;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			case LINEAR_CREATE_OPT_SB1_INDEX:
				version_1 = options->option[i].value.b;
				break;
			
			default:
				break;
			}

		} else {

			if (strcmp(options->option[i].name, LINEAR_CREATE_OPT_SB1_NAME) == 0){
				version_1 = options->option[i].value.b;
			}
		}
	}

	if (version_1 == TRUE) {
		sb_ver->major_version = MD_SB_VER_1;
		sb_ver->minor_version = 0;
		sb_ver->patchlevel = 0;
	} else {
		sb_ver->major_version = MD_SB_VER_0;
		sb_ver->minor_version = 90;
		sb_ver->patchlevel = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_create_new_region(md_volume_t * vol, list_anchor_t output_list)
{
	int rc=0;
	md_member_t *member;
	list_element_t iter;
	storage_object_t *region;

	LOG_ENTRY();
	rc = EngFncs->allocate_region(vol->name, &region);
	if (rc) {
		LOG_ERROR("Region %s is already created (rc=%d).\n",
			  vol->name, rc);
	}

	LOG_DEBUG("Creating new region %s: nr_disks=%d, raid_disks=%d, spares=%d, actives=%d, working=%d\n",
		  vol->name, vol->nr_disks, vol->raid_disks, vol->spare_disks, vol->active_disks, vol->working_disks);
	
	if (!rc) {
		LIST_FOR_EACH(vol->members, iter, member) {
			md_append_region_to_object(region, member->obj);
		}
		region->size = md_volume_calc_size(vol);
		region->data_type = DATA_TYPE;
		region->plugin = linear_plugin;
		region->private_data = (void *)vol;
		region->dev_major = MD_MAJOR;
		region->dev_minor = vol->md_minor;
		vol->region = region;
		region->flags |= SOFLAG_DIRTY;
		md_add_object_to_list(region, output_list);
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_create
 *
 *	Create a new MD volume.
 */
static int linear_create(list_anchor_t objects, 
			 option_array_t	*options, 
			 list_anchor_t	new_region_list )
{
	int rc = 0;
	md_volume_t *volume = NULL;
	storage_object_t * object;
	list_element_t iter1;
	list_element_t iter2;
	md_member_t *member;
	u_int64_t size=-1;
	md_sb_ver_t sb_ver = {MD_SB_VER_0, 90, 0};

	my_plugin = linear_plugin;
	LOG_ENTRY();

	// Must have at least 1 disk
	if (EngFncs->list_count(objects) < 1) {
		LOG_CRITICAL("Must have at least 1 object.\n");
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!(volume = md_allocate_volume())) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}
	
	rc = md_volume_get_available_name(volume, 256);
	if (rc) {
		goto error_free;
	}

	linear_get_create_options(options, &sb_ver);

	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		size = min(size, md_object_usable_size(object, &sb_ver, 0));
	}
	
	rc = md_init_sb(volume, &sb_ver, MD_LEVEL_LINEAR, 0, size, 0);

	if (rc) {
		goto error_free;
	}
	
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		member = md_allocate_member(object);
		if (member) {
			// This will add the member and update the MD superblock.
			member->data_size = md_object_usable_size(object, &sb_ver, 0);
			member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
			rc = md_volume_add_new_member(volume, member);
			if (rc) {
				md_free_member(member);
				goto error_free;
			}
		} else {
			rc = ENOMEM;
		}
		if (rc) {
			goto error_free;
		}
		EngFncs->delete_element(iter1);
	}


	rc = linear_create_new_region(volume, new_region_list);
	if (rc) {
		goto error_free;
	} else {
		volume->flags |= MD_DIRTY;
	}
	LOG_EXIT_INT(rc);
	return rc;

error_free:
	md_free_volume(volume);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: w_delete
 *
 * Worker function for linear_delete and linear_discard
 */
static int w_delete(storage_object_t *region, list_anchor_t	children, boolean tear_down)
{
	md_volume_t	* volume;
	int		rc = 0;

	LOG_ENTRY();

	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume, tear_down);
	region->private_data = NULL;
	EngFncs->free_region(region);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_delete
 *
 *	Remove the specified region
 */
static int linear_delete(storage_object_t *region, list_anchor_t children)
{
	int		rc = 0;

	LOG_ENTRY();
	rc = w_delete(region, children, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: linear_discard
 *
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the regions.
 */
static int linear_discard(list_anchor_t regions)
{
	storage_object_t * region;
	list_element_t le;

	LOG_ENTRY();

	LIST_FOR_EACH(regions, le, region) {
		w_delete(region, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/* Function: linear_expand_last_child
 *
 * Call expand() on the last child of the MD linear array to expand.
 */
static int linear_expand_last_child(storage_object_t	*region,
				    storage_object_t	*expand_object,
				    list_anchor_t		input_objects,
				    option_array_t	*options )
{
	int rc=0;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	storage_object_t *child;
	u_int64_t cur_size;
	md_member_t *last;
	list_element_t li;

	LOG_ENTRY();
	last = EngFncs->last_thing(vol->members, &li);
	child = last->obj;
	if (child != expand_object) {
		LOG_ERROR(" Error, expand obj (%s) is not the last child!\n",
			  expand_object->name);
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	cur_size = child->size;
	LOG_DEBUG(" %s's current size = %"PRIu64" sectors.\n",
		  child->name, child->size);
	vol->sb_func->zero_superblock(last, FALSE);
	rc = child->plugin->functions.plugin->expand(child, expand_object, input_objects, options);
	if (!rc) {
		LOG_DEBUG(" %s's new size = %"PRIu64" sectors.\n",
			  child->name, child->size);
		if (child->size > cur_size) {
			last->data_size = md_object_usable_size(child, &vol->sb_ver, 0);
			vol->flags |= MD_NEEDS_UPDATE_SIZE;
			md_volume_calc_size(vol);
			vol->flags |= MD_DIRTY;
			region->flags |= SOFLAG_DIRTY;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_shrink_last_child
 *
 * Call shrink() on the last child of the MD linear array to shrink.
 */
static int linear_shrink_last_child(storage_object_t	*region,
				    storage_object_t	*shrink_object,
				    list_anchor_t		input_objects,
                                    option_array_t	*options )
{
	int rc=0;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	storage_object_t *child;
	u_int64_t cur_size;
	md_member_t *last;
	list_element_t li;

	LOG_ENTRY();
	last = EngFncs->last_thing(vol->members, &li);
	child = last->obj;
	if (child != shrink_object) {
		LOG_ERROR(" Error, shrink obj (%s) is not the last child!\n",
			  shrink_object->name);
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	cur_size = child->size;
	LOG_DEBUG(" %s's current size = %"PRIu64" sectors.\n",
		  child->name, child->size);
	vol->sb_func->zero_superblock(last, FALSE);
	rc = child->plugin->functions.plugin->shrink(child, shrink_object, input_objects, options);
	if (!rc) {
		LOG_DEBUG(" %s's new size = %"PRIu64" sectors.\n", child->name, child->size);
		if (child->size < cur_size) {
			last->data_size = md_object_usable_size(child, &vol->sb_ver, 0);
			vol->flags |= MD_NEEDS_UPDATE_SIZE;
			md_volume_calc_size(vol);
			vol->flags |= MD_DIRTY;
			region->flags |= SOFLAG_DIRTY;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Function: linear_expand
 *
 * If the last child is the target for this call,
 * let it handle the expansion.  Otherwise, add one disk at a time.
 */

static int linear_expand(storage_object_t	*region,
			 storage_object_t	*expand_object,
			 list_anchor_t       	input_objects,
			 option_array_t		*options )
{
	int rc = 0;
	md_volume_t *vol;
	int saved_nr_disks;
	storage_object_t *candidate;
	md_member_t *last;
	list_element_t li;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	vol = (md_volume_t *)region->private_data;

	if (region != expand_object) {
		rc = linear_expand_last_child(region, expand_object, input_objects, options);
	} else {

		/* We're about to perform the expansion */
		saved_nr_disks = vol->nr_disks; /* just in case... */
		LIST_FOR_EACH(input_objects, li, candidate) {
			rc = linear_add_new_disk(vol, candidate);
			if (rc) {
				break;
			}
		}

		if (!rc) {
			/* Expansion was successful */
			vol->flags |= MD_DIRTY;
			region->flags |= SOFLAG_DIRTY;
		} else {
			/* There was an error, unwind the expand */
			while (vol->nr_disks > saved_nr_disks) {
				last = EngFncs->last_thing(vol->members, &li);
				linear_remove_last_disk(vol, last->obj, FALSE);
			}
		}
	}

	vol->flags |= MD_NEEDS_UPDATE_SIZE;
	region->size = md_volume_calc_size(vol);

	if ((!rc) && (region->flags & SOFLAG_ACTIVE)) {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_shrink
 */
static int linear_shrink(storage_object_t	* region,
			storage_object_t	* shrink_object,
			list_anchor_t		input_objects,
			option_array_t		* options )
{
	int rc = 0;
	md_volume_t *vol;
	storage_object_t *candidate;
	int removed_count, selected_count;
	list_anchor_t removed_list;
	md_member_t *last;
	list_element_t li = NULL;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (region != shrink_object) {
		rc = linear_shrink_last_child(region, shrink_object, input_objects, options);
	} else {
		vol = (md_volume_t *)region->private_data;
		selected_count = EngFncs->list_count(input_objects);
		removed_count = 0;
		removed_list = EngFncs->allocate_list();
		if (!removed_list) {
			LOG_ERROR("Can't allocate removed list.\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		while (!rc && (removed_count < selected_count)) {
			last = EngFncs->last_thing(vol->members, &li);
			candidate = last->obj;
			if (linear_find_object_in_list(input_objects, candidate)) {
				rc = linear_remove_last_disk(vol, candidate, TRUE);
				if (!rc) {
					removed_count++;
					li = EngFncs->insert_thing(removed_list,
								   candidate,
								   INSERT_AFTER,
								   NULL);
					if (!li) {
						rc = ENOMEM;
						LOG_ERROR("Could not insert to removed list.\n");
					}
				}
			} else {
				LOG_ERROR("%s does not exist in selected object list.\n",
					  candidate->name);
				rc = EINVAL;
				break;
			}
		}

		if (!rc && (removed_count == selected_count)) {
			/* The shrink was successful */
			vol->flags |= MD_DIRTY;
			region->flags |= SOFLAG_DIRTY;
		} else {

			/* There was an error, unwind the shrink */
			LIST_FOR_EACH(removed_list, li, candidate) {
				rc = linear_add_new_disk(vol, candidate);
				if (rc) {
					LOG_ERROR("Could not re-add %s back to %s region.\n",
						  candidate->name, vol->name);
					break;
				}
			}
		}
		EngFncs->destroy_list(removed_list);

		vol->flags |= MD_NEEDS_UPDATE_SIZE;
		region->size = md_volume_calc_size(vol);
	}

	if ((!rc) && (region->flags & SOFLAG_ACTIVE)) {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_replace_child(storage_object_t *region,
				storage_object_t *child,
				storage_object_t *new_child)
{
	int rc;
	my_plugin = linear_plugin;
	LOG_ENTRY();
	rc = md_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_add_sectors_to_kill_list
 *
 *	The kill_sectors list contains a list of sectors that need to be zeroed
 *	during the next commit. This function is very similar to read/write.
 */
static int linear_add_sectors_to_kill_list( storage_object_t	* region,
					lsn_t			lsn,
					sector_count_t		count )
{

	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_kill = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *vol = (md_volume_t *)region->private_data;
	int rc = 0;
	md_member_t *member;
	list_element_t iter;

	my_plugin = linear_plugin;
	LOG_ENTRY();


	if (vol->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, data is suspect \n",
			  vol->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	LIST_FOR_EACH(vol->members, iter, member) {

		current_end += member->data_size;
                if ( current_end >= io_lsn ) {

                    max_sector_count = current_end - io_lsn + 1;

                    if ( max_sector_count >= sectors_left_to_kill ) {
                        io_sector_count = sectors_left_to_kill;
                    }
                    else {
                        io_sector_count = max_sector_count;
                    }

                    link_lsn = io_lsn - current_start;

                    rc = KILL_SECTORS(member->obj, link_lsn, io_sector_count);

                    io_lsn               += io_sector_count;
                    sectors_left_to_kill -= io_sector_count;

                    if ((sectors_left_to_kill == 0) || (rc)) break;
                }
		current_start = current_end;
        }
	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_commit_changes( storage_object_t * region, uint phase )
{
	md_volume_t	* volume;
	int		rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (!region ||  !(volume = (md_volume_t *)region->private_data)) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if ( ! (region->flags & SOFLAG_DIRTY) ) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	switch (phase) {
	case FIRST_METADATA_WRITE:
		volume->flags |= MD_DIRTY;
		volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
		rc = md_write_sbs_to_disk(volume);   // write super blocks
		region->flags &= ~SOFLAG_DIRTY;  // mark clean
		volume->flags &= ~MD_NEW_REGION;
		break;
	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_can_activate_region(storage_object_t * region)
{
	md_volume_t   * volume = (md_volume_t *)region->private_data;
	my_plugin = raid0_plugin;
	LOG_ENTRY();

	if (volume->flags & MD_CORRUPT) {
		LOG_WARNING("MD region %s is corrupt.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}

static int linear_activate_region(storage_object_t * region)
{
	int rc = 0;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	dm_target_t *targets = NULL, *target=NULL;
	dm_device_t *dev;
	u_int64_t offset = 0;
	md_member_t *member;
	list_element_t iter;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (vol->flags & MD_CORRUPT) {
		LOG_WARNING("%s is corrupt.\n", vol->name);
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj) {
			target = EngFncs->dm_allocate_target(
				DM_TARGET_LINEAR, offset, member->data_size, 0, 0);
			offset += member->data_size;
			if (target) {
				dev = target->data.linear;
				dev->major = member->obj->dev_major;
				dev->minor = member->obj->dev_minor;
				dev->start = member->data_offset;
				EngFncs->dm_add_target(target, &targets);
			} else {
				rc = ENOMEM;
			}
			if (rc) {
				break;
			}
		}
	}

	if (!rc) {
		rc = EngFncs->dm_activate(region, targets);
		if (!rc) {
			region->flags &= ~SOFLAG_NEEDS_ACTIVATE;
			LOG_DEBUG("Region %s has been activated, DM device(%d, %d)\n",
				  region->name, region->dev_major, region->dev_minor);
		}
	}

	if (targets) EngFncs->dm_deallocate_targets(targets);

	LOG_EXIT_INT(rc);
	return rc;

}

static int linear_can_deactivate_region(storage_object_t * region)
{
	LOG_ENTRY();
	
	LOG_EXIT_INT(0);
	return 0;
}

static int linear_deactivate_region(storage_object_t * region)
{
	int rc;
	mdu_array_info_t info;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	rc = md_ioctl_get_array_info(region, &info);
	if (rc == 0) {
		/*
		 * This MD array was started by another tool.
		 * Stop the array via ioctl to the kernel MD driver.
		 */
		 rc = md_deactivate_region(region);
	} else {
		rc = EngFncs->dm_deactivate(region);
		if (!rc) {
			region->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_commit_changes
 *
 */


/* Function: linear_get_option_count
 *
 *	Determine the type of Task that is being performed, and return
 *	the number of options that are available for that Task.
 */
static int linear_get_option_count( task_context_t * task )
{
	int count = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	switch(task->action) {
	case EVMS_Task_Create:
		count = LINEAR_CREATE_OPT_COUNT;
		break;
	case EVMS_Task_Expand:
		count = 0;
		break;
	case EVMS_Task_Shrink:
		count = 0;
		break;
	
	default:
		count = 0;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}


/* Function: linear_init_task
 *
 *	Determine the type of Task that is being performed, and set up the
 *	context structure with the appropriate initial values.
 */
static int linear_init_task( task_context_t * context )
{
	int rc = 0;
	list_anchor_t tmp_list;
	md_volume_t * vol;
	md_member_t *first;
	md_member_t *member;
	list_element_t li;
	list_element_t iter;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	switch(context->action) {

	case EVMS_Task_Create:

		context->option_descriptors->count = LINEAR_CREATE_OPT_COUNT;

		// Version 1 Superblock Option
		if (md_can_create_sb_1() == TRUE) {
			context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].flags = 0;
			context->min_selected_objects = 1;
			context->max_selected_objects = MD_SB_1_DISKS;
		} else {
			context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
			context->min_selected_objects = 1;
			context->max_selected_objects = MD_SB_DISKS;
		}
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].constraint.list = NULL;
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].help = NULL;
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].name =
			EngFncs->engine_strdup( LINEAR_CREATE_OPT_SB1_NAME );
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].tip =
			EngFncs->engine_strdup( _("Choose Yes if you want to create MD version 1 super block.") );
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].title = EngFncs->engine_strdup( _("Version 1 Super Block") );
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].type = EVMS_Type_Boolean;
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].unit = EVMS_Unit_None;
		context->option_descriptors->option[LINEAR_CREATE_OPT_SB1_INDEX].value.b = FALSE;
		
		// get a list of all valid input disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					DATA_TYPE,
					NULL,
					NULL,
					VALID_INPUT_OBJECT,
					&tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);
		break;

	case EVMS_Task_Expand:
		
		vol = (md_volume_t *) context->object->private_data;
		context->option_descriptors->count = 0;

		/* get a list of all valid input disks, segments, and regions. */
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					DATA_TYPE,
					NULL,
					context->object->disk_group,
					VALID_INPUT_OBJECT | NO_DISK_GROUP,
					&tmp_list);

		/* Remove this MD object from the list */
		EngFncs->remove_thing(tmp_list, context->object);
		
		/* Remove all parents of this MD region from acceptable list */
		remove_parent_regions_from_list(tmp_list, context->object);
		
		/* move these items to the acceptable objects list. */
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);
		context->min_selected_objects = 1;
		context->max_selected_objects = MAX_DISKS(vol) - vol->nr_disks;
		break;

	case EVMS_Task_Shrink:
		
		vol = (md_volume_t *) context->object->private_data;
		context->option_descriptors->count = 0;

		if (vol->nr_disks <= 1) {
			rc = EINVAL;
			break;
		}

		EngFncs->delete_all_elements(context->acceptable_objects);

		first = EngFncs->first_thing(vol->members, &li);
		LIST_FOR_EACH(vol->members, iter, member) {
			if (member != first) {
				li = EngFncs->insert_thing(context->acceptable_objects,
							   member->obj,
							   INSERT_BEFORE,
							   NULL);
				if (!li) {
					LOG_ERROR("Could not insert %s into acceptable object list\n",
						  member->obj->name);
					rc = ENOMEM;
					break;
				}
			}
		}
		context->min_selected_objects = 1;
		context->max_selected_objects = vol->nr_disks - 1;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_set_option
 *
 *	Determine the type of Task that is being performed. Then examine the
 *	desired option (using the index), and verify that the given value is
 *	appropriate. Reset the value if necessary and possible. Adjust other
 *	options as appropriate.
 */
static int linear_set_option(task_context_t	* context,
			u_int32_t		index,
			value_t			* value,
			task_effect_t		* effect )
{
	int rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	switch (context->action) {

	case EVMS_Task_Create:
		switch (index) {
		case LINEAR_CREATE_OPT_SB1_INDEX:
			context->option_descriptors->option[index].value.b = value->b;
			if (value->b == TRUE) {
				context->max_selected_objects = MD_SB_1_DISKS;
			} else {
				context->max_selected_objects = MD_SB_DISKS;
			}
			break;

		default:
			break;
                }
		break;
	
	case EVMS_Task_Expand:
		// no options, just return 0
		break;
	
	case EVMS_Task_Shrink:
		// no options, just return 0
		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int linear_set_expand_object( task_context_t * context,
				     list_anchor_t declined_objects,
				     task_effect_t  * effect )
{
	int                      rc=0;
	list_element_t		li=NULL;
	list_element_t		li_declined=NULL;
	storage_object_t        *obj=NULL;
	declined_object_t       *declined_object=NULL;
	int                      selected_objects_count=0;
	int                      declined_object_count=0;
	int                      max_objects_allowed=0;
	md_volume_t *vol = (md_volume_t *)context->object->private_data;


	LOG_ENTRY();

	/* determine how many disks we can add */
	if (context) {
		max_objects_allowed = MAX_DISKS(vol) - vol->nr_disks;
		if (!max_objects_allowed) {
			LOG_EXIT_INT(EOVERFLOW);
			return EOVERFLOW;
		}
	} else {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LIST_FOR_EACH(context->selected_objects, li, obj) {

		if ( max_objects_allowed > selected_objects_count ) {

			++selected_objects_count;
			*effect |=  EVMS_Effect_Reload_Options;

		} else {

			LOG_WARNING("Overflow : declining object (%s)\n", obj->name);

			++declined_object_count;

			declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

			if (declined_object) {

				declined_object->object = obj;
				declined_object->reason = EOVERFLOW;

				li_declined = EngFncs->insert_thing(declined_objects,
								    declined_object,
								    INSERT_AFTER,
								    NULL);
				if (li_declined) {
					*effect |=  EVMS_Effect_Reload_Objects;
				} else {
					EngFncs->engine_free(declined_object);
					LOG_ERROR("Could not insert declined object into declined object list\n");
					rc = ENOMEM;
				}

			} else {
				LOG_ERROR("Could not allocate memory for a declined object.\n");
				rc = ENOMEM;
			}

		}
		if (rc) {
			break;
		}
	}


	if (declined_object_count) {
		*effect |= EVMS_Effect_Reload_Objects;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int linear_set_shrink_object( task_context_t * context,
				     list_anchor_t declined_objects,
				     task_effect_t  * effect )
{
	int rc=0;
	int rc2;
	list_anchor_t my_list = NULL;
	list_anchor_t decline_list = NULL;
	list_anchor_t reversed_list = NULL;
	list_element_t iter;
	storage_object_t *obj;
	declined_object_t *declined_obj;
	u_int64_t shrink_size;
	md_member_t *member;
	md_volume_t *vol = (md_volume_t *)context->object->private_data;


	LOG_ENTRY();

	decline_list = EngFncs->allocate_list();
	reversed_list = EngFncs->allocate_list();
	my_list = EngFncs->allocate_list();
	if (!decline_list || !my_list || !reversed_list) {
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Build reversed list */
	LIST_FOR_EACH(vol->members, iter, member) {
		EngFncs->insert_thing(reversed_list, member, INSERT_BEFORE, NULL);
	}

	/*
	 * Transfer all selected objects to local list (my_list).
	 * If we can shrink, we will transfer the objects back to selected list.
	 * If we cannot shrink, the selected object list will be empty.
	 */
	md_transfer_list(context->selected_objects, my_list);

	shrink_size = 0;
	LIST_FOR_EACH(reversed_list, iter, member) {
		obj = member->obj;
		if (linear_find_object_in_list(my_list, obj)) {
			shrink_size += MD_NEW_SIZE_SECTORS(obj->size);
			rc2 = EngFncs->can_shrink_by(context->object, &shrink_size);
			if (!rc2) {
				/* Move this object back to selected list */
				EngFncs->remove_thing(my_list, obj);
				EngFncs->insert_thing(context->selected_objects, obj, INSERT_AFTER, NULL);
			} else {
				/* Decline the remaining objects. */
				md_transfer_list(my_list, decline_list);
				break;
			}
		} else {
			/*
			 * The selection list is out of order.
			 * Decline the remaining objects.
			 */
			md_transfer_list(my_list, decline_list);
			break;
		}
	}

	/* Decline objects in decline list */
	LIST_FOR_EACH(decline_list, iter, obj) {
		declined_obj = EngFncs->engine_alloc(sizeof(declined_object_t));
		if (!declined_obj) {
			rc = ENOMEM; /* Set error code, but continue looping */
		} else {
			declined_obj->object = obj;
			declined_obj->reason = EINVAL;
			EngFncs->insert_thing(declined_objects, declined_obj, INSERT_AFTER, NULL);
		}
	}

	EngFncs->destroy_list(decline_list);
	EngFncs->destroy_list(reversed_list);
	EngFncs->destroy_list(my_list);
	

	*effect |= EVMS_Effect_Reload_Objects;

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_set_objects
 *
 * Validate the objects in the selected_objects dlist in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects dlist.
 * Modify the accepatble_objects dlist in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
static int linear_set_objects(	task_context_t	* context,
				list_anchor_t		declined_objects,
				task_effect_t	* effect )
{
	int rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	switch(context->action) {

	case EVMS_Task_Create:
		// since this is within task context, and we provided the initial list
		// do we need any further validation here?
		rc = 0;
		break;
	
	case EVMS_Task_Expand:
		rc = linear_set_expand_object( context, declined_objects, effect );
		break;
	
	case EVMS_Task_Shrink:
		rc = linear_set_shrink_object( context, declined_objects, effect );
		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_get_info
 *
 *	Return MD-specific information about the specified region. If the
 *	name field is set, only return the "extra" information pertaining
 *	to that name.
 */
static int linear_get_info(	storage_object_t	* region,
				char			* name,
				extended_info_array_t	** info_array )
{
	md_volume_t * volume = NULL;
	int           rc= 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

        LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_get_plugin_info
 *
 *	Return information about the MD plugin. There is no "extra"
 *	information about MD, so "name" should always be NULL.
 */
static int linear_get_plugin_info(char			* name,
				  extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buffer[50] = {0};
	int			i = 0;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();

	// Parameter check
	if ( ! info_array ) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if ( ! name ) {
		// Get memory for the info array
		if ( ! (info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*6)) ) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		// Short Name
		info->info[i].name = EngFncs->engine_strdup("ShortName");
		info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(linear_plugin->short_name);
		i++;

		// Long Name
		info->info[i].name = EngFncs->engine_strdup("LongName");
		info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(linear_plugin->long_name);
		i++;

		// Plugin Type
		info->info[i].name = EngFncs->engine_strdup("Type");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
		info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("Region Manager"));
		i++;

		// Plugin Version
		info->info[i].name = EngFncs->engine_strdup("Version");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		// Required Engine Services Version
		info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
							      "It will not run on older versions of the Engine services."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", linear_plugin->required_engine_api_version.major, linear_plugin->required_engine_api_version.minor, linear_plugin->required_engine_api_version.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		// Required Plug-in API Version
		info->info[i].name = EngFncs->engine_strdup("Required_Plugin_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Plug-in API Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API that this plug-in requires.  "
							      "It will not run on older versions of the Engine plug-in API."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", linear_plugin->required_plugin_api_version.plugin.major, linear_plugin->required_plugin_api_version.plugin.minor, linear_plugin->required_plugin_api_version.plugin.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

	}
	else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	info->count = i;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: linear_read
 *
 *	Perform a logical-to-physical remapping, and send the read down to
 *	the next plugin.
 */
static int linear_read(	storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	char                      *io_buffer_ptr = (char *)buffer;
	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_read = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *vol = (md_volume_t *)region->private_data;
	md_member_t *member;
	list_element_t iter;
	int rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	// to be in sync with the kernel MD driver we are not supporting
	// partial exports at this time.  Use EVMS Drive linking instead.
	if (vol->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n",
			  vol->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	rc = md_region_rw(region, lsn, count, buffer, 0);
	if (rc) {
		rc = 0;
		LIST_FOR_EACH(vol->members, iter, member) {
			current_end += member->data_size;
			if ( current_end >= io_lsn ) {
	
			    max_sector_count = current_end - io_lsn + 1;
	
			    if ( max_sector_count >= sectors_left_to_read ) {
				io_sector_count = sectors_left_to_read;
			    }
			    else {
				io_sector_count = max_sector_count;
			    }
	
			    link_lsn = (io_lsn - current_start);
	
			    rc = READ(member->obj, link_lsn + member->data_offset, io_sector_count, (void *)io_buffer_ptr );
	
			    io_lsn               += io_sector_count;
			    io_buffer_ptr        += io_sector_count*EVMS_VSECTOR_SIZE;
			    sectors_left_to_read -= io_sector_count;
	
			    if ((sectors_left_to_read == 0) || (rc)) break;
			}
			current_start = current_end;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_write
 *
 *	Perform a logical-to-physical remapping, and send the write down to
 *	the next plugin.
 */
static int linear_write(storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	char                      *io_buffer_ptr = (char *)buffer;
	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_write = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *vol = (md_volume_t *)region->private_data;
	md_member_t *member;
	list_element_t iter;
	int rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (vol->flags & MD_CORRUPT) {
		MESSAGE(_("MD region %s is corrupt.  Writing data is not allowed.\n"),
			vol->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	rc = md_region_rw(region, lsn, count, buffer, 1);
	if (rc) {
		rc = 0;
		LIST_FOR_EACH(vol->members, iter, member) {
			current_end += member->data_size;
			if ( current_end >= io_lsn ) {

			    max_sector_count = current_end - io_lsn + 1;

			    if ( max_sector_count >= sectors_left_to_write ) {
				io_sector_count = sectors_left_to_write;
			    }
			    else {
				io_sector_count = max_sector_count;
			    }

			    link_lsn = (io_lsn - current_start);

			    rc = WRITE(member->obj, link_lsn + member->data_offset, io_sector_count, (void *)io_buffer_ptr );

			    io_lsn               += io_sector_count;
			    io_buffer_ptr        += io_sector_count*EVMS_VSECTOR_SIZE;
			    sectors_left_to_write -= io_sector_count;

			    if ((sectors_left_to_write == 0) || (rc)) break;
			}
			current_start = current_end;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_get_plugin_functions(storage_object_t *region, function_info_array_t * * functions)
{
	my_plugin = linear_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

static int linear_plugin_function(storage_object_t * object,
				 task_action_t      action,
				 list_anchor_t            objects,
				 option_array_t   * options)
{
	my_plugin = linear_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/*
 * linear_backup_metadata
 *
 * Called to write metadata backup.
 */
int linear_backup_metadata(storage_object_t *region)
{
	md_volume_t *volume;
	int rc=0;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();


	volume = region->private_data;
	if (volume->flags & MD_CORRUPT) {
		rc = ENOSYS;
		goto out;
	}
	volume->commit_flag |= MD_COMMIT_BACKUP_METADATA;
	volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
	volume->flags |= MD_DIRTY;
	rc = md_write_sbs_to_disk(volume); 
	volume->commit_flag &= ~MD_COMMIT_BACKUP_METADATA;
	volume->commit_flag &= ~MD_COMMIT_DONT_CHECK_ACTIVE;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

static void linear_plugin_cleanup(void) {

	list_anchor_t regions_list;
	list_element_t li;
	storage_object_t * region;
	md_volume_t * volume;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	EngFncs->get_object_list(REGION, DATA_TYPE, linear_plugin, NULL, 0, &regions_list);

	LIST_FOR_EACH(regions_list, li, region) {
		volume = (md_volume_t *)region->private_data;
		md_free_volume(volume);
	}

	EngFncs->destroy_list(regions_list);

	LOG_EXIT_VOID();
	return;
}


/* Function tables for the MD Region Manager */
static plugin_functions_t linear_functions = {
	setup_evms_plugin		: linear_setup_evms_plugin,
	cleanup_evms_plugin		: linear_plugin_cleanup,
	can_delete			: linear_can_delete,
	can_expand			: linear_can_expand,
	can_expand_by			: linear_can_expand_by,
	can_shrink			: linear_can_shrink,
	can_shrink_by			: linear_can_shrink_by,
	can_replace_child		: linear_can_replace_child,
	discover			: linear_discover,
	create				: linear_create,
	delete				: linear_delete,
	discard				: linear_discard,
	expand				: linear_expand,
	shrink				: linear_shrink,
	replace_child			: linear_replace_child,
	add_sectors_to_kill_list	: linear_add_sectors_to_kill_list,
	commit_changes			: linear_commit_changes,
	activate			: linear_activate_region,
	can_activate			: linear_can_activate_region,
	can_deactivate			: linear_can_deactivate_region,
	deactivate			: linear_deactivate_region,
	get_option_count		: linear_get_option_count,
	init_task			: linear_init_task,
	set_option			: linear_set_option,
	set_objects			: linear_set_objects,
	get_info			: linear_get_info,
	get_plugin_info			: linear_get_plugin_info,
	read				: linear_read,
	write				: linear_write,
	get_plugin_functions		: linear_get_plugin_functions,
	plugin_function			: linear_plugin_function,
	backup_metadata                 : linear_backup_metadata
};



/* Function: PluginInit
 *
 *	Initializes the local plugin record
 */

plugin_record_t linear_plugin_record = {
    id:                    SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 4),

    version:              {major:      MAJOR_VERSION,
                           minor:      MINOR_VERSION,
                           patchlevel: PATCH_LEVEL},

    required_engine_api_version: {major:      15,
				  minor:      0,
				  patchlevel: 0},
    required_plugin_api_version: {plugin: {major:      13,
					   minor:      0,
					   patchlevel: 0} },

    short_name:            "MDLinearRegMgr",
    long_name:             "MD Linear Raid Region Manager",
    oem_name:              "IBM",

    functions:             {plugin: &linear_functions},

    container_functions:   NULL
};
