#![cfg(test)]

use http::header::{CONTENT_TYPE, LINK};
use miette::IntoDiagnostic;
use url::Url;

use crate::{
    http::CONTENT_TYPE_JSON,
    standards::indieauth::{Scopes, ServerMetadata},
};

use super::{ClientId, CodeChallenge};

#[tokio::test]
async fn obtain_metadata_none_found() -> miette::Result<()> {
    let mut server = mockito::Server::new_async().await;
    let html_remote_page_mock = server
        .mock("GET", "/profile")
        .expect_at_most(1)
        .create_async()
        .await;

    let headers_remote_page_mock = server
        .mock("HEAD", "/profile")
        .expect_at_most(1)
        .create_async()
        .await;

    let client = super::Client::new(
        "http://example.com",
        crate::http::reqwest::Client::default(),
    )?;

    let remote_url = format!("{}/profile", server.url()).parse().unwrap();
    let resp = client.obtain_metadata(&remote_url).await;

    assert_eq!(
        resp,
        Err(super::Error::NoMetadataEndpoint.into()),
        "expected no metadata anywhere due to lack of an endpoint"
    );

    headers_remote_page_mock.assert_async().await;
    html_remote_page_mock.assert_async().await;
    Ok(())
}

#[tokio::test]
async fn obtain_metadata_via_individual_endpoints() -> miette::Result<()> {
    let mut server = mockito::Server::new_async().await;
    let server_url = server.url();

    let html_remote_page_mock = server
        .mock("GET", "/profile")
        .expect_at_most(1)
        .with_body(format!(
            r#"
<html>
    <head>
        <link rel="authorization_endpoint" href="{server_url}/auth" />
        <link rel="token_endpoint" href="{server_url}/token" />
    </head>
</html>
            "#
        ))
        .create_async()
        .await;

    let headers_remote_page_mock = server
        .mock("HEAD", "/profile")
        .expect_at_most(1)
        .create_async()
        .await;

    let client = super::Client::new(
        "http://example.com",
        crate::http::reqwest::Client::default(),
    )?;

    let remote_url = format!("{}/profile", server.url()).parse().unwrap();
    let resp = client.obtain_metadata(&remote_url).await;

    headers_remote_page_mock.assert_async().await;
    html_remote_page_mock.assert_async().await;

    let metadata = resp?;
    assert_eq!(metadata.issuer, server.url().parse().unwrap());
    assert_eq!(
        metadata.authorization_endpoint,
        format!("{}/auth", server.url()).parse().unwrap()
    );
    assert_eq!(
        metadata.token_endpoint,
        format!("{}/token", server.url()).parse().unwrap()
    );
    assert_eq!(metadata.scopes_supported, super::Scopes::default());
    assert_eq!(
        metadata.code_challenge_methods_supported,
        super::ServerMetadata::recommended_code_challenge_methods()
    );

    Ok(())
}

#[tokio::test]
async fn obtain_metadata_via_individual_endpoints_headers() -> miette::Result<()> {
    let mut server = mockito::Server::new_async().await;
    let server_url = server.url();

    let html_remote_page_mock = server
        .mock("GET", "/profile")
        .expect_at_most(0)
        .create_async()
        .await;

    let headers_remote_page_mock = server
        .mock("HEAD", "/profile")
        .with_header(
            LINK,
            &format!(
                r#"<{}/auth>; rel="authorization_endpoint""#,
                server.url()
            ),
        )
        .with_header(
            LINK,
            &format!(r#"<{}/token>; rel="token_endpoint""#, server.url()),
        )
        .expect_at_most(1)
        .create_async()
        .await;

    let client = super::Client::new(
        "http://example.com",
        crate::http::reqwest::Client::default(),
    )?;

    let remote_url = format!("{}/profile", server.url()).parse().unwrap();
    let resp = client.obtain_metadata(&remote_url).await;

    headers_remote_page_mock.assert_async().await;
    html_remote_page_mock.assert_async().await;

    let metadata = resp?;
    assert_eq!(metadata.issuer, server.url().parse().unwrap());
    assert_eq!(
        metadata.authorization_endpoint,
        format!("{}/auth", server.url()).parse().unwrap()
    );
    assert_eq!(
        metadata.token_endpoint,
        format!("{}/token", server.url()).parse().unwrap()
    );

    Ok(())
}

#[tokio::test]
async fn obtain_metadata_individual_endpoints_missing_auth() -> miette::Result<()> {
    let mut server = mockito::Server::new_async().await;
    let server_url = server.url();

    let html_remote_page_mock = server
        .mock("GET", "/profile")
        .expect_at_most(1)
        .with_body(format!(
            r#"
<html>
    <head>
        <link rel="token_endpoint" href="{server_url}/token" />
    </head>
</html>
            "#
        ))
        .create_async()
        .await;

    let headers_remote_page_mock = server
        .mock("HEAD", "/profile")
        .expect_at_most(1)
        .create_async()
        .await;

    let client = super::Client::new(
        "http://example.com",
        crate::http::reqwest::Client::default(),
    )?;

    let remote_url = format!("{}/profile", server.url()).parse().unwrap();
    let resp = client.obtain_metadata(&remote_url).await;

    headers_remote_page_mock.assert_async().await;
    html_remote_page_mock.assert_async().await;

    assert_eq!(
        resp,
        Err(super::Error::NoMetadataEndpoint.into()),
        "expected no metadata when authorization_endpoint is missing"
    );

    Ok(())
}

#[tokio::test]
async fn obtain_metadata_individual_endpoints_missing_token() -> miette::Result<()> {
    let mut server = mockito::Server::new_async().await;
    let server_url = server.url();

    let html_remote_page_mock = server
        .mock("GET", "/profile")
        .expect_at_most(1)
        .with_body(format!(
            r#"
<html>
    <head>
        <link rel="authorization_endpoint" href="{server_url}/auth" />
    </head>
</html>
            "#
        ))
        .create_async()
        .await;

    let headers_remote_page_mock = server
        .mock("HEAD", "/profile")
        .expect_at_most(1)
        .create_async()
        .await;

    let client = super::Client::new(
        "http://example.com",
        crate::http::reqwest::Client::default(),
    )?;

    let remote_url = format!("{}/profile", server.url()).parse().unwrap();
    let resp = client.obtain_metadata(&remote_url).await;

    headers_remote_page_mock.assert_async().await;
    html_remote_page_mock.assert_async().await;

    assert_eq!(
        resp,
        Err(super::Error::NoMetadataEndpoint.into()),
        "expected no metadata when token_endpoint is missing"
    );

    Ok(())
}

#[tokio::test]
async fn obtain_metadata_individual_endpoints_relative_urls() -> miette::Result<()> {
    let mut server = mockito::Server::new_async().await;
    let server_url = server.url();

    let html_remote_page_mock = server
        .mock("GET", "/profile")
        .expect_at_most(1)
        .with_body(
            r#"
<html>
    <head>
        <link rel="authorization_endpoint" href="auth" />
        <link rel="token_endpoint" href="token" />
    </head>
</html>
            "#
            .to_string(),
        )
        .create_async()
        .await;

    let headers_remote_page_mock = server
        .mock("HEAD", "/profile")
        .expect_at_most(1)
        .create_async()
        .await;

    let client = super::Client::new(
        "http://example.com",
        crate::http::reqwest::Client::default(),
    )?;

    let remote_url = format!("{}/profile", server.url()).parse().unwrap();
    let resp = client.obtain_metadata(&remote_url).await;

    headers_remote_page_mock.assert_async().await;
    html_remote_page_mock.assert_async().await;

    let metadata = resp?;
    assert_eq!(metadata.issuer, server.url().parse().unwrap());
    assert_eq!(
        metadata.authorization_endpoint,
        format!("{}/auth", server.url()).parse().unwrap()
    );
    assert_eq!(
        metadata.token_endpoint,
        format!("{}/token", server.url()).parse().unwrap()
    );

    Ok(())
}

#[tokio::test]
async fn obtain_metadata_fallback_priority() -> miette::Result<()> {
    let mut server = mockito::Server::new_async().await;
    let server_url = server.url();
    let metadata = super::ServerMetadata {
        issuer: server.url().parse().unwrap(),
        authorization_endpoint: format!("{server_url}/endpoints/auth").parse().unwrap(),
        token_endpoint: format!("{server_url}/endpoints/token").parse().unwrap(),
        ticket_endpoint: None,
        introspection_endpoint: None,
        code_challenge_methods_supported: super::ServerMetadata::recommended_code_challenge_methods(),
        scopes_supported: super::Scopes::minimal(),
    };

    let html_remote_page_mock = server
        .mock("GET", "/profile")
        .expect_at_most(1)
        .with_body(format!(
            r#"
<html>
    <head>
        <link rel="indieauth-metadata" href="{server_url}/metadata" />
        <link rel="authorization_endpoint" href="{server_url}/auth" />
        <link rel="token_endpoint" href="{server_url}/token" />
    </head>
</html>
            "#
        ))
        .create_async()
        .await;

    let headers_remote_page_mock = server
        .mock("HEAD", "/profile")
        .expect_at_most(1)
        .create_async()
        .await;

    let metadata_endpoint_mock = server
        .mock("GET", "/metadata")
        .with_header(CONTENT_TYPE, CONTENT_TYPE_JSON)
        .with_body(serde_json::json!(metadata).to_string())
        .expect_at_most(1)
        .create_async()
        .await;

    let client = super::Client::new(
        "http://example.com",
        crate::http::reqwest::Client::default(),
    )?;

    let remote_url = format!("{}/profile", server.url()).parse().unwrap();
    let resp = client.obtain_metadata(&remote_url).await;

    headers_remote_page_mock.assert_async().await;
    metadata_endpoint_mock.assert_async().await;
    html_remote_page_mock.assert_async().await;

    // Should use metadata endpoint, not individual endpoints
    assert_eq!(resp, Ok(metadata));

    Ok(())
}


#[tokio::test]
async fn obtain_metadata_via_endpoint_headers() -> miette::Result<()> {
    let mut server = mockito::Server::new_async().await;
    let server_url = server.url();
    let var_name = ServerMetadata {
        issuer: server.url().parse().unwrap(),
        authorization_endpoint: format!("{server_url}/endpoints/auth").parse().unwrap(),
        token_endpoint: format!("{server_url}/endpoints/token").parse().unwrap(),
        ticket_endpoint: format!("{server_url}/endpoints/ticket").parse().ok(),
        introspection_endpoint: format!("{server_url}/endpoints/token").parse().ok(),
        code_challenge_methods_supported: ServerMetadata::recommended_code_challenge_methods(),
        scopes_supported: Scopes::minimal(),
    };
    let metadata = var_name;
    let html_remote_page_mock = server
        .mock("GET", "/profile")
        .expect_at_most(0)
        .create_async()
        .await;

    let headers_remote_page_mock = server
        .mock("HEAD", "/profile")
        .with_header(
            LINK,
            &format!(
                r#"<{}/metadata>; rel="{}""#,
                server.url(),
                super::Client::<crate::http::reqwest::Client>::LINK_REL
            ),
        )
        .expect_at_most(1)
        .create_async()
        .await;

    let metadata_endpoint_mock = server
        .mock("GET", "/metadata")
        .with_header(CONTENT_TYPE, CONTENT_TYPE_JSON)
        .with_body(serde_json::json!(metadata).to_string())
        .expect_at_most(1)
        .create_async()
        .await;

    let client = super::Client::new(
        "http://example.com",
        crate::http::reqwest::Client::default(),
    )?;

    let remote_url = format!("{}/profile", server.url()).parse().unwrap();
    let resp = client.obtain_metadata(&remote_url).await;

    headers_remote_page_mock.assert_async().await;
    html_remote_page_mock.assert_async().await;
    metadata_endpoint_mock.assert_async().await;
    assert_eq!(
        resp,
        Ok(metadata),
        "expected no metadata anywhere due to lack of an endpoint"
    );

    Ok(())
}

#[test]
fn server_metadata_new_authorization_request_url() -> miette::Result<()> {
    let issuer: Url = "https://example.com".parse().unwrap();
    let metadata = ServerMetadata {
        authorization_endpoint: format!("{}/auth", issuer.as_str()).parse().unwrap(),
        token_endpoint: format!("{}/token", issuer.as_str()).parse().unwrap(),
        ticket_endpoint: format!("{}/endpoints/ticket", issuer.as_str()).parse().ok(),
        introspection_endpoint: format!("{}/introspect", issuer.as_str()).parse().ok(),
        code_challenge_methods_supported: vec!["S256".to_string()],
        scopes_supported: Scopes::minimal(),
        issuer,
    };

    let (code_challenge, code_challenge_method) =
        CodeChallenge::generate(super::CodeChallengeMethod::S256)?;
    let formed_url = metadata.new_authorization_request_url(
        super::AuthorizationRequestFields {
            client_id: ClientId::new("http://client.example.com")?,
            redirect_uri: "http://client.example.com/redirect"
                .parse::<Url>()
                .into_diagnostic()?
                .into(),
            state: "nu-state".to_string(),
            challenge: code_challenge,
            challenge_method: code_challenge_method,
            scope: Default::default(),
        },
        Default::default(),
    )?;

    assert!(
        !formed_url.query().unwrap_or_default().contains("scope="),
        "does not includes an empty scope string"
    );

    assert!(
        formed_url
            .query()
            .unwrap_or_default()
            .contains("state=nu-state"),
        "includes the provided state value"
    );

    Ok(())
}
