;;;
;;; tl-str.el --- Emacs Lisp Library module about string
;;;
;;; Copyright (C) 1995 Free Software Foundation, Inc.
;;; Copyright (C) 1994 .. 1996 MORIOKA Tomohiko
;;;
;;; Author: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;; Version:
;;;	$Id: tl-str.el,v 7.1 1996/01/18 14:34:50 morioka Exp $
;;; Keywords: string
;;;
;;; This file is part of tl (Tiny Library).
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with This program.  If not, write to the Free Software
;;; Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; Code:

(require 'emu)


;;; @ converter
;;;

(defmacro char-list-to-string (char-list)
  "Convert list of character CHAR-LIST to string. [tl-str.el]"
  (` (mapconcat (function char-to-string)
		(, char-list)
		"")
     ))


;;; @ space
;;;

(defun eliminate-top-spaces (str)
  "Eliminate top sequence of space or tab and return it. [tl-str.el]"
  (if (string-match "^[ \t]+" str)
      (substring str (match-end 0))
    str))
  

;;; @ truncate-string
;;;

(if (not (fboundp 'truncate-string))
    (defun truncate-string (str width)
      "Truncate a string STR which is not longer than WIDTH column.
\[tl-str.el]"
      (let ((w (string-width str))
	    (col 0) (idx 0) (p-idx 0) chr)
	(if (<= w width)
	    str
	  (while (< col width)
	    (setq chr (aref str idx)
		  col (+ col (char-width chr))
		  p-idx idx
		  idx (+ idx (char-bytes chr))
		  ))
	  (substring str 0 (if (= col width)
			       idx
			     p-idx))
	  )))
  )

(defalias 'top-short-string 'truncate-string)
(defalias 'rightful-boundary-short-string 'truncate-string)


;;; @ RCS version
;;;

(defun get-version-string (id)
  "Return a version-string from RCS ID. [tl-str.el]"
  (and (string-match ",v \\([0-9][0-9.][0-9.]+\\)" id)
       (substring id (match-beginning 1)(match-end 1))
       ))


;;; @ file name
;;;

(defun replace-as-filename (str)
  (let ((dest "")
	(i 0)(len (length str))
	chr)
    (while (< i len)
      (setq chr (elt str i))
      (if (or (and (<= ?+ chr)(<= chr ?.))
	      (and (<= ?0 chr)(<= chr ?:))
	      (= chr ?=)
	      (and (<= ?@ chr)(<= chr ?\[))
	      (and (<= ?\] chr)(<= chr ?_))
	      (and (<= ?a chr)(<= chr ?{))
	      (and (<= ?} chr)(<= chr ?~))
	      )
	  (setq dest (concat dest
			     (char-to-string chr)))
	)
      (setq i (+ i 1))
      )
    dest))


;;; @ symbol
;;;

(defun symbol-concat (&rest args)
  "Return a symbol whose name is concatenation of arguments ARGS
which are string or symbol. [tl-str.el]"
  (intern (apply (function concat)
		 (mapcar (function
			  (lambda (s)
			    (cond ((symbolp s) (symbol-name s))
				  ((stringp s) s)
				  )
			    ))
			 args)))
  )


;;; @ matching
;;;

(defun top-string-match (pat str)
  "Return a list (MATCHED REST) if string PAT is top substring of
string STR. [tl-str.el]"
  (if (string-match
       (concat "^" (regexp-quote pat))
       str)
      (list pat (substring str (match-end 0)))
    ))

(defun middle-string-match (pat str)
  "Return a list (PREVIOUS MATCHED REST) if string PAT is found in
string STR. [tl-str.el]"
  (if (equal pat str)
      (list nil pat nil)
    (if (string-match (regexp-quote pat) str)
	(let ((b (match-beginning 0))
	      (e (match-end 0)) )
	  (list (if (not (= b 0))
		    (substring str 0 b)
		  )
		pat
		(if (> (length str) e)
		    (substring str e)
		  )
		)))))

(defun re-top-string-match (pat str)
  "Return a list (MATCHED REST) if regexp PAT is matched as top
substring of string STR. [tl-str.el]"
  (if (string-match (concat "^" pat) str)
      (let ((e (match-end 0)))
	(list (substring str 0 e)(substring str e))
	)))


;;; @ regexp
;;;

(defun regexp-* (regexp)
  (concat regexp "*"))

(defun regexp-or (&rest args)
  (concat "\\(" (mapconcat (function identity) args "\\|") "\\)"))


;;; @ end
;;;

(provide 'tl-str)

;;; tl-str.el ends here
