/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include "loader.h"
#include "mvmebug.h"

/*--------------------------------------------------------------------------*/

static BRDID_PKT		*brdid_ptr;
static unsigned char	disk_dlun;
static unsigned char	disk_clun;
static char				bootargs[256];

/*--------------------------------------------------------------------------*/
/* Perform architecture specific initialisations.
 *
 * Note: malloc() heap has not yet been initialised.
 */

void
loader_init
(	CALLREGS	*regs
)
{	char		*s;
	char		*d;
	char		*e;

	/* save disk and controller luns */
	disk_dlun = (unsigned char) regs->d[0];
	disk_clun = (unsigned char) regs->d[1];

	/* copy argument string */
	s = regs->a[5];
	d = bootargs;
	e = bootargs + sizeof(bootargs) - 1;

	while ((s != regs->a[6]) && (d != e))
		*d++ = *s++;
	*d = '\0';

	put_str("\nMotorola MVME Linux Loader V" VERNUMB "\n\n");

	/* identify board */
	brdid_ptr = MVMEBug_brdid();

	if ((brdid_ptr->bdid != 0x42444944)
	&&	(brdid_ptr->bdid != 0x21494421))
	{
		panic("Cannot identify board");
	}
}

/*--------------------------------------------------------------------------*/

unsigned long
get_time
(	void
)
{	unsigned char buf[8];

	MVMEBug_rtc_read(buf);
	return 	(BCD2BIN(buf[4]) * 24)
		+	(BCD2BIN(buf[5]) * 60)
		+    BCD2BIN(buf[6]);
}

/*--------------------------------------------------------------------------*/
/* Print character.
 */

void
put_char
(	const int	c
)
{
	if (c == '\n')
		MVMEBug_putchar('\r');

	MVMEBug_putchar(c);
}

/*--------------------------------------------------------------------------*/
/* Wait for and return character from keyboard.
 */

int
get_char
(	unsigned long	timeout		/* maximum time to wait for character		*/
)
{	unsigned long	start;
	unsigned long	now;

	if (timeout)
	{
		/* get start time */
		start = get_time();

		/* get timeout second */
		timeout += start;

		while (MVMEBug_getchar_status() == 0)
		{
			/* check for wrap at midnight */
			if ((now = get_time()) < start)
			{
				/* adjust */
				now += (60 * 60 * 24);
			}

			/* check for timeout */
			if (now > timeout)
			{
				/* timeout */
				break;
			}
		}

		if (MVMEBug_getchar_status() == 0)
			return -1;
	}

	return MVMEBug_getchar();
}

/*--------------------------------------------------------------------------*/

int
disk_read
(	void			*buf,
	unsigned long	sector,
	unsigned long	extent
)
{	DSKRD_CMND		cdb;

	/* convert to units of 256 byte blocks */
	sector *= (SECTOR_SIZE / 256);
	extent *= (SECTOR_SIZE / 256);

	/* build command descriptor */
	cdb.clun    = disk_clun;
	cdb.dlun    = disk_dlun;
	cdb.status  = 0xdead;
	cdb.address = (unsigned long) buf;
	cdb.block   = sector; 
	cdb.count   = extent;
	cdb.flags   = 0;
	cdb.mod     = 0;

	MVMEBug_disk_read(&cdb);

	if (cdb.status)
	{
		printf("\nREAD FAILED: status=%04x\n", cdb.status);
		return -1;
	}

	return 0;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_compat_booti_version
(	void
)
{
	switch (brdid_ptr->brdno)
	{
		case 0x0147: return COMPAT_MVME147_BOOTI_VERSION;
		case 0x0162:
		case 0x0172: return COMPAT_MVME162_BOOTI_VERSION;
		default    : return COMPAT_MVME167_BOOTI_VERSION;
	}
}

/*--------------------------------------------------------------------------*/

unsigned long
get_booti_version
(	void
)
{
	return MVME16x_BOOTI_VERSION;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_compat_machtype
(	void
)
{
	switch (brdid_ptr->brdno)
	{
		case 0x0147: return COMPAT_MACH_MVME147;
		case 0x0162:
		case 0x0172: return COMPAT_MACH_MVME162;
		default    : return COMPAT_MACH_MVME167;
	}
}

/*--------------------------------------------------------------------------*/

unsigned long
get_machtype
(	void
)
{
	switch (brdid_ptr->brdno)
	{
		case 0x0147: return MACH_MVME147;
		default    : return MACH_MVME16x;
	}
}

/*--------------------------------------------------------------------------*/
/*
 *	This assembler code is copied into local storage, and then executed.
 *	It copies the kernel and ramdisk images to their final resting places.
 *
 *	It is called with:
 *
 *      a0 = address of this code (very top of memory)
 *      a1 = kernel destination address (low memory 0x1000)
 *		a2 = kernel source address
 *		a3 = ramdisk destination address (just below this code)
 *		a4 = ramdisk source address
 *		d0 = kernel size + size of bootinfo data rounded up next multiple of 4
 *		d1 = ramdisk size rounded to next multiple of 1K
 *      d2 = non-zero if monitor should be called
 */

__asm(".text\n"
__ALIGN_STR "\n"
".globl startcode_beg;\n"
".globl startcode_end;\n"
"startcode_beg:																\n\
								| copy the ramdisk to the top of memory		\n\
								| (from back to front)						\n\
		addl	%d1,%a4			| src   = (u_long *) (rd_src + rd_size);	\n\
		movel	%a3,%a5														\n\
		addl	%d1,%a5			| dest  = (u_long *) (rd_dest + rd_size);	\n\
																			\n\
		bras	2f				| do										\n\
1:		movel	-(%a4),-(%a5)	|   *--dest = *--src;						\n\
2:		cmpl	%a3,%a5			| while (dest > ramdisk_dest)				\n\
		jgt		1b				| 											\n\
																			\n\
								| copy kernel text and data, and bootinfo	\n\
		movel	%a2,%a4			| limit =									\n\
		addl	%d0,%a4			|	(u_long *) (kernel_src + kernel_size);	\n\
		movel	%a1,%a5			| dest  = (u_long *) kernel_dest			\n\
																			\n\
		bras	2f				| do										\n\
1:		movel	(%a2)+,(%a5)+	|  *dest++ = *src++;						\n\
2:		cmpl	%a4,%a2			| while (src < limit)						\n\
		jlt		1b				|											\n\
																			\n\
		tstl	%d2				| call monitor?								\n\
		jeq		1f				| branch if not								\n\
																			\n\
		trap	#15															\n\
		dc.w	0x0063														\n\
																			\n\
1:		jmp		(%a1)			| start kernel or enter BUG					\n\
startcode_end:																\n\
");

/*--------------------------------------------------------------------------*/

void
print_model
(	void
)
{	char	suf[4];

	suf[1] = brdid_ptr->suffix[0];
	suf[2] = brdid_ptr->suffix[1];
	suf[3] = '\0';
	suf[0] = suf[1] ? '-' : '\0';

	printf("MVME%x%s", brdid_ptr->brdno, suf);
}

/*--------------------------------------------------------------------------*/

int
add_vme_bootinfo
(	void
)
{	unsigned long	vme_type;

	switch (brdid_ptr->brdno)
	{
		case 0x0147: vme_type = VME_TYPE_MVME147; break;
		case 0x0162: vme_type = VME_TYPE_MVME162; break;
		case 0x0166: vme_type = VME_TYPE_MVME166; break;
		case 0x0167: vme_type = VME_TYPE_MVME167; break;
		case 0x0172: vme_type = VME_TYPE_MVME172; break;
		case 0x0177: vme_type = VME_TYPE_MVME177; break;
	}

	if (!add_bi_record(BI_VME_TYPE, sizeof(vme_type), &vme_type))
		return FALSE;

	if (!add_bi_record(BI_VME_BRDINF, sizeof(BRDID_PKT), brdid_ptr))
		return FALSE;

	return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Enter ROM debug monitor
 */

void
call_bug
(	void
)
{
	MVMEBug_enter();
}

/*--------------------------------------------------------------------------*/
/* return pointer to additional kernel command line parameters from bootrom
 */

const char *
bootrom_args
(	void
)
{
	return bootargs;
}

/*-----------------------------< end of file >------------------------------*/
