// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// mathops.h
//
// Two classes - Vector and SolidAngle - to make 3-d math operations easier

#ifndef _MATHOPS_H
#define _MATHOPS_H

#include <cmath>
#include <cstdlib>
#include <climits>
using namespace std;

// useful constants:
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif
#ifndef M_PI_2
#define M_PI_2 1.57079632679489661923
#endif
#ifndef M_PI_12
#define M_PI_12 0.26179938779914943654
#endif
#ifndef M_PI_180
#define M_PI_180 0.017453292519943295769
#endif
#ifndef M_SQRT2 
#define M_SQRT2 1.41421356237309504880
#endif

// Euler angles for the conversions between celestial and galactic
//  coordinates:
//  A = 3.36473 +/- .00026 rad = RA of North Galactic Pole
//  D = 0.47370 +/- .00017 rad = Dec of North Galactic Pole
//  T = 0.57399 +/- .00017 rad = Galactic Longitude at the first point where
//                               Celestial and Galactic Equators cross
//
// Constants are given in sine and cosine form to decrease the number
//  of floating-point calculations.

const double COS_A = -0.97520799,
             SIN_A = -0.22129027,
             COS_D =  0.88988651,
             SIN_D =  0.45618198,
             COS_T =  0.83974115,
             SIN_T =  0.54298693;

// A couple rounding functions

double sigdigits(double number, int numdigits);

// roundoff(): Rounds off at a given decimal place.
//  e.g. to round to 2 decimal places, use numdigits = 2
inline double roundoff(double number, int numdigits)
{
  if (number < 0.0) return -roundoff(-number, numdigits);
  return pow(10.0,-numdigits) * (long int)(pow(10.0,numdigits) * number + 0.5);
}


// "Safe" versions of ato?()

inline double myatof(const char * numstring)
{ return strtod(numstring, 0); }

inline long int myatol(const char * numstring)
{ return strtol(numstring, 0, 10); }

inline int myatoi(const char * numstring)
{
  long int temp = strtol(numstring, 0, 10);
  if (temp >= (long int)INT_MAX) return INT_MAX;
  else if (temp <= (long int)INT_MIN) return INT_MIN;
  else return (int)temp;
}


class SolidAngle;  // forward declaration for below

// The Vector class defines the standard 3-D vector operations, plus
//  functions to convert into spherical coordinates and between
//  celestial and galactic systems.

class Vector { 
 private:
  double x, y, z; 
 public:
  Vector() { x = y = z = 0; }
  Vector(double X, double Y, double Z) { x = X; y = Y; z = Z; }

  Vector &operator += (Vector v);
  Vector &operator -= (Vector v);
  Vector operator + (Vector v) const;
  Vector operator - (Vector v) const;
  Vector operator * (double c) const;
  friend Vector operator * (double c, Vector v);
  Vector operator / (double c) const;

  inline bool operator == (Vector v) const
    { return x == v.x && y == v.y && z == v.z; }
  inline bool operator != (Vector v) const
    { return ! (*this == v); }

  double magnitude() const;
  Vector unitvector() const;
  double dot(Vector v) const;
  Vector cross(Vector v) const;
  SolidAngle toSpherical() const;
  Vector toGalactic() const;
  Vector toCelestial() const;

  inline double getX() const { return x; }
  inline double getY() const { return y; }
  inline double getZ() const { return z; }
};

// The SolidAngle class is used to hold latitude and longitude in a convenient
//  structure, and allow conversion to Cartesian coordinates and between
//  celestial and galactic systems.

class SolidAngle {
 private:
  double theta /* latitude */,   // both in radians, of course
         phi   /* longitude */;
 public:
  SolidAngle() { theta = phi = 0; }
  SolidAngle(double Phi, double Theta) 
  {
    theta = Theta; phi = Phi;
    while (phi >= 2 * M_PI) phi -= (2 * M_PI);
    while (phi < 0.0)       phi += (2 * M_PI);
    if (theta > M_PI_2) theta = M_PI_2;
    if (theta < -M_PI_2) theta = -M_PI_2;
  }

  inline bool operator == (SolidAngle a) const
    { return theta == a.theta && phi == a.phi; }
  inline bool operator != (SolidAngle a) const
    { return ! (*this == a); }

  Vector toCartesian(double radius) const;
  SolidAngle toGalactic() const;
  SolidAngle toCelestial() const;

  inline double getTheta() const { return theta; }
  inline double getPhi()   const { return phi;   }
};


#endif // #define _MATHOPS_H
