/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: stream.h,v 1.44 2004/02/09 11:25:49 sasa Exp $
 *
 ***************************************************************************/

#ifndef ZORP_STREAM_H_INCLUDED
#define ZORP_STREAM_H_INCLUDED

#include <zorp/zorplib.h>
#include <zorp/zobject.h>

#include <time.h>
#include <glib.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifdef G_OS_WIN32
  #define SHUT_RDWR SD_BOTH
  #define SHUT_WR SD_SEND
  #define SHUT_RD SD_RECEIVE
#endif

#define Z_STREAM_MAX_NAME   128

#define Z_STREAM_FLAG_READ  	1
#define Z_STREAM_FLAG_WRITE 	2
#define Z_STREAM_FLAG_PRI   	4
#define Z_STREAM_FLAG_TIMEOUT 	8

#define Z_STREAM_TIMEOUT_BLOCK 0

/* Stream types */

/* stream control messages:

     0. byte message
     1. byte stream type
     3. byte undefined
     4. byte stream flags
 */
 
#define ZST_CTRL_MSG(x)       ((x)&0xFFFF)
#define ZST_CTRL_MSG_FLAGS(x) ((x) & 0xFF000000)

#define ZST_CTRL_MSG_FORWARD  0x80000000

#define ZST_CTRL_GET_FD               (0x01)
#define ZST_CTRL_SET_COND_READ        (0x02)
#define ZST_CTRL_SET_COND_WRITE       (0x03)
#define ZST_CTRL_SET_COND_PRI         (0x04)
#define ZST_CTRL_SET_CALLBACK_READ    (0x06)
#define ZST_CTRL_SET_CALLBACK_WRITE   (0x07)
#define ZST_CTRL_SET_CALLBACK_PRI     (0x08)
#define ZST_CTRL_SET_TIMEOUT_BLOCK    (0x0A)
#define ZST_CTRL_GET_COND_READ        (0x0C)
#define ZST_CTRL_GET_COND_WRITE       (0x0D)
#define ZST_CTRL_GET_COND_PRI         (0x0E)
#define ZST_CTRL_GET_CALLBACK_READ    (0x10)
#define ZST_CTRL_GET_CALLBACK_WRITE   (0x11)
#define ZST_CTRL_GET_CALLBACK_PRI     (0x12)
#define ZST_CTRL_SET_NONBLOCK         (0x14)
#define ZST_CTRL_GET_NONBLOCK         (0x15)
#define ZST_CTRL_GET_BROKEN           (0x16)

#define ZST_LINE_OFS	('L' << 8)

typedef struct _ZStream ZStream;
typedef struct _ZStreamContext ZStreamContext;

typedef gboolean (*ZStreamCallback)(struct _ZStream *, GIOCondition, gpointer);

typedef struct _ZStreamSource
{
  GSource super;
  ZStream *stream;
} ZStreamSource;

GSource *z_stream_source_new(ZStream *stream);

struct _ZStreamContext
{
  gboolean restored;
  
  gboolean want_read;       /* do we want read callbacks? */
  gpointer user_data_read;  /* opaque pointer, can be used by read callback */
  GDestroyNotify user_data_read_notify;
  ZStreamCallback read_cb;  /* pointer to read callback */

  gboolean want_pri;        /* do we want urgent data callbacks? */
  gpointer user_data_pri;   /* opaque pointer, can be used by pri callback */
  GDestroyNotify user_data_pri_notify;
  ZStreamCallback pri_cb;   /* pointer to urgent data callback */

  gboolean want_write;      /* do we want write callbacks */
  gpointer user_data_write; /* opaque pointer, can be used by write callback */
  GDestroyNotify user_data_write_notify;
  ZStreamCallback write_cb; /* pointer to write callback */
  
  gint timeout;
  
  gpointer stream_extra;
};

void z_stream_context_destroy(ZStreamContext *self);


typedef struct _ZStreamSetCb
{
  ZStreamCallback cb;
  gpointer cb_data;
  GDestroyNotify cb_notify;
} ZStreamSetCb;

typedef struct _ZStreamFuncs
{
  ZObjectFuncs super;
  GIOStatus (*read)(ZStream *stream,
                    gchar   *buf,
                    gsize   count,
                    gsize   *bytes_read,
                    GError  **err);

  GIOStatus (*write)(ZStream     *stream,
                     const gchar *buf,
                     gsize       count,
                     gsize       *bytes_written,
                     GError      **err);

  GIOStatus (*read_pri)(ZStream *stream,
                        gchar   *buf,
                        gsize   count,
                        gsize   *bytes_read,
                        GError  **err);

  GIOStatus (*write_pri)(ZStream     *stream,
                         const gchar *buf,
                         gsize       count,
                         gsize       *bytes_written,
                         GError      **err);

  GIOStatus  (*shutdown)(ZStream *stream,
                         int     i,
                         GError  **err);

  GIOStatus  (*close) (ZStream *stream,
                       GError  **err);
  
  gboolean (*ctrl) (ZStream  *stream,
                    guint    function,
                    gpointer value,
                    guint    vlen);

  void (*attach_source)(ZStream *stream, GMainContext *context);
  void (*detach_source)(ZStream *stream);

  gboolean (*watch_prepare)(ZStream *s, GSource *src, gint *timeout);
  gboolean (*watch_check)(ZStream *s, GSource *src);
  gboolean (*watch_dispatch)(ZStream *s, GSource *src);
  void (*watch_finalize)(ZStream *s, GSource *source);
  
  gsize (*extra_get_size)(ZStream *s);
  gsize (*extra_save)(ZStream *s, gpointer context);
  gsize (*extra_restore)(ZStream *s, gpointer context);
} ZStreamFuncs;

#ifdef G_OS_WIN32
  LIBZORPLL_DLLIMPEX ZClass ZStream__class;
#else
  extern ZClass ZStream__class;
#endif

/*+

  ZStream encapsulates a Unix file descriptor. It contains some
  attributes and callbacks which are used by ZPoll.

  +*/
struct _ZStream 
{
  ZObject super;
  gchar name[Z_STREAM_MAX_NAME];
  gint timeout;
  gint umbrella_flags;
  
  ZStream *child;
  GSource *source;

  time_t time_open;
  guint64 bytes_recvd, bytes_sent;
  
  gboolean want_read;       /* do we want read callbacks? */
  gpointer user_data_read;  /* opaque pointer, can be used by read callback */
  GDestroyNotify user_data_read_notify;
  ZStreamCallback read_cb;  /* pointer to read callback */

  gboolean want_pri;        /* do we want urgent data callbacks? */
  gpointer user_data_pri;   /* opaque pointer, can be used by pri callback */
  GDestroyNotify user_data_pri_notify;
  ZStreamCallback pri_cb;   /* pointer to urgent data callback */

  gboolean want_write;      /* do we want write callbacks */
  gpointer user_data_write; /* opaque pointer, can be used by write callback */
  GDestroyNotify user_data_write_notify;
  ZStreamCallback write_cb; /* pointer to write callback */
};


ZStream *z_stream_new(ZClass *class, gchar *name, ZStream *child, gint umbrella_flags);
gboolean z_stream_save_context(ZStream *self, ZStreamContext *context);
gboolean z_stream_restore_context(ZStream *self, ZStreamContext *context);

/* virtual methods for static references like calling the superclass's function in derived classes */

gboolean z_stream_ctrl_method(ZStream *s, guint function, gpointer value, guint vlen);
void z_stream_free_method(ZObject *s);

static inline ZStream *
z_stream_ref(ZStream *self)
{
  return (ZStream *) z_object_ref(&self->super);
}

static inline void 
z_stream_unref(ZStream *self)
{
  z_object_unref(&self->super);
}

/* virtual functions */

static inline gboolean
z_stream_ctrl(ZStream *self, guint function, gpointer value, guint vlen)
{
  return Z_FUNCS(self, ZStream)->ctrl(self, function, value, vlen);
}

static inline GIOStatus 
z_stream_read(ZStream *self,
              gchar   *buf,
              gsize   count,
              gsize   *bytes_read,
              GError  **err)
{
  return Z_FUNCS(self, ZStream)->read(self, buf, count, bytes_read, err);
}

static inline GIOStatus 
z_stream_write(ZStream     *self,
               const gchar *buf,
               gsize       count,
               gsize       *bytes_written,
               GError      **err)
{
  return Z_FUNCS(self, ZStream)->write(self, buf, count, bytes_written, err);
}

static inline GIOStatus 
z_stream_read_pri(ZStream *self,
                  gchar   *buf,
                  gsize   count,
                  gsize   *bytes_read,
                  GError  **err)
{
  return Z_FUNCS(self, ZStream)->read_pri(self, buf, count, bytes_read, err);
}

static inline GIOStatus 
z_stream_write_pri(ZStream     *self,
                   const gchar *buf,
                   gsize       count,
                   gsize       *bytes_written,
                   GError      **err)
{
  return Z_FUNCS(self, ZStream)->write_pri(self, buf, count, bytes_written, err);
}

static inline GIOStatus 
z_stream_shutdown(ZStream *self,
                  int     how,
                  GError  **err)
{
  return Z_FUNCS(self, ZStream)->shutdown(self, how, err);
}

static inline GIOStatus
z_stream_close(ZStream *self, GError **err)
{
  return Z_FUNCS(self, ZStream)->close(self, err);
}

static inline void
z_stream_attach_source(ZStream *self, GMainContext *context)
{
  Z_FUNCS(self, ZStream)->attach_source(self, context);
}

static inline void
z_stream_detach_source(ZStream *self)
{
  Z_FUNCS(self, ZStream)->detach_source(self);
}

static inline gboolean
z_stream_watch_prepare(ZStream *self, GSource *s, gint *timeout)
{
  return Z_FUNCS(self, ZStream)->watch_prepare(self, s, timeout);
}

static inline gboolean
z_stream_watch_check(ZStream *self, GSource *s)
{
  return Z_FUNCS(self, ZStream)->watch_check(self, s);
}

static inline gboolean
z_stream_watch_dispatch(ZStream *self, GSource *s)
{
  return Z_FUNCS(self, ZStream)->watch_dispatch(self, s);
}

static inline void
z_stream_watch_finalize(ZStream *self, GSource *s)
{
  if (Z_FUNCS(self, ZStream)->watch_finalize)
    Z_FUNCS(self, ZStream)->watch_finalize(self, s);
}

static inline gsize
z_stream_extra_get_size(ZStream *s)
{
  return Z_FUNCS(s, ZStream)->extra_get_size(s);
}

static inline gsize
z_stream_extra_save(ZStream *s, gpointer context)
{
  return Z_FUNCS(s, ZStream)->extra_save(s, context);
}

static inline gsize
z_stream_extra_restore(ZStream *s, gpointer context)
{
  return Z_FUNCS(s, ZStream)->extra_restore(s, context);
}

/* helper functions */
static inline int
z_stream_get_fd(ZStream *s)
{
  gint ret = -1;
  if (!z_stream_ctrl(s, ZST_CTRL_GET_FD, &ret, sizeof(ret)))
    ret = -1;
  return ret;
}

static inline gboolean
z_stream_broken(ZStream *s)
{
  gboolean ret = -1;
  if (!z_stream_ctrl(s, ZST_CTRL_GET_BROKEN, &ret, sizeof(ret)))
    ret = FALSE;
  return ret;
}

static inline gboolean
z_stream_set_cond(ZStream *s, guint type, gboolean value)
{
  gboolean ret = FALSE;

  switch(type)
    {
    case Z_STREAM_FLAG_READ:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_COND_READ, &value, sizeof(value));
      break;
    case Z_STREAM_FLAG_WRITE:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_COND_WRITE, &value, sizeof(value));
      break;
    case Z_STREAM_FLAG_PRI:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_COND_PRI, &value, sizeof(value));
      break;
    default:
      break;
    }
  return ret;
}

static inline gboolean
z_stream_set_callback(ZStream *s, guint type, ZStreamCallback callback, gpointer user_data, GDestroyNotify notify)
{
  gboolean ret = FALSE;
  ZStreamSetCb cb;

  cb.cb = callback;
  cb.cb_data = user_data;
  cb.cb_notify = notify;

  switch(type)
    {
    case Z_STREAM_FLAG_READ:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_CALLBACK_READ, &cb, sizeof(cb));
      break;
    case Z_STREAM_FLAG_WRITE:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_CALLBACK_WRITE, &cb, sizeof(cb));
      break;
    case Z_STREAM_FLAG_PRI:
      ret = z_stream_ctrl(s, ZST_CTRL_SET_CALLBACK_PRI, &cb, sizeof(cb));
      break;
    default:
      break;
    }
  return ret;
}

static inline gboolean
z_stream_set_timeout(ZStream *s, gint timeout)
{
  s->timeout = timeout;
  return TRUE;
}

static inline gboolean
z_stream_set_nonblock(ZStream *self, gboolean nonblock)
{
  return z_stream_ctrl(self, ZST_CTRL_SET_NONBLOCK, &nonblock, sizeof(nonblock));
}

static inline gboolean
z_stream_get_nonblock(ZStream *self)
{
  gboolean nonblock;
  
  z_stream_ctrl(self, ZST_CTRL_GET_NONBLOCK, &nonblock, sizeof(nonblock));
  return nonblock;
}

ZStream *
z_stream_search_stack(ZStream *top, gint direction, ZClass *class);


#ifdef __cplusplus
}
#endif

#endif
