/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file ProjectOpusPlaylistExtensionReader.cpp
 * Implementation of ProjectOpusPlaylistExtensionReader.
 */

#include <spiff/ProjectOpus/ProjectOpusPlaylistExtensionReader.h>
#include <spiff/ProjectOpus/ProjectOpusPlaylistExtension.h>
#include <spiff/SpiffReader.h>
#include <spiff/SpiffStack.h>
#include <stdio.h>
#include <cstring>

namespace Spiff {
namespace ProjectOpus {



/**
 * Holds tag types that are pushed on the readers's tag stack.
 */
enum ProjectOpusTag {
	TAG_OPUS_PROJECT_INFO = TAG_USER ///< po:info tag
};



/// @cond DOXYGEN_NON_API

/**
 * D object for ProjectOpusPlaylistExtensionReader.
 */
class ProjectOpusPlaylistExtensionReaderPrivate {

	friend class ProjectOpusPlaylistExtensionReader;

	ProjectOpusPlaylistExtension extension; ///< Extension being built
	bool firstInfo; ///< First info element flag

	/**
	 * Creates a new D object.
	 */
	ProjectOpusPlaylistExtensionReaderPrivate()
			: extension(), firstInfo(true) {

	}

	/**
	 * Destroys this D object.
	 */
	~ProjectOpusPlaylistExtensionReaderPrivate() {

	}

};

/// @endcond



ProjectOpusPlaylistExtensionReader::ProjectOpusPlaylistExtensionReader(
		SpiffReader * reader)
		: SpiffExtensionReader(reader),
		d(new ProjectOpusPlaylistExtensionReaderPrivate()) {

}



ProjectOpusPlaylistExtensionReader::ProjectOpusPlaylistExtensionReader(const ProjectOpusPlaylistExtensionReader & source)
		: SpiffExtensionReader(source),
		d(new ProjectOpusPlaylistExtensionReaderPrivate(*(source.d))) {

}



ProjectOpusPlaylistExtensionReader & ProjectOpusPlaylistExtensionReader::operator=(const ProjectOpusPlaylistExtensionReader & source) {
	if (this != &source) {
		SpiffExtensionReader::operator=(source);
		*(this->d) = *(source.d);
	}
	return *this;
}



ProjectOpusPlaylistExtensionReader::~ProjectOpusPlaylistExtensionReader() {
	delete this->d;
}



bool ProjectOpusPlaylistExtensionReader::handleExtensionStart(const XML_Char * fullName,
		const XML_Char ** atts) {
	switch (this->stack->getSize() + 1) {
	case 2:
		// We only get called on this level for playlist.extension
		// Name and attributes have already been checked.
		this->stack->push(TAG_PLAYLIST_EXTENSION);
		return true;

	case 3:
		// Check and skip namespace
		if (::PORT_STRNCMP(fullName, PROJECT_OPUS_NS_HOME, PROJECT_OPUS_NS_HOME_LEN)
				|| ::PORT_STRCMP(fullName + PROJECT_OPUS_NS_HOME_LEN + 1, _PT("info"))) {
			setError(SPIFF_READER_ERROR_ELEMENT_FORBIDDEN,
					SPIFF_READER_TEXT_ONE_ELEMENT_FORBIDDEN,
					fullName);
			return false;
		}

		// One info at most
		if (!this->d->firstInfo) {
			setError(SPIFF_READER_ERROR_ELEMENT_TOOMANY,
					SPIFF_READER_TEXT_ZERO_TOO_MANY_ELEMENTS(PROJECT_OPUS_NS_HOME, _PT("info")));
			return false;
		}

		if (!handleInfoAttribs(atts)) {
			return false;
		}

		this->d->firstInfo = false;
		this->stack->push(TAG_OPUS_PROJECT_INFO);
		return true;

	case 4:
		if (this->stack->top() == TAG_PLAYLIST_TRACKLIST_TRACK) {
			// Name and attributes have already been checked.
			this->stack->push(TAG_PLAYLIST_TRACKLIST_TRACK_EXTENSION);
			return true;
		}
		break;

	}

	this->stack->push(TAG_UNKNOWN);
	return true;
}



bool ProjectOpusPlaylistExtensionReader::handleExtensionEnd(const XML_Char * /*fullName*/) {
	switch (this->stack->getSize()) {
	case 2:
		// One info minimum
		if (this->d->firstInfo) {
			setError(SPIFF_READER_ERROR_ELEMENT_MISSING,
					SPIFF_READER_TEXT_ZERO_ELEMENT_MISSING(PROJECT_OPUS_NS_HOME, _PT("info")));
			return false;
		}
		break;

	}

	this->stack->pop();
	return true;
}



bool ProjectOpusPlaylistExtensionReader::handleExtensionCharacters(const XML_Char * /*s*/, int /*len*/) {
	return true;
}



SpiffExtension * ProjectOpusPlaylistExtensionReader::wrap() {
	return this->d->extension.clone();
}



SpiffExtensionReader * ProjectOpusPlaylistExtensionReader::createBrother(SpiffReader * reader) const {
	return new ProjectOpusPlaylistExtensionReader(reader);
}



bool ProjectOpusPlaylistExtensionReader::handleInfoAttribs(const XML_Char ** atts) {
	bool typeFound = false;
	bool nidFound = false;

	while (atts[0] != NULL) {
		if (!::PORT_STRCMP(atts[0], _PT("type"))) {
			ProjectOpusPlaylistType dummyType = TYPE_PLAYLIST; // Init not needed
			if (!::PORT_STRCMP(atts[1], _PT("album"))) {
				dummyType = TYPE_ALBUM;
			} else if (!::PORT_STRCMP(atts[1], _PT("playlist"))) {
				dummyType = TYPE_PLAYLIST;
			} else {
				setError(SPIFF_READER_ERROR_ATTRIBUTE_INVALID,
						PROJECT_OPUS_TEXT_ZERO_ATTRIBUTE_OUT_OF_SET(_PT("type"), _PT("{'album', 'playlist'}")));
				return false;
			}
			this->d->extension.setType(dummyType);
			typeFound = true;
		} else if (!::PORT_STRCMP(atts[0], _PT("nid"))) {
			int dummyNid;
			if (!SpiffReader::extractInteger(atts[1], 0, &dummyNid)) {
				setError(SPIFF_READER_ERROR_ATTRIBUTE_INVALID,
						SPIFF_READER_TEXT_ZERO_WRONG_ATTRIBUTE_TYPE(_PT("nid"), _PT("unsigned integer")));
				return false;
			}
			this->d->extension.setNodeId(dummyNid);
			nidFound = true;
		} else {
			setError(SPIFF_READER_ERROR_ATTRIBUTE_FORBIDDEN,
					SPIFF_READER_TEXT_ONE_ATTRIBUTE_FORBIDDEN, atts[0]);
			return false;
		}
		atts += 2;
	}

	if (!typeFound) {
		setError(SPIFF_READER_ERROR_ATTRIBUTE_MISSING,
				SPIFF_READER_TEXT_ZERO_ATTRIBUTE_MISSING(_PT("type")));
		return false;
	}

	if (!nidFound) {
		setError(SPIFF_READER_ERROR_ATTRIBUTE_MISSING,
				SPIFF_READER_TEXT_ZERO_ATTRIBUTE_MISSING(_PT("nid")));
		return false;
	}

	return true;
}



}
}
