require "runit/testcase"
require "runit/cui/testrunner"

require "rexml/document"

class XPathTester < RUNIT::TestCase
	include REXML
	@@source = <<-EOF
		<a id='1'>
			<b id='2' x='y'>
				<c id='3'/>
				<c id='4'/>
			</b>
			<d id='5'>
				<c id='6' x='y'/>
				<c id='7'/>
				<c id='8'/>
				<q id='19'/>
			</d>
			<e id='9'>
				<f id='10' a='b'/>
				<f id='11' a='c'/>
				<f id='12' a='d'>
					<g id='13'/>
				</f>
				<f id='14' a='d'/>
			</e>
			<m id='15'>
				<n id='16'>
					<o id='17'>
						<p id='18'/>
					</o>
				</n>
			</m>
		</a>
		EOF

	@@doc = Document.new @@source

	def each_test( element, xpath, num_children )
		count = 0
		XPath::each( element, xpath ) { |child|
			count += 1
			yield child if block_given?
		}
		assert_equal num_children, count
	end

	def test_root
		source = "<a><b/></a>"
		doc = Document.new( source )
		assert_equal "a", doc.root.name
		assert_equal doc, XPath::first( doc, "/" )
	end

	def test_xpath
		assert_equal "a", XPath::first(@@doc, "a").name
		c = XPath::first( @@doc, "a/b/c" )
		assert_equal "3", XPath::first(@@doc, "a/b/c").attributes["id"]
		assert_equal "a", XPath::first(c, "/a").name
		assert_equal "2", XPath::first(c, "/a/b").attributes["id"]
		each_test( @@doc, nil, 1 )
		each_test( @@doc.root, nil, 4 )
		assert_equal "a", XPath::first(@@doc.root, ".").name
		assert_equal "b", XPath::first(c, "..").name
		assert_equal "a", XPath::first(@@doc, "a/b/..").name
		assert_equal "11", XPath::first(@@doc, "a/e/f[@a='c']").attributes["id"]
		assert_equal "11", XPath::first(@@doc, "a/e/[@a='c']").attributes["id"]
		assert_equal "12", XPath::first(@@doc, "a/e/f[3]").attributes["id"]
		assert_equal "13", XPath::first(@@doc, "a/e/f[3]/g").attributes["id"]
		assert_equal "14", XPath::first(@@doc, "a/e/f[@a='d'][2]").attributes["id"]
		assert_equal "14", XPath::first(@@doc, "a/e/f[@a='d'][@id='14']").attributes["id"]
		each_test( @@doc, "//[name()='f' and @a='d']", 2) { |i|
			assert_equal "f", i.name
		}


		each_test( @@doc, "//c", 5 ) { |child| assert_equal "c", child.name }
		each_test( @@doc.root, "b//c", 2) { |child|
			assert_equal "c", child.name
		}

		assert_equal "b", XPath::first( @@doc, "//b[@x]" ).name
		#puts XPath::first( @@doc, "//b[@x]" )
	end

	def test_expressions
		ems = [ Element.new("Sean") ]
		assert_equal 0, REXML::XPath.predicate( ems, "[1>2]" ).size
		assert_equal 1, REXML::XPath.predicate( ems, "[1<2]" ).size
		assert_equal 1, REXML::XPath.predicate( ems, "['sean' = 'sean']" ).size
		assert_equal 0, REXML::XPath.predicate( ems, "['monika' > 'sean']" ).size
		assert_equal 1, REXML::XPath.predicate( ems, "[-1 < 0 and 'monika' < 'sean']" ).size
		assert_equal 1, REXML::XPath.predicate( ems, "[-1 < 0 < 1]" ).size
	end

	def test_functions
		# trivial text() test
		source = "<a>text</a>"
		element = Element.new SourceFactory.create_from(source)
		res = XPath::first(element, "text()")
		assert_equal "text", res.to_s

		# confuse-a-function
		source = "<a xmlns:duh='uri'>more <b id='1'/><duh:b id='2'>dumb</duh:b><b id='3'/><c/> text</a>"
		element = Element.new SourceFactory.create_from(source)
		res = ""
		XPath::each(element, "text()") {|val| res << val.to_s}
		assert_equal "more  text", res

		# This SHOULD fail
		source = "<duh:b id='2'>dumb</b>"
		assert_exception( REXML::ParseException,
			"Should have gotten a missing end tag exception.") {
			REXML::Document.new source
		}

		last_b = XPath::first(element, "b[@id='3']")
		res = XPath::first(element, "b[last()]")
		assert_equal last_b.attributes['id'], res.attributes['id']
		res = XPath::first(element, "b[position()=3]")
		assert_equal last_b.attributes['id'], res.attributes['id']
		res = XPath::first(element, "[name()='c']")
		assert_equal "c", res.name
		res = XPath::first(element, "//[@id='2']")
		assert_equal 'uri', res.namespace
		res = XPath::first(element, "[namespace_uri()='uri']")
		assert_equal '2', res.attributes['id']
	end

	def no_test_ancestor
		doc = REXML::Document.new(File.new("test/testsrc.xml"))
		doc.elements.each("//item") { |el| print el.name
			if el.attributes['x']
				puts " -- "+el.attributes['x']
			else
				puts
			end
		}
		puts "####################################################"
		doc.elements.each("//item/ancestor::") { |el| print el.name
			if el.attributes['x']
				puts " -- "+el.attributes['x']
			else
				puts
			end
		}
	end

	# Here are some XPath tests that were originally submitted by 
	# The code has changed some, but the logic and the source documents are the
	# same.
	# This method reads a document from a file, and then a series of xpaths, 
	# also from a file.  It then checks each xpath against the source file.
	def test_more
    xmlsource   = "test/testsrc.xml"
    xpathtests  = "test/xp.tst"

		doc = REXML::Document.new(File.new(xmlsource))
		#results = ""
		results = REXML::Document.new
		results.add_element "test-results"
		for line in File.new(xpathtests)
			line.strip!
			begin
				rt = doc.root
				doc.elements.each(line) do |el| 
					results.root << el.clone
					#results << el.to_s
				end
			rescue Exception => z
				results.root.add_element( "error", {"path"=>line}).text = z.message+"\n"+z.backtrace[0,10].join("\n")
				#results << "<error path='"+line+"'>"+z.message+"</error>"
			end
		end
		results.elements.each("//error") { |el|
			puts "\nERROR:\n\tPath: #{el.attributes['path']}\n\tMessage: #{el.text}"
		}
		assert_nil results.elements["//error"].name if results.elements["//error"]
	end

	def test_axe_descendant
		assert_equal "f", XPath::first( @@doc, "descendant::f").name
		assert_equal "11", XPath::first( @@doc, "descendant::f[@a='c']").attributes['id']
		assert_equal "18", XPath::first( @@doc, "a/m//p" ).attributes['id']
		each_test( @@doc, "descendant-or-self::[@x='y']", 2 )
	end

	def test_axe_parent
		q = XPath.first( @@doc, "a/d/c/parent::/q" )
		assert_equal 19, q.attributes["id"].to_i
	end

	def test_axe_self
		c = XPath::first( @@doc, "a/b/c" )
		assert_not_nil c
		assert_equal "c", c.name
		assert_equal "c", XPath::first( c, "self::" ).name
	end

	def test_axe_ancestor
		doc = REXML::Document.new "
		<a>
			<b id='1'>
				<c>
					<b id='2'>
						<d/>
					</b>
				</c>
			</b>
		</a>"

		d = XPath.first( doc, "//d" )
		b = each_test( d, "ancestor::b", 2 ) { |el|
			assert((1..2) === el.attributes['id'].to_i, "Expected #{el.attributes['id']} to be either 1 or 2" ) 
		}
	end

	def test_axe_child
		m = XPath.first( @@doc, "a/child::m" )
		assert_equal 15, m.attributes['id'].to_i
	end

	def test_axe_attribute
		a = XPath.first( @@doc, "a/e/f[@id='14']/attribute::a" )
		assert_equal "d", a
	end

	def test_axe_sibling
		doc = Document.new "<a><b><c/></b><e><f id='10'/><f id='11'/><f id='12'/></e></a>"
		next_f = XPath.first( doc, "a/e/f[@id='11']/following-sibling::" )
		assert_equal 12, next_f.attributes['id'].to_i
		prev_f = XPath.first( doc, "a/e/f[@id='11']/previous-sibling::" )
		assert_equal 10, prev_f.attributes['id'].to_i
	end

	def test_lang
		doc = Document.new(File.new("test/lang.xml"))

		each_test( doc, "//language/*", 4 ) { |element|
			#puts "#{element.name}: #{element.text}"
		}
	end
end
