/* applyMove.h
 */
#ifndef _APPLY_MOVE_H
#define _APPLY_MOVE_H

#include "osl/apply_move/applyDoUndoXMove.h"
#include "osl/position.h"
#include "osl/move.h"
#include "osl/piece.h"
#include <boost/type_traits.hpp>

namespace osl
{
  namespace apply_move
  {
    /**
     * パス
     */
    struct ApplyPass
    {
      template <class State, class Func>
      static void doUndoPass(State& state, Func &f)
      {
	BOOST_STATIC_ASSERT(! boost::is_pointer<State>::value);
	state.changeTurn();
	f(Position::STAND());
	state.changeTurn();
      }
    };
    
    /**
     * doUndoMove を呼ぶためのインターフェース
     * 
     * ApplyMove<BLACK>::doUndoMove(state, move, function);
     * のように使う。
     * 各State は ApplyDoUndoXXXMove<State> の特殊化版を実装する
     */  
    template<Player P>
    struct ApplyMove
    {
      template <class State, class Func>
      static void doUndoMoveOrPass(State& state, Move move, Func &f)
      {
	BOOST_STATIC_ASSERT(! boost::is_pointer<State>::value);
	assert(state.getTurn() == move.player());
	if (move.isPass())
	  ApplyPass::doUndoPass(state, f);
	else
	  doUndoMove(state, move, f);
      }
      
      template <class State, class Func>
      static void doUndoMove(State& state, Move move, Func &f)
      {
	BOOST_STATIC_ASSERT(! boost::is_pointer<State>::value);
	assert(move.isValid());
	assert(state.isAlmostValidMove(move));
	assert(P == move.player());
	Position from=move.from();
	Position to=move.to();
	if (from.isPieceStand())
	{
	  assert(state.getPieceAt(to) == Piece::EMPTY());
	  ApplyDoUndoDropMove<P,State>::template doUndoDropMove<Func>(state,to,move.ptype(),f);
	}
	else
	{
	  assert(state.getPieceAt(from) != Piece::EMPTY());
	  Piece captured=state.getPieceAt(to);
	  if (captured != Piece::EMPTY())
	  {
	    ApplyDoUndoCaptureMove<P,State>::template doUndoCaptureMove<Func>
	      (state,from,to,captured,move.promoteMask(),f);
	  }
	  else
	  {
	    ApplyDoUndoSimpleMove<P,State>::template doUndoSimpleMove<Func>
	      (state,from,to,move.promoteMask(),f);
	  }
	}
      }
    };

    struct ApplyMoveOfTurn
    {
      /**
       * doUndoMove を呼ぶためのインターフェース
       * 
       * PASS可.
       * ApplyMoveOfTurn::doUndoMove(state, move, function);
       * のように使う。
       * ApplyMove と違って Player をtemplate 引数にする必要がない
       */  
      template <class State, class Func>
      static void doUndoMove(State& state, Move move, Func &f)
      {
	assert(state.getTurn() == move.player());
	if (move.isPass())
	{
	  ApplyPass::doUndoPass(state, f);
	  return;
	}
	if (move.player() == BLACK)
	  ApplyMove<BLACK>::doUndoMove(state, move, f);
	else
	  ApplyMove<WHITE>::doUndoMove(state, move, f);
      }

      /**
       * undo しない move を呼ぶ時の interface
       */
      template<typename State>
      static void doMove(State& state, Move move)
      {
	BOOST_STATIC_ASSERT(! boost::is_pointer<State>::value);
	assert(state.getTurn() == move.player());
	if (move.isPass()) {
	  state.makeMovePass();
	  return;
	}

	assert(state.isAlmostValidMove(move));
	const Position from=move.from();
	const Position to=move.to();
	if (from.isPieceStand())
	{
	  state.doDropMove(to,move.ptype());
	}
	else
	{
	  const Piece captured = state.getPieceOnBoard(to);
	  if (captured != Piece::EMPTY())
	  {
	    state.doCaptureMove(from,to,captured,move.promoteMask());
	  }
	  else
	  {
	    state.doSimpleMove(from,to,move.promoteMask());
	  }
	}
	state.changeTurn();
      }
    };
  } // namespace apply_move
  using apply_move::ApplyPass;
  using apply_move::ApplyMove;
  using apply_move::ApplyMoveOfTurn;
} // namespace osl


#endif /* _APPLY_MOVE_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
