/**
 * GUI Commands
 * Copyright 2005 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: CommandDelegate.java,v 1.6 2006/04/14 20:28:44 pietschy Exp $
 */
package org.pietschy.command.delegate;

import org.pietschy.command.ActionCommandExecutor;

import java.beans.PropertyChangeSupport;


import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

/**
 *
 * @author andrewp
 * @version $Revision: 1.6 $
 * @see DelegatingCommand
 * @see SwingActionDelegate
 */
public abstract class
CommandDelegate implements ActionCommandExecutor
{
   // Constants and variables
   // -------------------------------------------------------------------------
   private static final String _ID_ = "$Id: CommandDelegate.java,v 1.6 2006/04/14 20:28:44 pietschy Exp $";

   public static final String PROPERTY_ENABLED = "enabled";

   private String id;

   private PropertyChangeSupport pcs;

   private boolean enabled = true;

   /**
    * Creaets a new delegate with the specified id.  This delegate will be automatically used by any
    * {@link DelegatingCommand} with the same id.
    *
    * @param id the id of the {@link DelegatingCommand} to which this delegate should bind.
    */
   protected CommandDelegate(String id)
   {
      this.id = id;
      pcs = new PropertyChangeSupport(this);
   }

   /**
    * Gets the Id of this delegate.
    *
    * @return the delegates id.
    */
   public String
   getId()
   {
      return id;
   }

   /**
    * Checks if this delegate is enabled.
    * <p/>
    * This is a bound property, changes to its value will fire property change events.
    *
    * @return <tt>true</tt> if the delegate is enabled, <tt>false</tt> otherwise.
    */
   public boolean
   isEnabled()
   {
      return enabled;
   }

   /**
    * Sets the enabled state of the delegate.
    * <p/>
    * This is a bound property, changes to its value will fire property change events.
    *
    * @param enabled <tt>true</tt> if the delegate is enabled, <tt>false</tt> otherwise.
    */
   public void
   setEnabled(boolean enabled)
   {
      if (this.enabled != enabled)
      {
         boolean old = this.enabled;
         this.enabled = enabled;
         firePropertyChange(PROPERTY_ENABLED, old, enabled);
      }
   }

   public void
   addPropertyChangeListener(PropertyChangeListener listener)
   {
      pcs.addPropertyChangeListener(listener);
   }

   public void
   removePropertyChangeListener(PropertyChangeListener listener)
   {
      pcs.removePropertyChangeListener(listener);
   }

   public PropertyChangeListener[]
   getPropertyChangeListeners()
   {
      return pcs.getPropertyChangeListeners();
   }

   public void
   addPropertyChangeListener(String propertyName, PropertyChangeListener listener)
   {
      pcs.addPropertyChangeListener(propertyName, listener);
   }

   public void
   removePropertyChangeListener(String propertyName, PropertyChangeListener listener)
   {
      pcs.removePropertyChangeListener(propertyName, listener);
   }

   public PropertyChangeListener[]
   getPropertyChangeListeners(String propertyName)
   {
      return pcs.getPropertyChangeListeners(propertyName);
   }

   protected void firePropertyChange(String propertyName, int oldValue, int newValue)
   {
      pcs.firePropertyChange(propertyName, oldValue, newValue);
   }

   protected void firePropertyChange(PropertyChangeEvent evt)
   {
      pcs.firePropertyChange(evt);
   }

   protected void firePropertyChange(String propertyName, boolean oldValue, boolean newValue)
   {
      pcs.firePropertyChange(propertyName, oldValue, newValue);
   }

   protected void firePropertyChange(String propertyName, Object oldValue, Object newValue)
   {
      pcs.firePropertyChange(propertyName, oldValue, newValue);
   }

   private boolean areEqual(Object o1, Object o2)
   {
      return o1 == null ? o2 == null : o1.equals(o2);
   }
}
