/***************************************************************************
                          chublistmanager.cpp  -  description
                             -------------------
    begin                : Don Mai 16 2002
    copyright            : (C) 2002-2003 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <dclib/cclient.h>
#include <dclib/cconfig.h>
#include <dclib/core/cmanager.h>
#include <dclib/cdownloadmanager.h>
#include <dclib/dcobject.h>
#include <dclib/core/cbytearray.h>
#include <dclib/chttp.h>
#include <dclib/core/cbz.h>
#include <dclib/cfilemanager.h>
#include <dclib/cutils.h>

#include <dclib/core/ciconv.h>

#ifndef WIN32
#include <unistd.h>
#include <stdlib.h>
#endif

#include "chublistmanager.h"

/** */
CHubListManager::CHubListManager()
{
	m_pCallback = new CCallback<CHubListManager>( this, &CHubListManager::Callback );
	CManager::Instance()->Add( m_pCallback );

	if ( CConfig::Instance()->GetReloadHubListTime() != 0 )
	{
		m_nReloadHubListTimeout = time(0) + CConfig::Instance()->GetReloadHubListTime()*60*60;
	}
	else
	{
		m_nReloadHubListTimeout = 0;
	}

	// get hublist stuff
	m_pHttp           = 0;
	m_pHubListUrlList = 0;
	m_pHubListUrl     = 0;
	m_pHubListData    = 0;

	m_bGetHubListDone = FALSE;
	
	m_pXml = new CXml();

	SetInstance(this);
}

/** */
CHubListManager::~CHubListManager()
{
	m_Thread.Stop();

	SetInstance(0);

	CManager::Instance()->Remove( m_pCallback );

	if (m_pCallback)
	{
		delete m_pCallback;
		m_pCallback = 0;
	}
	
	if (m_pXml != 0)
	{
		delete m_pXml;
		m_pXml = 0;
	}
}

/** thread callbackfunction */
int CHubListManager::Callback( CObject *, CObject * )
{
	m_Thread.Lock();

	// check reload hublist timeout
	if ( CConfig::Instance() )
	{
		if ( CConfig::Instance()->GetReloadHubListTime() != 0 )
		{
			if ( m_nReloadHubListTimeout != 0 )
			{
				if ( time(0) >= m_nReloadHubListTimeout )
				{
					GetPublicHubList();

					m_nReloadHubListTimeout = time(0) + CConfig::Instance()->GetReloadHubListTime()*60*60;
				}
			}
			else // change from config
			{
				m_nReloadHubListTimeout = time(0) + CConfig::Instance()->GetReloadHubListTime()*60*60;
			}
		}
		else
		{
			m_nReloadHubListTimeout = 0;
		}
	}

	// cleanup all objects
	if ( m_bGetHubListDone == TRUE )
	{
		if ( m_pHttp )
		{
			delete m_pHttp;
			m_pHttp = 0;
		}

		if ( m_pHubListUrlList )
		{
			delete m_pHubListUrlList;
			m_pHubListUrlList = 0;
		}

		if ( m_pHubListData )
		{
			delete m_pHubListData;
			m_pHubListData = 0;
		}

		m_pHubListUrl = 0;

		DCMessageGetHubList * msg = new DCMessageGetHubList();

		msg->m_bRun = FALSE;

		if ( DC_CallBack(msg) == -1 )
		{
			delete msg;
		}

		m_bGetHubListDone = FALSE;
	}

	m_Thread.UnLock();

	return 0;
}

/** http callback function */
int CHubListManager::HttpCallBack( CObject *, CObject * object )
{
	CByteArray in,out;
	CBZ bz2;

	switch(((CDCMessage*)object)->m_eType)
	{
		case DC_MESSAGE_CONNECTION_STATE:
		{
			CMessageConnectionState *msg = (CMessageConnectionState*)object;

			if ( msg->m_eState == estDISCONNECTED )
			{
				if ( (m_pHttp->GetHttpError() == 200) && (m_pHttp->GetData(&in) == TRUE) )
				{
					if ( m_pHttp->GetUrl().Find(".bz2") != -1 )
					{
						if ( bz2.Decompress( &in, &out ) == TRUE )
						{
							if ( m_pHttp->GetUrl().Find(".xml.") != -1 )
							{
								if ( m_pXml->ParseMemory((const char*)out.Data(),out.Size()) == TRUE )
								{
									ParseXmlPublicHubList();
								}
								else
								{
									printf("Failed to parse XML hublist.\n");
								}
							}
							else
							{
								m_pHubListData->Append(out.Data(),out.Size());
								m_pHubListData->Append("\xD\xA",2);
							}
						}
						else
						{
							printf("bz2 decompress failed\n");
						}
					}
					else
					{
						if ( m_pHttp->GetUrl().Right(4) == ".xml" ) 
						{
							if ( m_pXml->ParseMemory((const char*)in.Data(),in.Size()) == TRUE )
							{
								ParseXmlPublicHubList();
							}
							else
							{
								printf("Failed to parse XML hublist.\n");
							}
						}
						else
						{
							m_pHubListData->Append(in.Data(),in.Size());
							m_pHubListData->Append("\xD\xA",2);
						}
					}
				}

				// redirect
				if ( m_pHttp->GetHttpError() == 302 )
				{
					m_pHttp->GetUrl(m_pHttp->GetLocation());
				}
				else if ( NextHubListUrl() == FALSE )
				{
					// parse public hub list in own thread
					CCallback<CHubListManager> * c = new CCallback<CHubListManager>( this, &CHubListManager::ParsePublicHubList );
					m_Thread.SetThreadCallBackFunction(c);
					m_Thread.Start();
				}
			}

			break;
		}

		case DC_MESSAGE_TRANSFER:
		{
			if ( DC_CallBack(object) != -1 )
			{
				object = 0;
			}

			break;
		}

		default:
		{
			break;
		}
	}

	if ( object )
	{
		delete object;
	}

	return 0;
}

/** */
bool CHubListManager::GetPublicHubList()
{
	bool res = FALSE;

	if ( m_pHttp != 0 )
	{
		// get hublist allready running
		return res;
	}

	m_pHubListUrlList = new CList<DCConfigHubListUrl>();

	// get all hublist urls
	CConfig::Instance()->GetHubListUrlList(m_pHubListUrlList);

	// check hublist url count
	if ( m_pHubListUrlList->Count() == 0 )
	{
		delete m_pHubListUrlList;
		m_pHubListUrlList = 0;
		return res;
	}

	// init first url
	m_pHubListUrl = 0;

	m_pHubListData = new CByteArray();
	m_pHttp = new CHttp();
	m_pHttp->SetCallBackFunction( new CCallback<CHubListManager>( this, &CHubListManager::HttpCallBack ) );

	res = NextHubListUrl();

	if ( !res )
	{
		m_bGetHubListDone = TRUE;
	}
	else
	{
		DCMessageGetHubList * msg = new DCMessageGetHubList();

		msg->m_bRun = TRUE;

		if ( DC_CallBack(msg) == -1 )
		{
			delete msg;
		}
	}

	return res;
}

/** */
int CHubListManager::ParsePublicHubList( CObject * , CObject * )
{
	CString line;
	CString s;
	CString s1,s2,s3,s4;
	long i=0,i1=0;

	if ( m_pHubListData->Size() == 0 )
	{
		m_Thread.Stop();
		m_Thread.SetThreadCallBackFunction(0);
		m_bGetHubListDone = TRUE;

		return 0;
	}

	s.Set((const char*)m_pHubListData->Data(),m_pHubListData->Size());

	while( (i = s.Find(0x0d,i)) != -1 )
	{
		line = s.Mid(i1,i-i1);

		if ( !line.IsEmpty() )
		{
			s1 = line.Section( '|', 0, 0 );
			s2 = line.Section( '|', 1, 1 );
			s3 = line.Section( '|', 2, 2 );
			s4 = line.Section( '|', 3, 3 );

			// replace all spaces
			s2 = s2.Replace(" ","");

			// remove line ends
			s1 = s1.Replace("\n", "");

			CConfig::Instance()->AddPublicHub( s1, s2, s3, s4 );
		}

		i1 = i+2;
		i += 2;
	}

	// store the list
	if ( CConfig::Instance()->GetHubListStoreLocal() == TRUE )
	{
		CConfig::Instance()->SaveDCHub();
	}

	m_Thread.Stop();
	m_Thread.SetThreadCallBackFunction(0);
	m_bGetHubListDone = TRUE;

	return 0;
}

/** */
bool CHubListManager::NextHubListUrl()
{
	bool res = FALSE;

	while( (m_pHubListUrl=m_pHubListUrlList->Next(m_pHubListUrl)) != 0 )
	{
		if ( m_pHubListUrl->bEnabled == TRUE )
		{
			if ( m_pHubListUrl->sUrl != "" )
			{
				m_pHttp->GetUrl(m_pHubListUrl->sUrl);

				res = TRUE;
				break;
			}
		}
	}

	return res;
}

/** */
int CHubListManager::ParseXmlPublicHubList()
{
	xmlNodePtr node, node2;
	int count = 0;
	CIconv * pIconv = new CIconv( "UTF-8", CConfig::Instance()->GetRemoteEncoding() );
	
	printf("Parse XML hub list...\n");
	
	for ( node=m_pXml->doc()->children; node != 0; node=node->next )
	{
		if ( m_pXml->name(node) == "Hublist" )
		{
			for ( node2 = node->children; node2 != 0; node2=node2->next )
			{
				if ( m_pXml->name(node2) == "Hubs" )
				{
					count += ParseXmlHubs(node2->children, pIconv);
				}
			}
		}
	}
	
	printf("XML hublist: %d hubs\n", count);
	
	if (count > 0)
	{
		m_pHubListData->Append("\xD\xA",2);
	}
	
	delete pIconv;
	
	return 0;
}

/** */
int CHubListManager::ParseXmlHubs( xmlNodePtr node, CIconv * pIconv )
{
	xmlNodePtr n1;
	CList<CXmlColumn> * cols = new CList<CXmlColumn>;
	int count = 0;
	
	for ( n1=node; n1 != 0; n1=n1->next )
	{
		if ( m_pXml->name(n1) == "Columns" )
		{
			ParseXmlColumns(n1->children, cols);
		}
		else if ( m_pXml->name(n1) == "Hub" )
		{
			ParseXmlHub(n1, cols, pIconv);
			count++;
		}
	}
	
	cols->Clear();
	delete cols;
	
	return count;
}

/** */
void CHubListManager::ParseXmlColumns( xmlNodePtr node, CList<CXmlColumn> * cols )
{
	xmlNodePtr n1;
	
	for ( n1=node; n1 != 0; n1=n1->next )
	{
		if ( m_pXml->name(n1) == "Column" )
		{
			CXmlColumn * col = new CXmlColumn();
			col->m_sName = m_pXml->prop(n1,"Name");
			col->m_sType = m_pXml->prop(n1,"Type");
			cols->Add(col);
		}
	}
}

/** */
void CHubListManager::ParseXmlHub( xmlNodePtr node, CList<CXmlColumn> * cols, CIconv * pIconv )
{
	CString name = "";
	CString server = "";
	CString desc = "";
	CString extra = " (";
	int user = 0;
	CXmlColumn * col = 0;
	CString line = "";
	
	while ( (col = cols->Next(col)) != 0 )
	{
		col->m_sValue = m_pXml->prop(node, col->m_sName);
		
		if (col->m_sType == "bytes")
		{
			col->m_sValue = CUtils::GetSizeString( col->m_sValue.asULL(), euAUTO );
		}
		
		if (col->m_sName == "Name")
		{
			name = col->m_sValue;
		}
		else if (col->m_sName == "Address")
		{
			server = col->m_sValue;
		}
		else if (col->m_sName == "Description")
		{
			desc = col->m_sValue;
		}
		else if (col->m_sName == "Users")
		{
			user = col->m_sValue.asINT();
		}
		else if (col->m_sName == "Port")
		{
			if (server.Find(':') == -1)
			{
				server = server + ":" + col->m_sValue;
			}
		}
		else
		{
			extra += col->m_sName + "=" + col->m_sValue + " ";
		}
		
		//printf("Name=%s Type=%s Value=%s\n", col->m_sName.Data(), col->m_sType.Data(), col->m_sValue.Data());
	}
	
	extra += ")";
	
	line = name + "|" + server + "|" + desc + extra + "|" + CString().setNum(user) + "|||||\n\r";
	
	line = pIconv->encode(line);
	
	m_pHubListData->Append(line.Data(),line.Length());
}
