/* 
**  utility.c
*/ 

#include "mod_mp3.h"

MP3_EXPORT(int) mp3_match(const char *s1, const char *s2) {
	if(!s1)
		return 1;	
	if(!s2)
		return 1;	

	return ap_fnmatch(s1, s2, FNM_CASE_BLIND);
}

MP3_EXPORT(char *) escape_xml(pool *p, const char *s) {
	int i, j;
	char *x;
	char *temp = NULL;

	if(!s)
		return NULL;

	for (i = 0, j = 0; s[i] != '\0'; i++) {
		if (s[i] == '<' || s[i] == '>')
			j += 3;
		else if (s[i] == '&')
			j += 4;
		else if (s[i] == '"')
			j += 4;
		else if (s[i] == '\'')
			j += 4;
		else if ((unsigned char)s[i] < 32)
			j += 6;
		else if ((unsigned char)s[i] > 126)
			j += 6;
	}

	if (j == 0)
		return ap_pstrndup(p, s, i);

	x = ap_palloc(p, i + j + 1);
	for (i = 0, j = 0; s[i] != '\0'; i++, j++) {
		if (s[i] == '<') {
			memcpy(&x[j], "&lt;", 4);
			j += 3;
		} else if (s[i] == '>') {
			memcpy(&x[j], "&gt;", 4);
			j += 3;
		} else if (s[i] == '&') {
			memcpy(&x[j], "&amp;", 5);
			j += 4;
		} else if (s[i] == '"') {
			memcpy(&x[j], "&quot;", 6);
			j += 5;
		} else if (s[i] == '\'') {
			memcpy(&x[j], "&apos;", 6);
			j += 5;
		} else if ((unsigned char)s[i] > 126) {
			temp = ap_psprintf(p,"&#%d;", (unsigned char)s[i]);
			memcpy(&x[j], temp, 6);
			j += 5;
		} else if ((unsigned char)s[i] < 32) {
			if(s[i] < 10)
				temp = ap_psprintf(p,"&#00%d;", (unsigned char)s[i]);
			else 
				temp = ap_psprintf(p,"&#0%d;", (unsigned char)s[i]);

				memcpy(&x[j], temp, 6);
				j += 5;
		} else {
			x[j] = s[i];
		}
	}

	x[j] = '\0';
	return x;
}

MP3_EXPORT(const char *) make_basename(request_rec *r) {
	return ap_pstrcat(r->pool, "http://", r->server->server_hostname, ":", (ap_psprintf(r->pool, "%d", r->server->port)), r->uri, NULL);
}

MP3_EXPORT(void) cleanup_scoreboard(void *data) {
	mp3_server_conf *cfg = (mp3_server_conf *)data;

	shmdt(cfg->board);
	shmctl(cfg->shmid, IPC_RMID, 0);
}

MP3_EXPORT(void) cleanup_file(void *data) {
  const char *filename = (const char *)data;
	unlink(filename);
}

/* This could be faster */
void cleanup_connection(void *rp) {
  request_rec *r = (request_rec *)rp;
	mp3_server_conf *scfg = ap_get_module_config(r->server->module_config, &mp3_module);

	scfg->board->servers[r->connection->child_num] = 0;
#ifdef DEBUG
	printf("Cleaning up %d(%d)\n", r->connection->child_num, scfg->board->servers[r->connection->child_num]);
#endif
}

MP3_EXPORT(int) register_connection(request_rec *r, mp3_server_conf *scfg, int limit, int type) {
	int active = 0;
	int x = 0;

#ifdef DEBUG
	printf("Connection limit %d\n", limit);
#endif
	if(limit) {
		for(x = 0; x < CHILDREN; x++) {
			if(scfg->board->servers[x]) {
#ifdef DEBUG
				printf("Server %d(%d) is active \n", x, scfg->board->servers[x]);
#endif
				active++;
			}
		}
		if(active >= limit) {
			return HTTP_FORBIDDEN;
		}
	}
		
	scfg->board->servers[r->connection->child_num] = type;
	ap_register_cleanup(r->pool, r, cleanup_connection, ap_null_cleanup);

	return 0;
}

MP3_EXPORT(array_header *) get_songs(pool *p, table * t) {
	array_header *songs  = NULL;
	array_header *hdrs_arr;
	table_entry *elts;
	int i;

	if (t == NULL)
		return NULL;

	songs  = ap_make_array (p, 5, sizeof (char *));

	hdrs_arr = ap_table_elts(t);
	elts = (table_entry *) hdrs_arr->elts;

	for (i = 0; i < hdrs_arr->nelts; ++i) {
		if(!strcasecmp("song", elts[i].key)) {
			*(char **) ap_push_array (songs) = ap_pstrdup (p, elts[i].val );
		}
	}

	if(songs->nelts) {
		return songs;
	} else {
		return NULL;
	}
}

MP3_EXPORT(table *) parse_args(request_rec *r) {
	const char *data = NULL;
	const char *keypair = NULL;
	const char *key = NULL;
	const char *value = NULL;
	table *arguments;
	int size=0;
	int x=0;

	if(r->args) {
		arguments = ap_make_table(r->pool, 4);
		data = ap_pstrdup(r->pool, r->args);
		while(*data && (keypair = ap_getword(r->pool, &data, '&'))){
			if((x = ap_ind(keypair, '=')) != -1) {
				size = strlen(keypair);
				value = (keypair) + (x+1);	
				key = ap_pstrndup(r->pool, keypair, x);
				ap_table_add(arguments, key, value);
			} else {
				ap_table_add(arguments,keypair, "");
			}
		}
	} else {
		return NULL;
	}

	return arguments;
}


MP3_EXPORT(void) table_list(char *string, table * t) {
	array_header *hdrs_arr;
	table_entry *elts;
	int i;

	if (t == NULL)
		return;
	if (string == NULL)
		return;

	hdrs_arr = ap_table_elts(t);
	elts = (table_entry *) hdrs_arr->elts;

	printf("<-- BEGIN %s -->\n", string);
	for (i = 0; i < hdrs_arr->nelts; ++i) {
		printf("%s: %s\n", elts[i].key, elts[i].val);
	}
	printf("<-- END %s -->\n", string);
}
