/*-
 * C-SaCzech
 * Copyright (c) 1996-2002 Jaromir Dolecek <dolecek@ics.muni.cz>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Jaromir Dolecek
 *	for the CSacek project.
 * 4. The name of Jaromir Dolecek may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JAROMIR DOLECEK ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL JAROMIR DOLECEK BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: csacek.h,v 1.243.2.1 2002/03/03 10:47:01 dolecek Exp $ */

#ifndef _CSA_CSACEK_H_
#define _CSA_CSACEK_H_

#include "csa_config.h"

#include <sys/types.h>

#if defined(_AIX) && defined(__GNUC__)
/* under AIX 4.2, blkcnt_t is defined in /usr/include/sys/types.h and
 * used by <sys/stat.h>
 * gcc doesn't define it in it's own <sys/types.h> and anything using
 * <sys/stat.h> fails to compile; so define it the way it's defined
 * in system <sys/types.h> */
typedef long blkcnt_t;
#endif /* AIX && GNU C */

#include <sys/stat.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>

#ifdef CSA_MUTACE_FASTCGI
  /* fcgi library; put it first */
# ifdef HAVE_FCGI_STDIO_H
#  include <fcgi_stdio.h>
# else
#  include "fastcgi/fcgi_stdio.h"
# endif /* HAVE_FCGI_STDIO_H */
#else  /* ! CSA_MUTACE_FASTCGI */
# include <stdio.h>
#endif /* CSA_MUTACE_FASTCGI */

/* ensure CSA_MUTACE_APACHE is defined when compiling under Apache */
#if defined(MODULE_MAGIC_NUMBER) && !defined(CSA_MUTACE_APACHE)
#define CSA_MUTACE_APACHE
#endif

#ifdef CSA_MUTACE_APACHE
#if MODULE_MAGIC_NUMBER < 19970526
#  define CSA_USE_APACHE11_API
#elif MODULE_MAGIC_NUMBER >= 19980413
#  define CSA_USE_APACHE13_API
#else
#  define CSA_USE_APACHE12_API
#endif /* MODULE_MAGIC_NUMBER */
#endif /* CSA_MUTACE_APACHE */

/* define this if it's not defined yet */
#ifndef CSACEK
#  define CSACEK		1
#endif

#ifdef CSA_DEBUG
#ifndef CSA_ALLOW_CACHE
/* turn CSA_DO_NOT_CACHE on for debugging */
#ifndef CSA_DO_NOT_CACHE
#  define CSA_DO_NOT_CACHE
#endif
#endif /* !CSA_ALLOW_CACHE */
#endif /* CSA_DEBUG */

#if defined(CSA_MUTACE_CGI) || defined(CSA_MUTACE_FASTCGI)
#  define CSA_MUTACE_XCGI
#endif

/* CSacek constants */
#include "csa_defaults.h"

/* csa_compat.h uses csa_params_t in prototypes */
typedef struct csa_params csa_params_t;
#include "csa_compat.h"

/* encoding engine */
#include "cstools.h"

#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif /* HAVE_SYS_TIME_H */
#endif /* TIME_WITH_SYS_TIME */ 

#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif

/* general defines */

#ifdef CSA_MUTACE_CGI
#  define CSA_MUTACE_NAME   "CGI"
#elif defined(CSA_MUTACE_FASTCGI)
#  define CSA_MUTACE_NAME   "FastCGI"
#elif defined(CSA_MUTACE_APACHE)
# if defined(CSA_USE_APACHE11_API)
#  define CSA_MUTACE_NAME   "Apache 1.1 Module"
# elif !defined(CSA_USE_APACHE13_API)
#  define CSA_MUTACE_NAME   "Apache 1.2 Module"
# else
#   ifdef EAPI
#     define CSA_MUTACE_NAME   "Apache 1.3+ EAPI Module"
#   else
#     define CSA_MUTACE_NAME   "Apache 1.3+ Module"
#   endif /* EAPI */
# endif
#elif defined(CSA_MUTACE_ISAPI)
#  define CSA_MUTACE_NAME   "ISAPI Filter"
#elif defined(CSA_MUTACE_FILE)
#  define CSA_MUTACE_NAME   "File (benchmarking)"
#endif

/* values returned by some CSacek functions */
#define CSA_CONTINUE		6
#define CSA_NEED_MORE		5
#define CSA_INFO		4
#define CSA_WHICHCODE		3
#define CSA_SERVED		2
#define CSA_DONE		1
#define CSA_OK			0
#define CSA_FAILED		-1
#define CSA_FATAL		-2

/* maximum length of substitute string in csa_find_subs() - do not forget
 * to update this value when changing csa_find_subs() */
#define CSA_FINDSUBS_MAX	16

/* number of bytes to check for JavaScript indicies when TESTJS is on */
#define CSA_TESTJSBYTES		200

/* constants used to mark type of items in csa_arg_t structure */
#define CSA_P_STRING 	1
#define CSA_P_EQUATION	2

/* values used as flag values for csa_setitem() */
#define CSA_I_IFNOTSET		1  /* set item only when it doesn't exist yet */
#define CSA_I_HEADERCAT		2  /* add value to old value of the item, */
				   /* separating them by comma */
#define CSA_I_COPYKEY		8  /* key will be copied instead of linking */
#define CSA_I_COPYVALUE		16 /* value will be copied instead of linking */
#define CSA_I_TMP		32 /* use temporary pool for allocation */
#define CSA_I_OVERWRITE		64 /* overwrite old value of item */

/* values used as flags to csa_add_output() */
#define CSA_OUT_RAW		1 /* insert data directly into output, with  */
				  /* no compression or any other conversions */
#define CSA_OUT_STR		2 /* ignore len parameter and get len as */
				  /* strlen() */

/* these constants are used by CSacek cmds cmd function to find out which */
/* command has been used */
#define CSA_PART		1
#define CSA_DOMAIN		2
#define CSA_CHARSET		3
#define CSA_EXECCMDS		4
#define CSA_NOEXECCMDS		5

/*
 * This is used by several CSacek commands to indicate to csa_ChangeURL()
 * which parameter holds the URL
 */
#define CSA_HREF		1
#define CSA_SRC			2
#define CSA_ACTION		3

/*
 * Misc. flags
 */
#define CSA_FL_HEADERS_READING	0x0000001 /* set if reading resp.headers */
#define CSA_FL_HEADERS_READ	0x0000002 /* set if all response headers read */
#define CSA_FL_HEADERS_SENT	0x0000004 /* resp. headers sent to client */
#define CSA_FL_HEADERS_ONLY	0x0000008 /* req. method has been HEAD */
#define CSA_FL_OUT_CHUNKED	0x0000010 /* send output chunked */
#define CSA_FL_OUT_CONT		0x0000020 /* send output continually */
#define CSA_FL_CHUNKED_RESP	0x0000040 /* response was originally chunked */
#define CSA_FL_CHUNK_FINISHED	0x0000080 /* finished reading one chunk */
#define CSA_FL_CHUNK_ALL	0x0000100 /* read all chunks */
#define CSA_FL_CONVERT		0x0000200 /* do anything with document ? */
#define CSA_FL_ISHTML		0x0000400 /* is the doc. text/html ? */
#define CSA_FL_ISHTTPS		0x0000800 /* SSL used */
#define CSA_FL_NOEXECCMDS	0x0001000 /* exec only "core" CSacek commands */
#define CSA_FL_CHANGEURL	0x0002000 /* rewrite URL to go through CSacek */
#define CSA_FL_RECODEINPUT	0x0004000 /* recode input data to server enc. */
#define CSA_FL_PART_IS_DEF	0x0008000 /* part name is the default one */
#define CSA_FL_PART_IS_EMPTY	0x0010000 /* part name was not specified */
#define CSA_FL_C_INITIALIZED	0x0020000 /* compression engine initialized */
#define CSA_FL_NOHTML		0x0040000 /* disable HTML tag processing */
#define CSA_FL_VARY_UA		0x0080000 /* Vary by User-Agent */
#define CSA_FL_VARY_AL		0x0100000 /* Vary by Accept-Language */
#define CSA_FL_VARY_AC		0x0200000 /* Vary by Accept-Charset */
#define CSA_FL_COMPRESSABLE	0x0400000 /* result data can be compressed */
#define CSA_FL_CLRTMPPOOL	0x0800000 /* temp pool needs to be cleared */
#define CSA_FL_TESTJS		0x1000000 /* check if document is JavaScript */

/* 
 * CSacek parts related flags.
 * The CSA_VALID_FOO flag is set if we are currently processing part FOO,
 * CSA_VALIDOTHER_FOO is set if part name "OTHER" would still has any sense.
 */
#define CSA_VALID_CHARSET	0x001
#define CSA_VALID_DOMAIN	0x002
#define CSA_VALID_PART		0x004
#define CSA_VALID_MASK	(CSA_VALID_CHARSET|CSA_VALID_DOMAIN|CSA_VALID_PART)
#define CSA_VALIDOTHER_CHARSET	0x010
#define CSA_VALIDOTHER_DOMAIN	0x020
#define CSA_VALIDOTHER_PART	0x040
#define CSA_VALID_2OTHER(x)	((x)<<4)
#define CSA_VALIDOTHER_MASK	CSA_VALID_2OTHER(CSA_VALID_MASK)

/* content type of form data */
#define CSA_CT_FORMDATA		"application/x-www-form-urlencoded"

/* CSacek macros */

/* returns true if data should be directly forwarded to client */
#define CSA_SHOULD_DIRECT_FORWARD(p)  (!CSA_ISSET(p->flags, CSA_FL_CONVERT) && \
	(p->available_in >= 0 || CSA_ISSET(p->flags, CSA_FL_CHUNKED_RESP) \
	 || p->status_no == 304))

/* returns non-zero if we can send headers along request - i.e. client used */
/* HTTP/1.0+ */
#define CSA_SEND_HEADERS(p) \
	(p->protocol > 9 && !CSA_ISSET(p->flags, CSA_FL_HEADERS_SENT))

/* support for Apache-SSL */
#define CSA_METHOD(p)	(CSA_ISSET(p->flags, CSA_FL_ISHTTPS) ? "https" : "http")

#ifdef HAVE_TOUPPER
#define	CSA_UPPER(c)	toupper((unsigned char) (c))
#else
/* to get the upper char, we need to mask out the 0x20 bit */
#define CSA_UPPER(c)	( (c) >= 'a' && (c) <= 'z' ? (c) & 0xdf : (c) )
#endif

#define CSA_ISHEXA(x) ( ((x) >= '0' && (x) <= '9') || ( (CSA_UPPER(x) >= 'A') && (CSA_UPPER(x) <= 'F') ) )
#define CSA_HEX2DEC(x) ((x) >= 'A' ? (((x) & 0xdf) - 'A')+10 : ((x) - '0'))

/* macros to checking and modifying (bit) flags */
#define CSA_ISSET(x, y)		(((x) & (y)) == (y))
#define CSA_ISSET_ANY(x, y)	((x) & (y))
#define CSA_SET(x, y)		((x) |= (y))
#define CSA_UNSET(x, y)		((x) &= ~(y))

/* system dependant value - maximum count of numbers given type is able */
/* to display when writen in decimal */
#define CSA_GETMAXNUMCOUNT(x)	((int) (sizeof(x) * 256 / 100))

/* comparison of two char values with explicit cast to unsigned chars */
/* to avoid portability problems */
#define CSA_CHCMP(x, y)	((unsigned char) x == (unsigned char) y)

#ifdef HAVE_ALLOCA
#define csa_alloca(len, p)	alloca(len)
#else
#define csa_alloca(len, p)	ap_palloc(p, (int) len)
#endif

#define CSA_FILLSTRING(str, val, slen, smaxlen)	\
		(str)->value = (val);		\
		(str)->len = (slen);		\
		(str)->maxlen = (smaxlen);	\
			
/* CSacek structures & enums */

/* this holds value of various variables and random strings */
typedef struct {
	const char	*value;
	size_t	len, maxlen;
} csa_String;

typedef struct {
	char	*value;
	size_t	len, maxlen;
} csa_String_b;

/* struct holding information mandatory for command BAR to work */
/* implemented in bar.c */
typedef struct csa_bar csa_bar_t;

/* simple queue implementation - see share.c */
typedef struct csa_queue {
	char *value;
	size_t len, maxlen;
        struct csa_queue *next, *prev;    
} csa_queue_t;

/* type of function, called as implementation of CSacek command */
typedef int csa_cmdfunc_t __P((csa_params_t *, void *));

/* opaque pointers for passing argument data - see cmds.y for usage */
typedef struct csa_arg csa_arg_t;
typedef struct csa_arglist csa_arglist_t;
	
/* used to hold state while processing a command  - see loop.c */
typedef struct csa_yy csa_yy_t;

/* used for implementation of lists in share.c */
typedef struct csa_item {
        csa_String	key;
        csa_String	value;
        struct csa_item *prev, *next;
} csa_item_t;

/* used to implement list of servers for which we do implicit */
/* rewriting of URL to go through CSacek */
typedef struct csa_slist csa_slist_t;

/* this structure is used by csa_[un]parse_url() */
typedef struct {
	const char	*method;	/* method, such as "http" */
	const char	*server;	/* server name (NULL if not spec.) */
	unsigned short	 port;		/* port (0 if not specified) */
	const char	*csacek;	/* CSacek part of URI */
	const char	*uri;		/* URI w/o CSacek */
	const char	*qs;		/* query string */
	const csa_String *csa_dir;	/* CSacek dir (URI before CSacek)*/
	const char	*csa_part;	/* CSacek part specified in URI */
	int can_rewrite;		/* can rewrite CSacek part */
} csa_url_t;

/* this is used for implementation of Ranges */
typedef struct {
       signed int from, to;
} csa_range_t;

/* types of compressions supported */
typedef enum {
	CSA_C_NONE,	/* no compression */
	CSA_C_DEFLATE,	/* deflate */
	CSA_C_GZIP,	/* gzip = header + deflate + crc */
	CSA_C_COMPRESS	/* standard Unix compress(1) */
} csa_compress_t;

/* structure for storing the return of csa_yyparse() */
struct csa_yyres {
	const char *s;
	int opaque;
};

/* this holds state of request */
struct csa_params {
	void	*m_cookie;		/* mutation dependant data */

	struct pool	*pool_req; /* this gets cleaned on the end of req */
	struct pool	*pool_tmp; /* temporary pool */

	csa_FILE_t	*resp;	   /* response data stream */

	cstools_t	 incharset;
	cstools_t	 outcharset;

	csa_bar_t	*Bar;	    /* settings of values for BAR printing */
	cstools_cstocs_t mp;	    /* translation table */

	csa_String_b	*body_buf;	/* buffer used in csa_process_body() */ 

#ifdef CSA_DEBUG
	FILE		*dbg;		/* debug log */
#endif

#ifdef CSA_WANT_COMPRESSION
	/* compression related bits */
	csa_compress_t	 compress; 	/* used compression */
	csa_compress_t	 ua_compress; 	/* compression requested by UA */
	void		*c_cookie;	/* hold state in compression */
	unsigned long	 crc;		/* CRC for gzip */
#endif

	/* input */
	csa_item_t	*headersin;	/* headers sent by client */
	csa_item_t	*vars;		/* variables used within CSacek */
	int		 protocol;	/* HTTP protocol used for this req*10 */
					/* HTTP/0.9 --> 9 and so on */
	int		 req_protocol;	/* protocol used by CSacek for the
					 * subrequest - normally same as
					 * p->protocol */

	/* output related bits */
	const char	*status_raw;	/* Status as sent by server */
	int		 status_no;	/* number of returned Status */
	csa_item_t	*headersout;	/* output headers */
	csa_queue_t	*output;	/* output data */
	signed int	 available_in;  /* amount on data in sub-request */
					/* or -1 if it's unknown */
	size_t		 content_length;/* output data length */

	/* Range support bits */
	csa_range_t	**range;	/* parsed contents of Range header,  */
					/* if client sent it */
	const char	*if_range;	/* If-Range header after stripping */
					/* CSacek encoding number */

	/* chunked Transfer-Encoding support bits */
	size_t		 chunk_remaining;

	int		 flags;		/* misc flags */
	int		 flags_parts;	/* CSacek parts related flags */
	
	/* servers which use CSacek and we are allowed to "mangle" URL to go */
	/* through CSacek, preserving encoding&language preferences of client*/
	csa_slist_t	*csacek_servers;

	/* used to hold state during cmd parsing */
	csa_yy_t	*yy;

	/* a few often needed values -- needed mainly in csa_find_subs() */
	const csa_String	*csacek;
	csa_String		charset;
	csa_String		part;
	csa_String		lampacharset;
	const csa_String	*csacek_dir;
	const csa_String	*ignoreprefix;
	const char		*csa_suffix; /* ignored, but on some OSes needed
					      * suffix of CSacek, such as .cgi*/

	const char		*dd;	/* default partname */

	/* auxiliary pointer to hold a value for a while (used in */
	/* implementation of some CSacek commands) */
	const csa_String **tmp_strp;
};

/* flags for csa_conf_t */
#define	CSA_CFG_ENGINEON	0x001
#define	CSA_CFG_RECODEINPUT	0x002
#define	CSA_CFG_COMPRESS	0x004
#define	CSA_CFG_CHANGEURL	0x008
#define	CSA_CFG_IMPLICITWORK	0x010
#define	CSA_CFG_TESTJS		0x020	/* Test presence of JavaScript */

/* options set in server configuration */
typedef struct csa_conf {
	int		 flags;
	const char	*DefaultCharset;
	const char	*DefaultPartname;
	const char	*TemplateDir;
	const char	*BarDef;
	const char	*IgnorePrefix;
} csa_conf_t;

/* CSacek API functions */

__CSA_BEGIN_DECLS

#ifndef CSA_MUTACE_APACHE
/* apache/alloc.c or Apache itself */
struct pool *ap_init_alloc __P((void));
void ap_clear_pool __P((struct pool *));
void ap_destroy_pool __P((struct pool *));
void csa_done_alloc __P((void));
struct pool *ap_make_sub_pool __P((struct pool *));
void ap_note_cleanups_for_file __P((struct pool *, FILE *));
void *ap_palloc __P((struct pool *, int size));
void *ap_pcalloc __P((struct pool *, int size));
char *ap_pstrdup __P((struct pool *, const char *));
char *ap_pstrndup __P((struct pool *, const char *, int len));
void ap_register_cleanup __P((struct pool *, void *data,
		void (*plain_cleanup)(void *), void (*child_cleanup)(void *) ));
#endif /* !CSA_MUTACE_APACHE */

#ifdef CSA_MUTACE_APACHE

/* apache_eapi.c */
void csa_eapi_init __P((void));
void csa_eapi_addmodule __P((module *));
void csa_eapi_removemodule __P((module *));
void csa_eapi_newconnection __P((conn_rec *));
void csa_eapi_closeconnection __P((conn_rec *));

#endif /* CSA_MUTACE_APACHE */

/* bar.c */
csa_bar_t *csa_bar_create __P((struct pool *));
int csa_bar_printalways __P((const csa_bar_t *));
csa_cmdfunc_t csa_Bar;
csa_cmdfunc_t csa_BarDef;

/* cmds.[yc] */
const char *csa_arg_getkey __P((const csa_arg_t *));
const char *csa_arg_getvalue __P((const csa_arg_t *));
int csa_arg_getflags __P((const csa_arg_t *));
const csa_arg_t *csa_arg_peek __P((csa_params_t *));
const csa_arg_t *csa_arg_take __P((csa_params_t *));
csa_arglist_t *csa_arg_newlist __P((struct pool *));
void csa_arg_clrlist __P((csa_arglist_t *));

/* compress.c */
int csa_done_compress __P((csa_params_t *));
int csa_compress __P((csa_params_t *p, const char *buf, int len));
int csa_init_compress __P((csa_params_t *));

#ifdef CSA_DEBUG
/* debug.c */
void csa_debug __P((FILE *dbg, const char * fmt, ...));
FILE *csa_debug_start __P((void));	/* called on start of any session */
void csa_debug_end __P((FILE *));	/* called on end of any session */
void csa_debug_register_cleanup __P((struct pool *, FILE *));
#endif

/* guess.c */
void csa_toguess __P((csa_params_t *p, int redirect_client));

/* http.c */
char   *csa_construct_url __P((csa_params_t *, const char *script_name,
				const char *extra_path));
void csa_decodequery __P((csa_String *, csa_params_t *p, char *query,
				size_t len));
const char *csa_gethostbyaddr __P((struct pool *, const char *));
const char *csa_get_ct __P((struct pool *, cstools_t, const char *));
int csa_getmethodport __P((const char *));
void	csa_http_error __P((csa_params_t *p, const char *title,
				const char *body));
csa_item_t *csa_make_headersin __P((csa_params_t *p));
int	 csa_parse_sn __P((struct pool *, const char *, char **, char **,
				char **, char **, int *));
csa_url_t *csa_parse_url __P((csa_params_t *, const char *));
csa_range_t **csa_range_compile __P((struct pool *, const char *));
void csa_range_fixup __P((csa_range_t **, size_t final_content_length));
int	 csa_read_response __P((csa_params_t *, char *, size_t));
char	*csa_unparse_url __P((csa_params_t *, const csa_url_t *, const char *));
int	 csa_process_headers __P((csa_params_t *));
int  csa_split_header __P((struct pool *pool, char *line, char **header,
		char **value, char **val, char **options));

/* info.c */
int csa_info __P((csa_params_t *, const csa_conf_t *));
void csa_version __P((void));

/* loop.c */
int csa_add_recode_output __P((csa_params_t *p, char *str, size_t len,
				csa_String_b *buf));
int csa_add_subs_output __P((csa_params_t *, csa_String_b *, size_t, int));
int csa_cmd_execute __P((csa_params_t *));
int csa_process_body __P((csa_params_t *));
int csa_run_cmd __P((csa_params_t *, const char *, size_t, int));
csa_arglist_t *csa_yy_getarglist __P((csa_yy_t *));
const char *csa_yy_getcmdname __P((csa_yy_t *));
int csa_yy_getcmdparammask __P((csa_yy_t *));
int csa_yy_gettags __P((csa_params_t *, const char **, const char **));
int csa_yyparse __P((void *params));
int csa_yylex __P((csa_params_t *, void *));

/* servlist.c */
csa_slist_t *csa_slist_init __P((struct pool *));
void csa_cs_slist_init __P((struct pool *));
int	 csa_slist_add __P((csa_slist_t *, const char *, int));
int	 csa_is_csacek_server __P((const csa_slist_t *, const char *, int));
#define CSA_IS_CSACEK_SERVER(p, urlt) \
	(csa_is_csacek_server(p->csacek_servers, urlt->server, urlt->port) \
	  || (csacek_servers &&						   \
	      csa_is_csacek_server(csacek_servers, urlt->server, urlt->port)))
void	csa_slist_dump __P((csa_params_t *));

/* share.c */
extern const csa_conf_t csa_cfg_def;
extern csa_slist_t *csacek_servers;
csa_cmdfunc_t csa_ChangeURL;
csa_cmdfunc_t csa_DocParts;
csa_cmdfunc_t csa_Doctype;
csa_cmdfunc_t csa_Font;
csa_cmdfunc_t csa_Meta;
csa_cmdfunc_t csa_MyCharset;
csa_cmdfunc_t csa_Set;
csa_cmdfunc_t csa_Script;
csa_cmdfunc_t csa_Xml;
csa_cmdfunc_t csa_csacekServers;
void csa_alloc_fail __P((void));
int  csa_add_output __P((csa_params_t *p, const char *str, size_t l,int flags));
void csa_add_servers __P((struct pool *, csa_slist_t *, char *, size_t));
void csa_direct_forward __P((csa_params_t *));
void csa_flush_output __P((csa_params_t *));
const csa_String * csa_getitem __P((csa_item_t *il, const char *name));
int csa_init_params __P((csa_params_t *, struct pool *, void *,
				const csa_conf_t *, FILE *));
int csa_process_whichcode_file __P((csa_params_t *, const char *));
int csa_output __P((csa_params_t *));
int csa_send_body __P((csa_params_t *));
int csa_send_headersout __P((csa_params_t *));
int csa_set_headersin __P((csa_params_t *));
int csa_setitem __P((csa_params_t *p, csa_item_t **il, const char *name,
			const char *value, int flags));
void csa_unsetitem __P((csa_item_t **item_list, const char *name));
int csa_switch_incharset __P((csa_params_t *, cstools_t));

#define csa_getvar(p, x)	csa_getitem((p)->vars, (x))
#define csa_setvar(p, x, y, z)	csa_setitem(p, &((p)->vars), (x), (y) , CSA_I_OVERWRITE|(z))
#define csa_unsetvar(p, x)	csa_unsetitem(&((p)->vars), (x))
#define csa_getheaderout(p, x) csa_getitem((p)->headersout, (x))
#define csa_setheaderout(p, x, y, z) csa_setitem(p, &((p)->headersout), (x), (y), (z))
#define csa_unsetheaderout(p, x) csa_unsetitem(&((p)->headersout), (x))
#define csa_getheaderin(p, x) csa_getitem((p)->headersin, (x))
#define csa_setheaderin(p, x, y, z) csa_setitem(p, &((p)->headersin), (x), (y), (z))

/* string.c */
void csa_fillstring __P((csa_String *, const char *, int, int));
size_t csa_find_subs __P((csa_params_t *, const char *, size_t, size_t *,
			const csa_String **));
const char * csa_has_suffix __P((const char *str, const char *list, int sep));
csa_String *csa_createstring __P((struct pool *, const char *));
int csa_n_strcmp __P((const char *s1, const char *s2));
const char *csa_strcasestr __P(( const char *b1, const char *b2 ));
char *csa_subs_string __P((csa_params_t *, char *));

/* zlib.c */
int csa_init_deflate __P((csa_params_t *p));
int csa_done_deflate __P((csa_params_t *p));
int csa_init_gzip __P((csa_params_t *p));
int csa_done_gzip __P((csa_params_t *p));
int csa_deflate __P((csa_params_t *p, const char *buf, size_t len));

/* MD functions -- each mutation has to implement this it's own way */
void csa_md_alloc_fail __P((void));
int csa_md_call_whichcode __P((csa_params_t *, const char *));
const char *csa_md_getvalueof __P((csa_params_t *, const char *));
int csa_md_log_error __P((csa_params_t *, const char *));
int csa_md_read_response __P((csa_params_t *, char *, size_t));
void csa_md_send_header __P((csa_params_t *p, const char *header_name,
				const char *header_value));
int csa_md_send_output __P((csa_params_t *p, const char *dta, size_t len));
int csa_md_send_separator __P((csa_params_t *)); 
int csa_md_set_headersin __P((csa_params_t *));

__CSA_END_DECLS

#endif /* _CSA_CSACEK_H_ */

