/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     element_1d.c                                                   */
/*                                                                          */
/*                                                                          */
/* description:  routines on elements that depend on the dimension in 1d    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

/****************************************************************************/
/*  world_to_coord_1d():  return -1 if inside, otherwise index of lambda<0  */
/****************************************************************************/

int world_to_coord_1d(const EL_INFO *el_info,
		      const REAL *x, REAL_B lambda)
{
  FUNCNAME("world_to_coord_1d");
  REAL lmin;
  REAL a = el_info->coord[0][0];
  REAL length = (el_info->coord[1][0] - a);
  REAL det = ABS(length);
  int i, j, k;

#if (DIM_OF_WORLD != 1)
  ERROR_EXIT("not yet for DIM_OF_WORLD != 1\n");
#endif

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->world_to_coord() to access the parametric mesh\n");

  if (det < 1.E-20) {
    ERROR_EXIT("length = %le; abort\n", length);
    return(-2);
  }

  lambda[1] = (x[0]-a)/length;
  lambda[0] = 1.0 - lambda[1];

  k = -1;
  lmin = 0.0;
  j = 0;
  for (i = 0; i <= 1; i++) {
    if ((lambda[i]*det) < -1.E-15) {
      if (lambda[i] < lmin) {
        k = i;
        lmin = lambda[i];
      }
      j++;
    }
  }

  return(k);
}

/****************************************************************************/
/*  transform local coordintes l to world coordinates; if w is non nil      */
/*  store them at w otherwise return a pointer to some local static         */
/*  area containing world coordintes                                        */
/****************************************************************************/

const REAL *coord_to_world_1d(const EL_INFO *el_info, const REAL *l, REAL_D w)
{
  FUNCNAME("coord_to_world_1d");
  static REAL world[DIM_OF_WORLD];
  REAL        c, *ret;
  const REAL  *v;
  int         i, j;

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->coord_to_world() to access the parametric mesh\n");

  ret = w ? w : world;

  v = el_info->coord[0];
  c = l[0];
  for (j = 0; j < DIM_OF_WORLD; j++)
    ret[j] = c*v[j];
  
  for (i = 1; i < N_VERTICES_1D; i++)
  {
    v = el_info->coord[i];
    c = l[i];
    for (j = 0; j < DIM_OF_WORLD; j++)
      ret[j] += c*v[j];
  }
  return((const REAL *) ret);
}

/*--------------------------------------------------------------------------*/
/*  compute volume of an element                                            */
/*--------------------------------------------------------------------------*/

REAL el_det_1d(const EL_INFO *el_info)
{
  FUNCNAME("el_det_1");

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->det() to access the parametric mesh\n");

  return(DIST_DOW(el_info->coord[0], el_info->coord[1]));
}

REAL el_volume_1d(const EL_INFO *el_info)
{
  return el_det_1d(el_info);
}

/*--------------------------------------------------------------------------*/
/*  compute gradients of barycentric coords on element; return the absolute */
/*  value of the determinant of the Jacobian of the transformation to the   */
/*  reference element                                                       */
/*  Notice: for dim < DIM_OF_WORLD grd_lam will contain the tangential      */
/*  derivative of the barycentric coordinates!                              */
/*--------------------------------------------------------------------------*/

REAL el_grd_lambda_1d(const EL_INFO *el_info,
		      REAL grd_lam[N_LAMBDA][DIM_OF_WORLD])
{
  FUNCNAME("el_grd_lambda_1d");
  REAL   det2;
  int i;

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->grd_lambda() to access the parametric mesh\n");

  det2  = DST2_DOW(el_info->coord[0], el_info->coord[1]);

  for(i = 0; i < DIM_OF_WORLD; i++) {
    grd_lam[0][i] = (el_info->coord[0][i] - el_info->coord[1][i]) / det2;
    grd_lam[1][i] = -grd_lam[0][i];
  }

  return(sqrt(det2));
}


/*---8<---------------------------------------------------------------------*/
/*---  wall normal in 1d is really rather easy....                       ---*/
/*--------------------------------------------------------------------->8---*/

REAL get_wall_normal_1d(const EL_INFO *el_info, int i0, REAL *normal)
{
#if DIM_OF_WORLD == 1
  normal[0] = i0 ? -1.0 : 1.0;
#else
  int          n;
  REAL         det;
  const REAL_D *coord = el_info->coord;

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);

  for (n = 0; n < DIM_OF_WORLD; n++)
    normal[n] = i0? coord[0][n] - coord[1][n] : coord[1][n] - coord[0][n];

  det = NORM_DOW(normal);
  for (n = 0; n < DIM_OF_WORLD; n++)
    normal[n] /= det;
#endif
  return(1.0);
}

/*--------------------------------------------------------------------------*/
/* orient the vertices of walls (i.e. codim 1 subsimplexes)                 */
/* used by estimator for the jumps => same quadrature nodes from both sides!*/
/*--------------------------------------------------------------------------*/

int *sorted_wall_indices_1d(int wall, int permno)
{
  static int sorted[2][1] = { {1}, {0} };

  return sorted[wall];
}

int wall_orientation_1d(const EL *el, int wall, int **vecp)
{
  if (vecp)
    (*vecp) = sorted_wall_indices_1d(wall, 0);

  return 0;
}

int *sort_wall_indices_1d(const EL *el, int wall, int *vec)
{
  int *ret;

  (void)wall_orientation_1d(el, wall, &ret);

  if (vec)
  {
    vec[0] = ret[0];
    return vec;
  }
  else
    return ret;
}
