/* ConjecturingWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace.compat;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.expr.Expression;
import org.grinvin.expr.ExpressionLoader;
import org.grinvin.worksheet.GraphListComponentType;
import org.grinvin.worksheet.WorksheetWindow;
import org.grinvin.worksheet.WorksheetWindowModel;
import org.grinvin.worksheet.WorksheetWindowModel.SubwindowType;
import org.grinvin.io.list.GraphListLoader;
import org.grinvin.io.IOFormatException;
import org.grinvin.io.list.InvariantListLoader;
import org.grinvin.io.SectionLoader;
import org.grinvin.workspace.DefaultWindowWrapper;

import org.jdom.Element;

/**
 *
 */
public class ConjecturingWindowWrapper extends DefaultWindowWrapper<WorksheetWindow> {
    
    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle("org.grinvin.worksheet.worksheetwindow");
    
    //
    public ConjecturingWindowWrapper() {
    }
    
    /**
     * Load the window, its graph list and its invariaht list.
     */
    public WorksheetWindow load(Element element, SectionLoader sloader) throws IOException {
        
        try {
            
            WorksheetWindowModel worksheetWindowModel = new WorksheetWindowModel();
            
            // Load the corresponding graph list
            URI graphlistURI = new URI(element.getAttributeValue("graphlistURI"));
            InputStream input = sloader.openSection(graphlistURI.getSchemeSpecificPart());
            GraphListLoader.load_1_0(worksheetWindowModel.getGraphList(), input, sloader);
            worksheetWindowModel.getGraphList().setURI(graphlistURI);
            worksheetWindowModel.setName(BUNDLE.getString("conjecturingwindow.title"));            
            
            InputStream invariantlist = sloader.openSection(graphlistURI.getSchemeSpecificPart());
            InvariantListLoader.load_1_0(worksheetWindowModel.getInvariantList(), invariantlist);

            // load list of conjectures
            Element child = element.getChild("conjectures");
            if (child != null) { // null when old workspace
                List<Expression> conjectures = worksheetWindowModel.getConjectureList();
                for (Object obj : child.getChildren()) {
                    try {
                        conjectures.add(ExpressionLoader.loadExpression((Element)obj));
                    } catch (IOFormatException ex) {
                        Logger.getLogger("org.grinvin.workspace").log(
                                Level.WARNING, "Ignoring incorrect expression", ex);
                    }
                }
            }
            
            worksheetWindowModel.getSubwindowModel(SubwindowType.FILTERLIST).setVisible(false);
            worksheetWindowModel.getSubwindowModel(SubwindowType.GENERATORINSTANCELIST).setVisible(false);

            String component = element.getAttributeValue("component");
            if (component.equals("graphInvariantTable")) {
                worksheetWindowModel.setGraphListComponentSelected(GraphListComponentType.TABLE);
            } else if (component.equals("graphCellList")) {
                worksheetWindowModel.setGraphListComponentSelected(GraphListComponentType.CELL);
            } else if (component.equals("graphList")) {
                worksheetWindowModel.setGraphListComponentSelected(GraphListComponentType.LIST);
            } else {
                throw new IOFormatException("Invalid component type: " + component);
            }
            
            // create and initialize the window
            WorksheetWindow window = new WorksheetWindow(worksheetWindowModel);
            
            // initialize properties of this window
            loadWindowProperties(element, window);
            
            return window;
        } catch (URISyntaxException ex) {
            throw new IOFormatException("Invalid URI in meta.xml", ex);
        }
        
    }

    public WorksheetWindow create() {
        throw new UnsupportedOperationException("Unsupported method called");
    }

    public String getElementName() {
        throw new UnsupportedOperationException("Unsupported method called");
    }
    
    
}
