/* LoggingWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.io.IOException;

import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;

import org.grinvin.io.IOFormatException;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.io.SectionLoader;
import org.grinvin.io.SectionSaver;
import org.grinvin.logging.Logging;
import org.grinvin.logging.LoggingWindow;

import org.jdom.DocType;
import org.jdom.Document;
import org.jdom.Element;

/**
 * Window wrapper for the factory window.
 */
public class LoggingWindowWrapper extends DefaultWindowWrapper<LoggingWindow> {
    
    //
    private static final String PUBLIC_ID = "-//GrInvIn IO//Logging 1.0//EN";
    
    @Override
    public LoggingWindow create() {
        return Logging.getLoggingWindow();
    }
    
    public String getElementName() {
        return "loggingWindow";
    }
    
    public LoggingWindow load(Element element, SectionLoader sloader) throws IOException {
        try {
            URI uri = new URI(element.getAttributeValue("uri"));
            String name = uri.getSchemeSpecificPart();
            LoggingWindow window = load(sloader, name);
            window.setURI(uri);
            return window;
        } catch (URISyntaxException ex) {
            throw new IOFormatException("invalid uri");
        }
    }

    
    private LoggingWindow load(SectionLoader sloader, String name) throws IOException {
        InputStream rootinput = sloader.openSection(name);
        
        Document document = LoaderSaverHelper.loadDocument(rootinput);
        if (!PUBLIC_ID.equals(document.getDocType().getPublicID()))
            throw new IOFormatException("Unkown document type: " + document.getDocType().getPublicID() + " expects " + PUBLIC_ID);
        Element element = document.getRootElement();

        LoggingWindow window = create();
        loadWindowProperties(element, window);
        return window;
    }
    
    public void save(LoggingWindow window, Element parent, SectionSaver ssaver) throws IOException {
        String name = LoaderSaverHelper.prepareSessionURI(window, "loggingwindows", "glog");
        save(window, ssaver, name);
        
        Element loggingwindow = new Element(getElementName());
        loggingwindow.setAttribute("uri", window.getURI().toString());
        parent.addContent(loggingwindow);
    }
    
    private void save(LoggingWindow window, SectionSaver ssaver, String name) throws IOException {
        Element element = baseElement();
        saveWindowProperties(element, window);
        // save tab
        LoaderSaverHelper.outputXML(element,
                new DocType("loggingWindow", PUBLIC_ID, "http://downloads.grinvin.org/dtds/io/logging-1.0.dtd"),
                ssaver.createSection(name));
    }
    
}
