/* EditorWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;

import org.grinvin.gred.GraphEditor;
import org.grinvin.io.graphs.AnnotationLoader;
import org.grinvin.io.graphs.AnnotationSaver;
import org.grinvin.io.graphs.EmbeddingLoader;
import org.grinvin.io.graphs.EmbeddingSaver;
import org.grinvin.io.list.GraphListLoader;
import org.grinvin.io.list.GraphListSaver;
import org.grinvin.io.graphs.GraphLoader;
import org.grinvin.io.graphs.GraphSaver;
import org.grinvin.io.IOFormatException;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.io.SectionLoader;
import org.grinvin.io.SectionSaver;
import org.grinvin.list.graphs.DefaultGraphListModel;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.graphs.GraphListModel;

import org.jdom.DataConversionException;
import org.jdom.DocType;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;

/**
 * Wrapper for the EditorWindow
 */
public class EditorWindowWrapper extends DefaultWindowWrapper<GraphEditor> {
    
    //
    private static final String PUBLIC_ID = "-//GrInvIn IO//Editor 1.1//EN";
    
    @Override
    public GraphEditor create() {
        return new GraphEditor(new DefaultGraphListModel());
    }
    
    public String getElementName() {
        return "editorWindow";
    }
    
    /**
     * Load the editor window, its graph list, the graph being edited
     * and the guides tool configuration.
     */
    @Override
    public GraphEditor load(Element element, SectionLoader sloader) throws IOException {
        try {
            String editorURI = element.getAttributeValue("uri");
            URI uri = new URI(editorURI);
            InputStream rootinput = sloader.openSection(uri.getSchemeSpecificPart() + "/meta.xml");
            Document document = LoaderSaverHelper.loadDocument(rootinput);
            if (!PUBLIC_ID.equals(document.getDocType().getPublicID()))
                throw new IOFormatException("Unkown document type: " + document.getDocType().getPublicID() + " expects " + PUBLIC_ID);
            GraphEditor editor = loadAll(document.getRootElement(), sloader);
            editor.setURI(uri);
            return editor;
        } catch (URISyntaxException ex) {
            throw new IOFormatException("Invalid URI in meta.xml", ex);
        } catch (DataConversionException ex) {
            throw new IOFormatException("Incorrect boolean attribute value in meta.xml", ex);
        } catch (JDOMException ex) {
            throw new IOFormatException("XML parse error", ex);
            
        }
    }
    
    private GraphEditor loadAll(Element element, SectionLoader sloader) throws IOException, URISyntaxException, DataConversionException {
        
        GraphListModel list = new DefaultGraphListModel();
        URI graphListURI = new URI(element.getChild("graphList").getAttributeValue("uri"));
        InputStream input = sloader.openSection(graphListURI.getSchemeSpecificPart());
        GraphListLoader.load(list, input, sloader);
        list.setURI(graphListURI);
        
        // create and initialize the window
        GraphEditor editor = new GraphEditor(list);
        // initialize properties of this window and of its guides tool
        loadWindowProperties(element, editor);
        editor.setDirty(element.getAttribute("dirty").getBooleanValue());
        
        if (element.getAttributeValue("showAdvancedToolbar") != null)
            editor.setAdvancedToolbar(element.getAttribute("showAdvancedToolbar").getBooleanValue());
        
        // load the graph
        InputStream graph = sloader.openSection(element.getAttributeValue("graph"));
        GraphLoader.load(editor.getGraph(), graph);
        InputStream embedding = sloader.openSection(element.getAttributeValue("embedding"));
        EmbeddingLoader.load(editor.getEmbedding(), embedding);
        if (element.getAttributeValue("annotation") != null) {
            InputStream annotation = sloader.openSection(element.getAttributeValue("annotation"));
            AnnotationLoader.load(editor.getAnnotation(), annotation);
        }
        
        return editor;
        
    }
    
    /**
     * Save the editor window, its graph list, the graph being edited
     * and the guides tool configuration.
     */
    @Override
    public void save(GraphEditor window, Element parent, SectionSaver ssaver) throws IOException {
        
        // save properties of window and of guides tool
        Element element = baseElement();
        
        String name = LoaderSaverHelper.prepareSessionURI(window, "editors", "editor");
        
        Element editor = new Element("editorWindow");
        editor.setAttribute("uri", window.getURI().toString());
        parent.addContent(editor);
        
        saveWindowProperties(element, window);
        element.setAttribute("dirty", Boolean.toString(window.isDirty()));
        element.setAttribute("showAdvancedToolbar", Boolean.toString(window.showAdvancedToolbar()));
        
        // save graph being edited
        EmbeddingSaver.save(window.getEmbedding(), ssaver.createSection(name + "/embedding.xml"));
        AnnotationSaver.save(window.getAnnotation(), ssaver.createSection(name + "/annotation.xml"));
        GraphSaver.save(window.getGraph(), ssaver.createSection(name + "/graph.xml"));
        
        // save graph list (including contents)
        GraphList list = window.getGraphList();
        GraphListSaver.saveIntoWorkspace(list, ssaver);
        
        Element graphList = new Element("graphList");
        graphList.setAttribute("uri", list.getURI().toString());
        element.addContent(graphList);
        
        element.setAttribute("embedding", name + "/embedding.xml");
        element.setAttribute("annotation", name + "/annotation.xml");
        element.setAttribute("graph", name + "/graph.xml");
        
        LoaderSaverHelper.outputXML(element,
                new DocType("editorWindow", PUBLIC_ID, "http://downloads.grinvin.org/dtds/io/editor-1.1.dtd"),
                ssaver.createSection(name + "/meta.xml"));
        
    }

}
