/* MoveSelection.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.mouse;

import be.ugent.caagt.swirl.mouse.NullMouseHandler;

import java.awt.event.MouseEvent;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;

import org.grinvin.graphs.Element;
import org.grinvin.gred.GraphContext;
import org.grinvin.gred.GraphPanel;
import org.grinvin.gred.MutableGraphPanel;
import org.grinvin.util.GrinvinUtilities;

/**
 * Mouse handler for adding edges. Consumes
 * all first button click events which were initiated on an element.
 */
public class MoveSelection extends NullMouseHandler{
    
    // create a singleton
    private MoveSelection () { }
    
    //
    private static final MoveSelection SINGLETON = new MoveSelection();
    
    //
    private static final String RESOURCE_KEY = "MoveSelection";
    
    /**
     * Return the singleton object of this type.
     */
    public static MoveSelection getHandler () {
        return SINGLETON;
    }
    
    /**
     * Select elements clicked.
     */
    @Override public void doMousePressed (JComponent parent, MouseEvent mouseEvent) {
        if (SwingUtilities.isLeftMouseButton (mouseEvent)) {
            GraphContext context = ((GraphPanel)parent).getContext ();
            Element element = context.getRollOver ();
            if (element != null) {
                
                // change selection (note difference with SelectElements)
                if (GrinvinUtilities.isMultipleSelectionEvent(mouseEvent) || context.isSelected (element)) {
                    context.addSelection (element);
                } else {
                    context.setSelection (element);
                }
                
                // extend selection
                context.extendEdgeSelection ();
                mouseEvent.consume ();
                
                // save a copy of the current embedding
                ((MutableGraphPanel)parent).initiateEmbeddingChange ();
            }
        }
    }
    
    /**
     * Move the selected elements.
     */
    @Override public void doMouseDragged (JComponent parent, MouseEvent mouseEvent, MouseEvent previousEvent, MouseEvent pressedEvent) {
        ((MutableGraphPanel)parent).moveSelection
                (mouseEvent.getX () - pressedEvent.getX (),
                mouseEvent.getY () - pressedEvent.getY ());
    }
    
    /**
     * Consolidate the drag.
     */
    @Override public void doMouseReleased (JComponent parent, MouseEvent mouseEvent, MouseEvent pressedEvent) {
        MutableGraphPanel panel = (MutableGraphPanel)parent;
        if (! panel.getContext ().isSelectionEmpty ())
            panel.finalizeEmbeddingChange (RESOURCE_KEY);
    }
    
    /**
     * {@inheritDoc} This mouse handler needs a {@link MutableGraphPanel}
     * as parent.
     */
    @Override public boolean canHandle (JComponent parent) {
        return parent instanceof MutableGraphPanel;
    }
}
