
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.utilities;

import java.io.*;
import java.util.*;

/**
 * Provides a mechanism for a program to keep a set of options that can
 * be stored in a file between program sessions.
 * Each option has a string name associated with it, and can be read
 * or altered using the various <tt>get</tt> and <tt>set</tt> member
 * functions.
 */
public class OptionSet {
    /**
     * Comment to be placed at the beginning of the option file when saving.
     */
    private String comment;

    /**
     * File to/from which this option set is to be written/read.
     */
    private File optionFile;

    /**
     * Structure containing the actual options and their values, all
     * in string form.
     */
    private Properties options = new Properties();

    /**
     * Creates a new option set not associated with any file.
     * For such an option set, when routines <tt>writeToFile()</tt> and
     * <tt>readFromFile()</tt> are called, nothing will be done.
     * <p>
     * The initialisation routine <tt>init()</tt> will be called
     * from this constructor.
     *
     * @see #init
     */
    public OptionSet() {
        this.optionFile = null;
        this.comment = null;
        init();
    }

    /**
     * Creates a new option set based on the given file.
     * All options stored in the file will be loaded.  Any errors
     * that occur when reading from file will be ignored.
     * <p>
     * The initialisation routine <tt>init()</tt> will be called before
     * the attempt to read the file.
     *
     * @param optionFile the file to/from which this option set is to
     * be written/read.
     * @param comment a comment to place at the beginning of the option
     * file when writing; this parameter is essentially irrelevant unless
     * you wish to hand-examine the option file.  It does not matter if
     * the comment passed here is different to the existing comment in the
     * file; the existing comment will simply be overwritten.
     * @see #OptionSet(File, String, boolean)
     * @see #init
     */
    public OptionSet(File optionFile, String comment) {
        this.optionFile = optionFile;
        this.comment = comment;
        init();
        try {
            readFromFile();
        } catch (IOException e) {}
    }

    /**
     * Creates a new option set based on the given file.
     * All options stored in the file will be loaded.
     * <p>
     * The initialisation routine <tt>init()</tt> will be called before
     * the attempt to read the file.
     *
     * @param optionFile the file to/from which this option set is to
     * be written/read.
     * @param comment a comment to place at the beginning of the option
     * file when writing; this parameter is essentially irrelevant unless
     * you wish to hand-examine the option file.  It does not matter if
     * the comment passed here is different to the existing comment in the
     * file; the existing comment will simply be overwritten.
     * @param forceLoad if set to <tt>false</tt>, we will ignore
     * any errors in reading from file.  If set to <tt>true</tt>, an
     * exception will be thrown if an error occurs.
     * @throws IOException thrown if an error occurs in reading from file.
     * @see #OptionSet(File, String)
     * @see #init
     */
    public OptionSet(File optionFile, String comment, boolean forceLoad)
            throws IOException {
        this.optionFile = optionFile;
        this.comment = comment;
        init();
        try {
            readFromFile();
        } catch (IOException e) {
            if (forceLoad)
                throw e;
        }
    }

    /**
     * Performs extra initialisation for a new option set that should take
     * place before the options are read from file.
     * This implementation does nothing, but subclasses may wish to
     * override this routine.  It is called from every
     * <tt>OptionSet</tt> constructor.
     */
    protected void init() {
    }
        
    /**
     * Attempt to read this option set from file.
     *
     * @throws IOException thrown if an error occurs in reading from file.
     */
    protected void readFromFile() throws IOException {
        if (optionFile == null)
            return;

        FileInputStream in = null;
        try {
            in = new FileInputStream(optionFile);
            options.load(in);
        }
        catch (IOException e) {
            try {
                if (in != null)
                    in.close();
            } catch (IOException eClose) {}
            throw e;
        }
        try {
            in.close();
        } catch (IOException eClose) {}
    }

    /**
     * Removes the given option from this option set.
     *
     * @param key the name of the requested option.
     */
    public void removeOption(String key) {
        options.remove(key);
    }

    /**
     * Retrieves the value of a string option.
     *
     * @param key the name of the requested option.
     * @return the value of the requested option, or <tt>null</tt> if no
     * such option is in the set.
     */
    public String getStringOption(String key) {
        return options.getProperty(key);
    }

    /**
     * Retrieves the value of a string option.
     *
     * @param key the name of the requested option.
     * @param defaultOption the value to use if the requested option is
     * not in the set.
     * @return the value of the requested option, or <tt>defaultOption</tt>
     * if no such option is in the set.
     */
    public String getStringOption(String key, String defaultOption) {
        return options.getProperty(key, defaultOption);
    }

    /**
     * Sets the value of a given string option.
     * The options will not be written to file until
     * <tt>writeToFile()</tt> is called.
     *
     * @param key the name of the option to set.
     * @param value the value to assign to the option.
     * @see #writeToFile()
     */
    public void setStringOption(String key, String value) {
        options.put(key, value);
    }

    /**
     * Retrieves the value of a boolean option.
     *
     * @param key the name of the requested option.
     * @return the value of the requested option, or <tt>false</tt> if no
     * such option is in the set.
     */
    public boolean getBooleanOption(String key) {
        return getBooleanOption(key, false);
    }

    /**
     * Retrieves the value of a boolean option.
     *
     * @param key the name of the requested option.
     * @param defaultOption the value to use if the requested option is
     * not in the set.
     * @return the value of the requested option, or <tt>defaultOption</tt>
     * if no such option is in the set.
     */
    public boolean getBooleanOption(String key, boolean defaultOption) {
        String string = options.getProperty(key,
            String.valueOf(defaultOption));
        if (string == null)
            return defaultOption;
        return Boolean.valueOf(string).booleanValue();
    }

    /**
     * Sets the value of a given boolean option.
     * The options will not be written to file until
     * <tt>writeToFile()</tt> is called.
     *
     * @param key the name of the option to set.
     * @param value the value to assign to the option.
     * @see #writeToFile()
     */
    public void setBooleanOption(String key, boolean value) {
        options.put(key, String.valueOf(value));
    }

    /**
     * Retrieves the value of an integer option.
     *
     * @param key the name of the requested option.
     * @return the value of the requested option, or 0 if no
     * such option is in the set.
     */
    public int getIntOption(String key) {
        return getIntOption(key, 0);
    }

    /**
     * Retrieves the value of an integer option.
     * If the option is not an integer, 0 will be returned.
     *
     * @param key the name of the requested option.
     * @param defaultOption the value to use if the requested option is
     * not in the set.
     * @return the value of the requested option, or <tt>defaultOption</tt>
     * if no such option is in the set.
     */
    public int getIntOption(String key, int defaultOption) {
        String string = options.getProperty(key,
            String.valueOf(defaultOption));
        if (string == null)
            return defaultOption;
        try {
            return Integer.valueOf(string).intValue();
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    /**
     * Sets the value of a given integer option.
     * The options will not be written to file until
     * <tt>writeToFile()</tt> is called.
     *
     * @param key the name of the option to set.
     * @param value the value to assign to the option.
     * @see #writeToFile()
     */
    public void setIntOption(String key, int value) {
        options.put(key, String.valueOf(value));
    }

    /**
     * Attempt to write this option set to file.
     * Don't worry if the operation failed.
     */
    public void writeToFile() {
        try {
            writeToFile(false);
        } catch (IOException e) {}
    }

    /**
     * Attempt to write this option set to file.
     *
     * @param forceWrite if set to <tt>false</tt>, we will ignore
     * any errors in writing to file.  If set to <tt>true</tt>, an
     * exception will be thrown if an error occurs.
     * @throws IOException thrown if an error occurs in writing to file.
     */
    public void writeToFile(boolean forceWrite) throws IOException {
        if (optionFile == null)
            return;

        // Store the options to file.
        FileOutputStream out = null;
        try {
            out = new FileOutputStream(optionFile);
            // Now we have to use store() for JDK 1.2 or save() for JDK 1.1.
            try {
                options.store(out, comment);
            }
            catch (NoSuchMethodError err) {
                // Must be using JDK1.1; use the deprecated method.
                options.save(out, comment);
            }
        }
        catch (IOException e) {
            try {
                if (out != null)
                    out.close();
            } catch (IOException eClose) {}
            if (forceWrite)
                throw e;
            else
                return;
        }
        try {
            out.close();
        } catch (IOException eClose) {}
    }

    /**
     * Gets the system property indicated by the specified key.
     * This is a convenience wrapper for <tt>System.getProperty()</tt>
     * that does exception handling.  If any exception or error
     * (such as a security exception) is thrown, this routine will catch
     * it and simply return <tt>null</tt>.
     *
     * @param key the name of the system property to retrieve.
     * @return the string value of the requested system property,
     * or <tt>null</tt> if there is no property with that key or if an
     * error occurs.
     */
    public static String getSystemProperty(String key) {
        try {
            return System.getProperty(key);
        } catch (Throwable th) {
            return null;
        }
    }

    /**
     * Gets the system property indicated by the specified key.
     * This is a convenience wrapper for <tt>System.getProperty()</tt>
     * that does exception handling.  If any exception or error
     * (such as a security exception) is thrown, this routine will catch
     * it and simply return the given default value.
     *
     * @param key the name of the system property to retrieve.
     * @param def the default value to use if there is no property with
     * the given key.
     * @return the string value of the requested system property,
     * or <tt>def</tt> if there is no property with that key or if an
     * error occurs.
     */
    public static String getSystemProperty(String key, String def) {
        try {
            return System.getProperty(key, def);
        } catch (Throwable th) {
            return def;
        }
    }
}

