/*
 * main.c  -  Main routine of romcheck program
 *
 * Copyright (C) 1998 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <version.h>
#include <stdlib.h>
#include <stdio.h>
#include "romcheck.h"



/*
 * Print help text
 */
static void printhelp()
{
  printf("\n\nValid commands:\n");
  printf("    h  -  print this help\n");
  printf("    n  -  find and identify netboot bootrom\n");
  printf("    p  -  print memory map graphically\n");
  printf("    q  -  terminate romcheck and reboot\n");
  printf("    x  -  print rom area as a hex dump\n\n");
  printf("Explanation of symbols in memory map display:\n");
  printf("    %c  -  Area with valid rom signature  ", CHAR_VALID);
  printf("    %c  -  Physical rom area\n", CHAR_ROM);
  printf("    %c  -  Area with invalid rom checksum ", CHAR_INVALID);
  printf("    %c  -  Physical ram area\n", CHAR_RAM);
  printf("                                         ");
  printf("    %c  -  Unknown (maybe empty)\n", CHAR_EMPTY);
  printf("    %c  -  Marks beginning of flash ROM   ", CHAR_FLASH);
  printf("    %c  -  Unknown\n", CHAR_UNKNOWN);
}



/*
 * Print information about netboot bootrom
 */
static void printnb(sip)
struct scaninfo *sip;
{
  unsigned short index = sip->nbindex;

  if (index >= 0 && index < sip->signum) {
	printf("\n\nFound netboot bootrom:\n");
	printf("    Rom area number:  %d\n", index + 1);
	printf("    Starting segment: %4X", sip->sigsegs[index]);
	if (sip->flashseg == sip->sigsegs[index])
		printf("   (at beginning of flash EPROM)");
	printf("\n    Rom size:         %d kB\n", sip->sigsize[index] / 64);
	printf("    Version:          %d.%d\n", sip->nbmajor, sip->nbminor);
	if (!sip->sigvalid[index])
		printf("    The checksum of the netboot bootrom is invalid!\n");
  } else {
	printf("\nNo netboot bootrom found\n");
  }
}



/*
 * Main routine of romcheck. When it returns, the runtime library will
 * reboot the system.
 */
void main()
{
  struct scaninfo si;

  printf("\036\fromcheck, ");
  printf("%s\n", VERSION);
  printf("%s\n", COPYRIGHT);
  printf("\n\nThis program is free software; you can redistribute it and/or modify\n");
  printf("it under the terms of the GNU General Public License as published by\n");
  printf("the Free Software Foundation; either version 2 of the License, or any later\n");
  printf("version. If you have any comments or suggestions, please send an email to\n");
  printf("gero@gkminix.han.de.\n\n\n");
  printf("Scanning memory... ");
  doscan(&si);
  printf("done\n");
  do {
	printf("\n\nEnter command ('h' for help): ");
	switch (waitchar()) {
		case 'h':
			/* Print help text */
			printhelp();
			break;
		case 'n':
			/* Print netboot bootrom information */
			printnb(&si);
			break;
		case 'p':
			/* Print scanning results */
			printscan(&si);
			break;
		case 'q':
			/* Terminate program */
			printf("\n\n\nRemove diskette and press <Enter> to reboot...");
			while (waitchar() != '\r') ;
			return;
		case 'x':
			/* Print hex dump */
			printhex(&si);
			break;
		default:
			break;
	}
  } while (TRUE);
}

