/*
    ras - Redundant Archive System
    Copyright (C) 1999  Nick Cleaton

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    Nick Cleaton <nick@cleaton.net>
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>
#include <stddef.h>

#include "common.h"
#include "utils.h"


/*************************************************************************
**
**  Exiting (and ensuring that all output files are removed) if an error
**  occurs.
*/

struct outfile_elt
{
   struct outfile_elt *next;
   char *filename;
};

struct outfile_elt *outfile_linkedlist_head;

/*************************************************************************/

void exit_because_of_error(int errtype)
{
   struct outfile_elt *oe;

   for( oe = outfile_linkedlist_head ; oe ; oe = oe->next )
   {  unlink(oe->filename);
   }

   switch (errtype)
   {
      case INVALID_INPUT:
         exit(2);
      case NOT_ENOUGH:
         exit(1);
      case EXTERNAL_FAILURE:
         exit(2); 
      case INTERNAL_FAILURE:
         exit(2);
      default:
         fprintf(stderr,PROGNAME ": internal failure: invalid arg to exit\n");
	 exit(2);
   }
}

/*************************************************************************/

FILE *open_output_file(char *name)
{
   struct outfile_elt *oe;
   FILE *f;

   oe = nofree(Malloc( sizeof(*oe) ));

   f = fopen(name, "wb");
   if( !f )
   {  fprintf(stderr, PROGNAME ": unable to open %s for output: %s\n",
                                                    name, strerror(errno));
      exit_because_of_error(EXTERNAL_FAILURE);
   }

   /* prepend new filename to linked list */
   oe->filename = name;
   oe->next = outfile_linkedlist_head;
   outfile_linkedlist_head = oe;

   return f;
}

/**************************************************************************/

void init_unlink_list()
{
   outfile_linkedlist_head = NULL;
}

/**************************************************************************/



/*************************************************************************
**
**  Testing for error conditions
*/

void myassert(int cond, char *msg)
{
   if( !cond )
   {  fprintf(stderr, PROGNAME ": assert failed: %s\n",msg);
      exit_because_of_error(INTERNAL_FAILURE);
   }
}

/*************************************************************************/



/**************************************************************************
**
**  Keeping track of how many segments there are in the archive, and giving
**  an error if 2 things that fix how many segments there must be disagree.
*/

static int segment_count = -1;  /*  -1 indicates not fixed yet  */

static char *segment_count_source;

int get_segment_count()
{
   return segment_count;
}

void register_segment_count(int count, char *source)
{
   if( (count < 2) || (count > MAX_SEGMENT_COUNT) )
   {  fprintf(stderr,PROGNAME ": %s specifies a segment count of %i, "
                              "out of bounds\n", source, count);
      exit_because_of_error(INVALID_INPUT);
   }

   if( segment_count == -1 )
   {  segment_count = count;
      segment_count_source = source;
   }
   else if( segment_count != count )
   {  fprintf(stderr,PROGNAME ": %s specifies a segment count of %i,\n"
                     PROGNAME ": but %s has already fixed it at %i\n",
		     source, count, segment_count_source, segment_count );
      exit_because_of_error(INVALID_INPUT);
   }
}

/*****************************************************************************/




/**************************************************************************
**
**  String manipulation
*/

unsigned char fit_int_in_uchar(int i)
{
   if( (i<0) || (i>255) )
   {  fprintf(stderr,PROGNAME ": can't fit int %i in a uchar\n", i);
      exit_because_of_error(INTERNAL_FAILURE);
   }
   return (unsigned char) i;
}

/*************************************************************************/

int countchar(char *str, char c)
{
   int count =0;

   while( *str )
      if(*str++ == c)
         count++;

   return count;
}

/**************************************************************************/

char *popfirstword(char **s, char sep)
{
   char *first_word;

   if( !**s )
      return NULL;

   first_word = *s;
   while( **s && (**s != sep) )
      (*s)++;

   if( **s )
   {  **s = '\0';
      (*s)++;
   }

   return first_word;
}

/*************************************************************************/




/**************************************************************************
**
**  Memory allocation
*/

void *Malloc(size_t bytes)
{  
   void *p;
   p = malloc(bytes);
   if( !p )
   {  fprintf(stderr, PROGNAME ": out of memory\n");
      exit_because_of_error(EXTERNAL_FAILURE);
   }
   return p;
}

/*************************************************************************/

char *Strdup(char *s)
{
   char *a;
   
   a = Malloc(strlen(s) + 1);
   strcpy(a, s);
   return a;
}

/*************************************************************************/

void *nofree(void *ptr)
{
   return ptr;
}

/*************************************************************************/
   
   

