/**************************************************************************** 
**
** File: error.c
**
** Author: Mike Borella
**
** Comments: Try to gracefully handle errors.  
**
*****************************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include "error.h"
#include "config.h"

int GWF_error_flag = 0;

/*
 * This avoids a warning with glibc compilation 
 */

#ifndef errno
extern int errno;
#endif

/*
 * If we have strerror() we won't use sys_errlist, so don't bother defining
 * it.
 */

#ifndef HAVE_STRERROR
extern char * sys_errlist[];
#endif

/*----------------------------------------------------------------------------
**
** Function: GWFi_strerror()
**
** Comments: Use strerror() if this system supports it. Roll our own otherwise.
**
**----------------------------------------------------------------------------
*/

char *GWFi_strerror(int errnum)
{
#ifdef HAVE_STRERROR
  return (strerror(errnum));
#else
  extern int sys_nerr;
  static char ebuf[20];
  
  if ((int) errnum < sys_nerr)
    return ((char *) sys_errlist[errnum]);
  sprintf(ebuf, "Unknown error: %d", errnum);
  return(ebuf);
#endif
}

/*----------------------------------------------------------------------------
**
** GWF_error()
**
** Generic error dumping.  Here's how the arguments are used
**
** syserr: 0 is not an error from a system call, non-zero otherwise
** filename: filename of the caller.  Optional.
** line: line number of the caller.  Optional, set to 0 is filename and 
**       line number info is not to be printed
**
**----------------------------------------------------------------------------
*/

void GWF_error(int syserr, char *filename, int line, const char *fmt, 
	       va_list args)
{
  /*
   * Dump the error message
   */

  fprintf(stderr,"ERROR: ");
  if (line)
    fprintf(stderr, "%s line %d: ", filename, line);
  vfprintf(stderr, fmt, args);

  /*
   * If this is a system error, print the errno info.  Otherwise, don't
   */

  if (syserr)
    {
      fprintf(stderr," (%d ", errno);
      fprintf(stderr, "%s)\n", GWFi_strerror(errno));
    }
  else
    fprintf(stderr, "\n");
}


/*****************************************************************************
 * Externally-visible functions are below here
 ****************************************************************************/


/*----------------------------------------------------------------------------
**
** GWF_error_fatal()
**
** General fatal error.
**
**----------------------------------------------------------------------------
*/

void GWF_error_fatal(char *fmt, ...)
{
  va_list       args;

  va_start(args,fmt);
  GWF_error(0, "", 0, fmt, args);
  va_end(args);
  abort();
}

/*----------------------------------------------------------------------------
**
** GWF_error_system()
**
** General system error.
**
**----------------------------------------------------------------------------
*/

void GWF_error_system(char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  GWF_error(1, "", 0, fmt, args);
  va_end(args);
  abort();
}

/*----------------------------------------------------------------------------
**
** GWF_error_message()
**
** General non-terminal error.
**
**----------------------------------------------------------------------------
*/

void GWF_error_message(char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  GWF_error(1, "", 0, fmt, args);
  va_end(args);
}
