/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include "common.h"
#include "error.h"
#include "expr.h"
#include "misc.h"
#include "str.h"
#include "lexer.h"
#include "var.h"

/* SELECT IF transformation. */
typedef struct
  {
    trns_header h;
    struct expression *e;	/* Test expression. */
  }
select_if_trns;

static int select_if_proc (any_trns *, ccase *);
static void select_if_free (any_trns *);

/* Parses the SELECT IF transformation. */
int
cmd_select_if (void)
{
  expression *e;
  select_if_trns *t;

  match_id (SELECT);
  match_id (IF);

  e = parse_expression (PXP_BOOLEAN);
  if (!e)
    return 0;

  if (token != '.')
    {
      free_expression (e);
      return syntax_error (_("expecting end of command"));
    }

  t = xmalloc (sizeof (select_if_trns));
  t->h.proc = select_if_proc;
  t->h.free = select_if_free;
  t->e = e;
  add_transformation ((any_trns *) t);

  return 1;
}

/* Performs the SELECT IF transformation T on case C. */
static int
select_if_proc (any_trns * t, ccase * c)
{
  return (evaluate_expression (((select_if_trns *) t)->e, c, NULL) == 1.0) - 2;
}

/* Frees SELECT IF transformation T. */
static void
select_if_free (any_trns * t)
{
  free_expression (((select_if_trns *) t)->e);
}

/* Parses the FILTER command. */
int
cmd_filter (void)
{
  match_id (FILTER);

  if (match_id (OFF))
    default_dict.filter_var[0] = 0;
  else
    {
      variable *v;

      match_tok (BY);
      v = parse_variable ();
      if (!v)
	return 0;
      if (v->type == ALPHA)
	return msg (SE, _("The filter variable must be numeric."));
      if (v->name[0] == '#')
	return msg (SE, _("The filter variable may not be scratch."));
      strcpy (default_dict.filter_var, v->name);

      FILTER_before_TEMPORARY = !temporary;
    }

  return 1;
}

/* Parses the PROCESS IF command. */
int
cmd_process_if (void)
{
  expression *e;

  match_id (PROCESS);
  match_id (IF);

  e = parse_expression (PXP_BOOLEAN);
  if (!e)
    return 0;

  if (token != '.')
    {
      free_expression (e);
      return syntax_error (_("expecting end of command"));
    }

  if (process_if_expr)
    {
      msg (MW, _("Only last instance of this command is in effect."));
      free_expression (process_if_expr);
    }
  process_if_expr = e;

  return 1;
}
