#pragma implementation
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dialog.h>
#include <ctype.h>
#include <misc.h>
#include <configf.h>
#include <translat.h>
#include <userconf.h>
#include <subsys.h>
#include <fviews.h>
#include "inetdconf.h"
#include "inetdconf.m"
#include "inetdconfedit.h"
#include "etcservice.h"
#include "etcprotocol.h"

static LINUXCONF_SUBSYS subb (subsys_inetdconf,P_MSG_U(T_INETD,
	"Internet super server"
	));

static HELP_FILE help_inetdconf ("inetdconf","inetdconf");
static CONFIG_FILE f_config_file(
		"/etc/inetd.conf"
		,help_inetdconf
		,CONFIGF_MANAGED
		,"root"
		,"root"
		,0644
		,subsys_inetdconf);

static INETDCONFLIST *inetdconflist = NULL;
static ETCSERVICELIST *etcservicelist = NULL;
static ETCPROTOCOLLIST *etcprotocollist = NULL;
static long modified_time = 0;
static bool config_file_modified = false;

/*
 * INETDCONF
 */
#define	K_FIELD_INETDCONF_NAME		1
#define	K_FIELD_SOCKET_TYPE		2
#define	K_FIELD_PROTOCOL		3
#define	K_FIELD_WAIT_MAX		4
#define	K_FIELD_USER_GROUP		5
#define	K_FIELD_PATH			6

PRIVATE void INETDCONF::init()
{
//fprintf(stderr,"INETDCONF::init\n");
	enabled = 1;
	new_inetdconf = 0;
	socket_type.setfrom( "stream" );
	protocol.setfrom( "tcp" );
	user.setfrom( "root" );
	max = 0;
	multi = 0;
	port = 0;
}

PUBLIC INETDCONF::INETDCONF(const char *_service_name)
{
//fprintf(stderr,"INETDCONF::INETDCONF _service_name=%s\n", _service_name);
	service_name.setfrom (_service_name);
	init();
}

PUBLIC INETDCONF::INETDCONF( )
{
//fprintf(stderr,"INETDCONF::INETDCONF\n");
	init();
}

PUBLIC int INETDCONF::write( int button )
{
//fprintf(stderr,"INETDCONF::write\n");
	int ret = -1;
	if ( ! perm_rootaccess( MSG_U(P_EDITINETDCONF, "change inetdconf configuration") ) ) {
		return( ret );
	}
	long this_modified_time = file_date( f_config_file.getpath() );
	if ( this_modified_time > modified_time ) {
		xconf_error(MSG_U(E_FILE_MODIFIED,
			"Config file %s has been modified recently\n"
			"by someone else. The list of records in the\n"
			"module may be incorrect.\n"
			"\n"
			"This update can not be done.\n"
			), f_config_file.getpath() );
		config_file_modified = true;
		return( ret );
	}
	VIEWITEMS items;
	items.setcomcar( '\002' );	// Set comment to "impossible" char
	items.read( f_config_file );	// Read current version of config file
	if ( new_inetdconf ) {
		VIEWITEM *item = new VIEWITEM("");
		inetdconf_line = items.getnb();
		modify_service( item );
		items.add( item );
	} else {
		for ( int line=0; line<items.getnb(); line++ ) {
			if ( line == inetdconf_line ) {
				VIEWITEM *item = items.getitem(line);
				if ( button == MENU_DEL ) {
					items.remove( item );
				} else {
					modify_service( item );
				}
			}
		}
	}
	items.write( f_config_file, (PRIVILEGE *)NULL );

	/*
	 * Last modified set when config file read or written.
	 */
	modified_time = file_date( f_config_file.getpath() );
	return( ret );
}


/**
 * Update line
 */
PRIVATE void INETDCONF::modify_service( VIEWITEM *item )
{
//fprintf(stderr,"INETDCONF::modify_service: item=%s\n", item->line.get());
	char line[2048];
	char multi[100];
	char user_group[100];
	if ( max ) {
		snprintf( multi, sizeof(multi),"%s.%d",
			multi?"wait":"nowait",
			max );
	} else {
		snprintf( multi, sizeof(multi),"%s",
			multi?"wait":"nowait" );
	}

	if ( group.is_empty() ) {
		snprintf( user_group, sizeof(user_group),"%s",
			user.get() );
	} else {
		snprintf( user_group, sizeof(user_group),"%s.%s",
			user.get(),
			group.get() );
	}

	snprintf( line, sizeof(line), "%s%s\t%s\t%s\t%s\t%s\t%s\t%s",
		enabled?"":"#",
		service_name.get(),
		socket_type.get(),
		protocol.get(),
		multi,
		user_group,
		path.get(),
		arguments.get() );
	item->line.setfrom( line );
}

/**
 * Edit inetdconf entry
 */
PUBLIC int INETDCONF::edit( INETDCONFLIST *list )
{
	inetdconflist = list;
	return( edit( ) );
}

PUBLIC int INETDCONF::edit()
{
//fprintf(stderr,"INETDCONF::edit\n");
	DIALOG dia;

	if ( new_inetdconf ) {
		FIELD_COMBO *combo = dia.newf_combo(MSG_U(F_SERVICES
			,"Service name"),service_name);
		ETCSERVICE *etcservice = NULL;
		SSTRINGS strings;
		for (int i=0; i<etcservicelist->getnb(); i++ ) {
			etcservice = etcservicelist->getitem( i );
			SSTRING *service = new SSTRING(etcservice->service_name.get());
			strings.add( service );
		}
		strings.sort();
		strings.remove_dups();
		for (int i=0; i<strings.getnb(); i++ ) {
			combo->addopt(strings.getitem(i)->get());
		}
	} else {
		dia.newf_info( MSG_R(F_SERVICES), service_name.get() );
		/*
		 * The same service can occur on another port with
		 * a different protocol.
		 */
		ETCSERVICE *etcservice = NULL;
		SSTRINGS strings;
		for (int i=0; i<etcservicelist->getnb(); i++ ) {
			bool add_service = false;
			etcservice = etcservicelist->getitem( i );
			if ( etcservice->service_name.cmp( service_name ) == 0 ) {
				add_service = true;
			} else {
				/*
				 * Service alias may be used..
				 */
				char word[100];
				const char *p = etcservice->aliases.get( );
				while ( *p ) {
					p = str_copyword( word, p, sizeof( word ) );
					if (strcmp(word, service_name.get()) == 0 ) {
						add_service = true;
					}
				}
			}
			if ( add_service ) {
				char portnumber[10];
				sprintf(portnumber, "%d", etcservice->port);
				SSTRING *pn = new SSTRING(portnumber);
				strings.add( pn );
			}
		}
		strings.sort();
		strings.remove_dups();
		SSTRING valid_ports;
		for (int i=0; i<strings.getnb(); i++ ) {
			if ( valid_ports.is_empty() ) {
				valid_ports.setfrom( strings.getitem(i)->get() );
			} else {
				valid_ports.append( "," );
				valid_ports.append( strings.getitem(i)->get() );
			}
		}
		if ( ! valid_ports.is_empty() ) {
			dia.newf_info( MSG_U(F_ETCSERVICEPORT,"Port"), valid_ports.get() );
		}
	}
	{
		static const char *enabled_use[]={
			MSG_U(F_DISABLED,"Disabled"),
			MSG_U(F_ENABLED,"Enabled"),
			NULL
		};
		dia.newf_chkm (MSG_U(F_STATE,"State"),enabled,enabled_use);
	}

	{
	FIELD_LIST *combo = dia.newf_list(MSG_U(F_PROTOCOLS
		,"Protocol"),protocol);
	if ( new_inetdconf ) {
		/*
		 * Show all protocols in /etc/protocols
		 */
		ETCPROTOCOL *etcprotocol = NULL;
		SSTRINGS strings;
		for (int i=0; i<etcprotocollist->getnb(); i++ ) {
			etcprotocol = etcprotocollist->getitem( i );
			SSTRING *protocol_name = new SSTRING(etcprotocol->protocol_name.get());
			strings.add( protocol_name );
		}
		strings.sort();
		for (int i=0; i<strings.getnb(); i++ ) {
			combo->addopt(strings.getitem(i)->get());
		}
	} else {
		/*
		 * Find valid protocols for service in /etc/services
		 */
		SSTRINGS strings;
		ETCSERVICE *etcservice = NULL;
		for (int i=0; i<etcservicelist->getnb(); i++ ) {
			bool add_service = false;
			etcservice = etcservicelist->getitem( i );
			if ( etcservice->service_name.cmp( service_name ) == 0 ) {
				add_service = true;
			} else {
				/*
				 * Service alias may be used..
				 */
				char word[100];
				const char *p = etcservice->aliases.get( );
				while ( *p ) {
					p = str_copyword( word, p, sizeof( word ) );
					if (strcmp(word, service_name.get()) == 0 ) {
						add_service = true;
					}
				}
			}
			if ( add_service ) {
				SSTRING *service = new SSTRING(etcservice->protocol.get());
				strings.add( service );
			}
		}
		strings.sort();
		strings.remove_dups();
		for (int i=0; i<strings.getnb(); i++ ) {
			combo->addopt(strings.getitem(i)->get());
		}
	}
	}

	{
		ETCPROTOCOL *etcprotocol = NULL;
		etcprotocol = etcprotocollist->getitem( protocol.get() );
		if ( etcprotocol == NULL ) {
			etcprotocol = etcprotocollist->getitem_alias( protocol.get() );
		}
		if ( etcprotocol != NULL ) {
			dia.newf_info( MSG_U(F_ETCPROTOCOLCOMMENT,"Protocol description"), etcprotocol->comment.get() );
		}
	}

	{
		FIELD_LIST *combo = dia.newf_list(MSG_U(F_SOCKET_TYPE
			,"Socket type"),socket_type);
		combo->addopt ("dgram",MSG_U(F_DATAGRAM,"Datagram"));
		combo->addopt ("raw",MSG_U(F_RAW,"Raw"));
		combo->addopt ("rdm",MSG_U(F_RDM,"Reliable delivered message"));
		combo->addopt ("seqpacket",MSG_U(F_SEQPACKET,"Sequenced packet socket"));
		combo->addopt ("stream",MSG_U(F_STREAM,"Stream"));
	}

	{
		static const char *multi_use[]={
			MSG_U(F_NO_WAIT,"Yes (nowait)"),
			MSG_U(F_WAIT,"No (wait)"),
			NULL
		};
		dia.newf_chkm (MSG_U(F_MULTIPROCESS,"Concurrent processes"),multi,multi_use);
	}

	{
		static const char *tbmsg[]={
			MSG_U(F_MAXDEFAULT,"Default"),NULL};
		static int tbv[]={0,0};
		dia.newf_chkm_num( MSG_U(F_MAX,
			"Max processes per minute"),max,tbv,tbmsg );
	}

	{
		USERS users;
		FIELD_COMBO *usrl = dia.newf_combo (
			MSG_U(F_USER,"Run as user")
			,user);
		SSTRINGS names;
		for (int i=0; i<users.getnb(); i++){
			SSTRING *name=new SSTRING(users.getitem(i)->getname());
			if ( name->is_empty() ) continue;
			names.add( name );
		}
		names.sort();
		for (int i=0; i<names.getnb(); i++){
			usrl->addopt (names.getitem(i)->get());
		}
	}

	{
		GROUPS groups;
		FIELD_COMBO *grpl = dia.newf_combo (
			MSG_U(F_GROUP,"Run in group (opt)")
			,group);
		SSTRINGS names;
		for (int i=0; i<groups.getnb(); i++){
			SSTRING *name=new SSTRING(groups.getitem(i)->getname());
			if ( name->is_empty() ) continue;
			names.add( name );
		}
		names.sort();
		for (int i=0; i<names.getnb(); i++){
			grpl->addopt (names.getitem(i)->get());
		}
	}
 
	{
		FIELD_COMBO *paths = dia.newf_combo (
			MSG_U(F_PATH,"Server path"), path );
		SSTRINGS names;
		SSTRING *name=new SSTRING("internal");
		names.add( name );
		name=new SSTRING(daemon_findpath( "tcpd" ) );
		names.add( name );
		names.sort();
		for (int i=0; i<names.getnb(); i++){
			paths->addopt (names.getitem(i)->get());
		}
	}
	dia.newf_str( MSG_U(F_ARGUMENTS,"Arguments"), arguments );

	int buttons;
	if ( new_inetdconf ) {
		buttons = (MENUBUT_CANCEL|MENUBUT_ACCEPT);
	} else {
		buttons = (MENUBUT_DEL|MENUBUT_CANCEL|MENUBUT_ACCEPT);
	}
	int ret = 0;
	int nof = 0;
	while (1){
		MENU_STATUS code = dia.edit(
			MSG_U(T_INETDCONF,
			"Internet server")
			,""
			,help_inetdconf
			,nof
			,buttons);
		if (code == MENU_CANCEL || code == MENU_ESCAPE){
			ret = -1;
			break;
		}else if (code == MENU_DEL){
			if ( xconf_delok() ) {
				write( MENU_DEL );
				ret = 1;
				break;
			}
		}else if (code == MENU_ACCEPT ) {
			if ( input_error( ) ) continue;
			write ( MENU_ACCEPT );
			ret = 0;
			break;
		}
	}
	return ret;
}

PRIVATE void INETDCONF::present(
	char *buf1,
	int size1,
	char *buf2,
	int size2)
{
	snprintf (buf1,size1-1,"[%s]",enabled ? "X" : " ");
	snprintf (buf2,size2-1,"%s\t%s\t%d/%s\t%s\t%s"
		,service_name.get()
		,socket_type.get()
		,port
		,protocol.get()
		,path.get()
		,arguments.get());
}

PRIVATE int INETDCONF::input_error( )
{
	if ( path.is_empty() ) {
		xconf_error(MSG_U(E_NOSERVERPATH,
			"Server path missing") );
		return( 1 );
	}
	if ( path.cmp( "internal" ) != 0 ) {
		if ( path.get()[0] != '/' ) {
			xconf_error(MSG_U(E_SERVERPATHINCORRECT,
				"Server path incorrect.\n"
				"First character must be a /\n"
				) );
			return( 1 );
		}
	}
	ETCSERVICE *etcservice = NULL;
	etcservice = etcservicelist->getitem( service_name.get() );
	if ( etcservice == NULL ) {
		etcservice = etcservicelist->getitem_alias( service_name.get() );
	}
	if ( etcservice == NULL ) {
		xconf_error(MSG_U(E_NOSERVICE,
			"Service %s\n"
			"does not exist."), service_name.get() );
		return( 1 );
	}
	/*
	 * Port is set here as it is not present in the dialog.
	 */
	port = etcservice->port;

	if ( user.is_empty() ) {
		user.setfrom( "root" );
	}

	ETCPROTOCOL *etcprotocol = NULL;
	etcprotocol = etcprotocollist->getitem( protocol.get() );
	if ( etcprotocol == NULL ) {
		etcprotocol = etcprotocollist->getitem_alias( protocol.get() );
	}
	if ( etcprotocol == NULL ) {
		xconf_error(MSG_U(E_NOPROTOCOL,
			"This protocol is missing.\n"
			"The protocol %s\n"
			"does not exist."), protocol.get() );
		return( 1 );
	}
	if ( enabled ) {
		for ( int i=0; i<inetdconflist->getnb(); i++ ) {
			INETDCONF *inetdconf = inetdconflist->getitem(i);
			if ( inetdconf->enabled
			&& ( inetdconf->port == port )
			&& ( inetdconf->inetdconf_line != inetdconf_line )
			&& ( inetdconf->protocol.cmp( protocol ) == 0 )
			) {
				xconf_error(MSG_U(E_DUPPORTPROTOCOL,
					"Port/protocol (%d/%s) is already.\n"
					"enabled in service (%s).\n"
					"(%s: lines %d and %d)\n"
					),
					port, protocol.get(), inetdconf->service_name.get(), f_config_file.getpath(), inetdconf->inetdconf_line, inetdconf_line );
				return( 1 );
			}
		}
	}
	return( 0 );
}


/**
 * INETDCONFLIST
 */

static int cmp_by_name (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	INETDCONF *s1 = (INETDCONF*)o1;
	INETDCONF *s2 = (INETDCONF*)o2;
	int ret = s1->service_name.cmp(s2->service_name);
	if (ret == 0){
		ret = s1->protocol.cmp(s2->protocol);
	}
	if (ret == 0){
		ret = s1->socket_type.cmp(s2->socket_type);
	}
	return ret;
}

PUBLIC void INETDCONFLIST::sort()
{
	ARRAY::sort (cmp_by_name);
}

PUBLIC INETDCONF *INETDCONFLIST::getitem (int no) const
{
	return (INETDCONF*)ARRAY::getitem (no);
}

PUBLIC INETDCONF *INETDCONFLIST::getitem (const char *_service_name) const
{
	INETDCONF *ret = NULL;
	int n = getnb();
	for (int i=0; i<n; i++) {
		INETDCONF *inetdconf = getitem(i);
		if (inetdconf->service_name.cmp(_service_name)==0){
			ret = inetdconf;
			break;
		}
	}
	return ret;
}

PUBLIC INETDCONFLIST::INETDCONFLIST( )
{
//fprintf(stderr,"INETDCONFLIST::INETDCONFLIST\n");
}

PRIVATE char * INETDCONFLIST::next_word(char *d, char *s, int size)
{
	while ( *s ) {
		switch ( *s ) {
			case ' ':
			case '\t':
			case '\n':
				s++;
				continue;
			default:
				break;
		}
		break;
	}
	for ( size--; *s && size; size-- ) {
		switch ( *s ) {
			case ' ':
			case '\t':
			case '\n':
				*d = '\0';
				return( s );
			default:
				*d++ = *s++;
				break;
		}
	}
	*d = '\0';
	return( s );
}

PRIVATE int INETDCONFLIST::valid_etcservice( char *word, INETDCONF *inetdconf )
{
	ETCSERVICE *etcservice = NULL;
	etcservice = etcservicelist->getitem( word );
	if ( etcservice == NULL ) {
//fprintf( stderr, "inetdconfedit.cc: valid_etcservice: word=%s etcservice->service_name=NULL\n", word);
		etcservice = etcservicelist->getitem_alias( word );
		if ( etcservice == NULL ) {
//fprintf( stderr, "inetdconfedit.cc: valid_etcservice: word=%s etcservice->service_name: alias=NULL\n", word);
			return( 0 );
		}
	}
//fprintf( stderr, "inetdconfedit.cc: etcservice->service_name=%s\n", etcservice->service_name.get());
	inetdconf->service_name.setfrom( word );
	inetdconf->port = etcservice->port;
	return( 1 );
}

PRIVATE int INETDCONFLIST::valid_socket_type( char *word, INETDCONF *inetdconf )
{
	if ( ( strcmp( word, "stream" ) == 0 )
	||   ( strcmp( word, "dgram" ) == 0 )
	||   ( strcmp( word, "raw" ) == 0 )
	||   ( strcmp( word, "rdm" ) == 0 )
	||   ( strcmp( word, "seqpacket" ) == 0 ) ) {
		inetdconf->socket_type.setfrom( word );
		return( 1 );
	}
	return( 0 );
}

PRIVATE int INETDCONFLIST::valid_protocol( char *word, INETDCONF *inetdconf )
{
	ETCPROTOCOL *etcprotocol = NULL;
	etcprotocol = etcprotocollist->getitem( word );
	if ( etcprotocol == NULL ) {
//fprintf( stderr, "inetdconfedit.cc: valid_etcprotocol: word=%s etcprotocol->service_name=NULL\n", word);
		etcprotocol = etcprotocollist->getitem_alias( word );
		if ( etcprotocol == NULL ) {
//fprintf( stderr, "inetdconfedit.cc: valid_etcprotocol: word=%s etcprotocol->service_name: alias=NULL\n", word);
			return( 0 );
		}
	}
	inetdconf->protocol.setfrom( word );
	return( 1 );
}

PRIVATE int INETDCONFLIST::valid_multi_max( char *word, INETDCONF *inetdconf )
{
	char *p = word;
	char *s = word;
	for ( ; *p && *p != '.'; p++ );
	if ( *p == '.' ) {
		*p++ = '\0';
		inetdconf->max = atoi( p );
	}
	if ( strncmp( s, "wait", 4 ) == 0 ) {
		inetdconf->multi = 1;
		return( 1 );
	} else {
		if ( strncmp( s, "nowait", 6 ) == 0 ) {
			inetdconf->multi = 0;
			return( 1 );
		}
	}
	return( 0 );
}

PRIVATE int INETDCONFLIST::valid_user_group( char *word, INETDCONF *inetdconf )
{
	char *p = word;
	char *s = word;
	for ( ; *p && *p != '.'; p++ );
	if ( *p == '.' ) {
		*p++ = '\0';
		inetdconf->user.setfrom( s );
		inetdconf->group.setfrom( p );
	} else {
		inetdconf->user.setfrom( s );
	}
	return( 1 );
}

PRIVATE int INETDCONFLIST::valid_path( char *word, INETDCONF *inetdconf )
{
	if ( strcmp( word, "internal" ) == 0 ) {
		inetdconf->path.setfrom( word );
		return( 1 );
	}
	if ( word[0] == '/' ) {
		inetdconf->path.setfrom( word );
		return( 1 );
	}
	return( 0 );
}

PRIVATE void INETDCONFLIST::add_service( int line_number, VIEWITEM *item )
{
//fprintf(stderr,"%s\n", item->line.get());
	bool enabled = true;
	INETDCONF *inetdconf = NULL;
	char word[1024];
	char *line = (char *)item->line.get();
	char *p = line;
	while ( *p && *p == '#' ) {
		p++;
		enabled = false;
	}
	if ( strlen( p ) < 3 ) return;
	int field = 0;
	int valid_keywords = 0;
	while ( *p ) {
		p = next_word( word, p, sizeof( word ));
//fprintf(stderr,"%s\n", word);
		field++;
		switch ( field ) {
			case K_FIELD_INETDCONF_NAME:
				inetdconf = new INETDCONF();
				if ( valid_etcservice( word, inetdconf ) ) {
					valid_keywords++;
				}
				if ( ! enabled ) {
					inetdconf->enabled = 0;
				}
				break;
			case K_FIELD_SOCKET_TYPE:
				if ( valid_socket_type( word, inetdconf ) ) {
					valid_keywords++;
				}
				break;
			case K_FIELD_PROTOCOL:
				if ( valid_protocol( word, inetdconf ) ) {
					valid_keywords++;
				}
				break;
			case K_FIELD_WAIT_MAX:
				if ( valid_multi_max( word, inetdconf ) ) {
					valid_keywords++;
				}
				break;
			case K_FIELD_USER_GROUP:
				if ( valid_user_group( word, inetdconf ) ) {
					valid_keywords++;
				}
				break;
			case K_FIELD_PATH:
				if ( valid_path( word, inetdconf ) ) {
					valid_keywords++;
				}
				break;
			default:	// Arguments
				if ( inetdconf->arguments.is_empty( ) ) {
					inetdconf->arguments.setfrom( word );
				} else {
					inetdconf->arguments.append(" " );
					inetdconf->arguments.append( word );
				}
				break;
		}
	}
	if ( valid_keywords == 6 ) {
		inetdconf->inetdconf_line = line_number;
//fprintf(stderr,"INETDCONFLIST::add_service: %s line=%d\n", inetdconf->service_name.get(), inetdconf->inetdconf_line);
		add( inetdconf );
	} else {
		delete inetdconf;
	}
	return;
}

/**
 * Read config file and parse inetdconfs
 */
PUBLIC void INETDCONFLIST::read()
{
	if ( etcservicelist == NULL ) {
		etcservicelist = new ETCSERVICELIST();
		etcservicelist->read( );
	}
	if ( etcprotocollist == NULL ) {
		etcprotocollist = new ETCPROTOCOLLIST();
		etcprotocollist->read( );
	}
	VIEWITEMS items;
	items.setcomcar( '\002' );	// Set comment to "impossible" char
	items.read( f_config_file );	// Read config file
//fprintf(stderr,"INETDCONFLIST::read items.getnb()=%d\n", items.getnb());
	for ( int line=0; line<items.getnb(); line++ ) {
		VIEWITEM *item = items.getitem(line);
		add_service( line, item );
	}
	/*
	 * Last modified set when config file read or written.
	 */
	modified_time = file_date( f_config_file.getpath() );
}

/**
 * Edit inetdconflist
 */
PUBLIC int INETDCONFLIST::edit()
{
//fprintf(stderr,"INETDCONFLIST::edit\n");
	DIALOG_LISTE *dia = NULL;
	int nof = 0;
	while (1) {
		if (dia == NULL) {
			sort();
			dia = new DIALOG_LISTE;
			dia->newf_head ("",MSG_U(H_INETD,"Enabled\tService\tType\tProtocol\tServer\tArguments"));
			for (int i=0; i<getnb(); i++){
				INETDCONF *inetdconf = getitem(i);
				char buf1[10],buf2[100];
				inetdconf->present(buf1,sizeof(buf1),buf2,sizeof(buf2));
				dia->new_menuitem( buf1, buf2 );
			}
			dia->addwhat (MSG_U(I_ADDINETDCONF,"Select [Add] to add a new server.\n"));
		}
		MENU_STATUS code = dia->editmenu (MSG_U(T_INETDCONFLIST,"Internet servers")
			,MSG_U(I_INETDCONFLIST,
			"This is the list of all services which are presently\n"
			"available in the system. These services can be enabled\n"
			"or disabled.\n"
			)
			,help_inetdconf
			,nof,0);
		bool mustdelete=false;
		if (code == MENU_QUIT || code == MENU_ESCAPE) {
			break;
		} else if (code == MENU_ADD) {
			INETDCONF *inetdconf = new INETDCONF;
			inetdconf->new_inetdconf = true;
			if ( editone(inetdconf) != -1 ) mustdelete = true;
		} else {
			INETDCONF *inetdconf = getitem( nof );
			inetdconf->new_inetdconf = false;
//fprintf(stderr,"INETDCONFLIST::edit: %s line=%d\n", inetdconf->service_name.get(), inetdconf->inetdconf_line);
			switch ( editone(nof) ) {
				case -1:
					mustdelete = false;
					break;
				case 1:
					mustdelete = true;
					break;
				case 0:
					mustdelete = true;
					break;
			}
		}
		if (mustdelete){
			delete dia;
			dia = NULL;
			if ( config_file_modified ) {
				inetdconflist->remove_all();
				inetdconflist->read();
				config_file_modified = false;
			}
		}
	}
	delete dia;
	return 0;
}

PUBLIC void inetdconf_edit( void )
{
//fprintf(stderr,"inetdconf_edit\n");
	inetdconflist = new INETDCONFLIST();
	inetdconflist->read();
	inetdconflist->edit();

	delete inetdconflist;
	inetdconflist = NULL;
	delete etcservicelist;
	etcservicelist = NULL;
	delete etcprotocollist;
	etcprotocollist = NULL;
}

PUBLIC void inetdconf_enable_service( int argc, char *service[], int status )
{
	inetdconflist = new INETDCONFLIST();
	inetdconflist->read();
	while ( argc-- ) {
//fprintf(stderr,"inetdconf_enable_service: argc=%d %s\n", argc, service[argc]);
		INETDCONF *inetdconf = inetdconflist->getitem( service[argc] );
		if ( inetdconf != NULL ) {
//fprintf(stderr,"inetdconf_enable_service: %s\n", inetdconf->service_name.get());
			inetdconf->enabled = status;
			inetdconf->write( MENU_ACCEPT );
		}
	}
	delete inetdconflist;
	inetdconflist = NULL;
	delete etcservicelist;
	etcservicelist = NULL;
	delete etcprotocollist;
	etcprotocollist = NULL;
}

