#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <grp.h>
#include "internal.h"
#include "userconf.h"
#include "../paths.h"
#include <misc.h>
#include "userconf.m"
#include <quota.h>
#include <dialog.h>

static USERCONF_HELP_FILE help_group ("group");
static USERCONF_HELP_FILE help_homebase ("homebase");

PRIVATE void GROUP::settbmem (char **members)
{
	tbmem.delall();
	if (members != NULL){
		int i=0;
		while (members[i] != NULL){
			tbmem.add (new SSTRING(members[i]));
			i++;
		}
	}
}

PRIVATE void GROUP::init(
	const char *_name,
	const char *_passwd,
	int _gid,
	char **members)
{
	name.setfrom (_name);
	passwd.setfrom(_passwd);
	gid = _gid;
	settbmem (members);
}

PUBLIC GROUP::GROUP(
	const char *_name,
	const char *_passwd,
	int _gid,
	char **members)
{
	init (_name,_passwd,_gid,members);
}

PUBLIC GROUP::GROUP()
{
	init ("","",-1,NULL);
}
PUBLIC GROUP::GROUP(const char *line)
{
	// We split the line in four. The end of the line is
	// the list of group members which may be very long
	// str_splitline will pick the first 3 fields
	char tb[16][100];
	str_splitline (line,':',tb,3);
	init (tb[0],tb[1],atoi(tb[2]),NULL);
	line = strrchr(line,':');
	if (line != NULL){
		str_splitline (line+1,',',tbmem);
	}
}
PUBLIC GROUP::~GROUP()
{
}

/*
	Write one record of /etc/passwd
*/
PUBLIC void GROUP::write(FILE *fout)
{
	fprintf (fout,"%s:%s:%d:",name.get(),passwd.get(),gid);
	char *sep = "";
	int nb = tbmem.getnb();
	int len = 0;
	for (int i=0; i<nb; i++){
		const char *member = tbmem.getitem(i)->get();
		if (len > 80){
			fputs ("\\\n",fout);
			len = 0;
		}
		fputs (sep,fout);
		fputs (member,fout);
		sep = ",";
		len += strlen(member)+1;
	}
	fputc ('\n',fout);
}

PUBLIC const char *GROUP::getname()
{
	return name.get();
}
PUBLIC int GROUP::getgid()
{
	return gid;
}

/*
	Check if a group is correctly configured.
	Return -1 if not.
*/
PRIVATE int GROUP::check(
	USERS &users,
	GROUPS &groups,
	GROUP *realone)
{
	char status[1000];
	status[0] = '\0';
	GROUP *other = groups.getitem(name.get());
	if (other != NULL && other != realone){
		strcat (status,MSG_U(E_GROUPEXIST,"Group already exist\n"));
	}
	other = groups.getfromgid(gid);
	if (other != NULL && other != realone){
		strcat (status,MSG_U(E_GROUPEXISTID
			,"Group already exist (Group ID)\n"));
	}
	int nb = tbmem.getnb();
	for (int i=0; i<nb; i++){
		const char *user = tbmem.getitem(i)->get();
		if (users.getitem(user)==NULL){
			strcat (status,MSG_U(E_UNKNOWNMEMBER,"Unknown member user "));
			strcat (status,user);
			strcat (status,"\n");
		}
	}
	int ret = 0;
	if (status[0] != '\0'){
		xconf_error ("%s",status);
		ret = -1;
	}
	return ret;
}

static char **group_parsemem (const char *str, int &nb)
{
	nb = 0;
	char **ret = (char**)malloc_err (100*sizeof(char*));
	while (str[0] != '\0'){
		str = str_skip(str);
		char member[200];
		char *ptmem = member;
		while (*str > ' ') *ptmem++ = *str++;
		if (ptmem > member){
			*ptmem = '\0';
			ret[nb++] = strdup(member);
		}
	}
	ret[nb] = NULL;
	return ret;
}
const char K_CREATEHOME[]="createhome";
const char K_CREATEPERM[]="createperm";
const char K_HOMEBASE[]="homebase";

/*
	Check and create the home base directory if needed
*/
static int group_checkhomebase(const char *base)
{
	int ret = 0;
	if (base[0] != '\0'
		&& file_type(base) == -1){
		ret = -1;
		char buf[1000];
		snprintf (buf,sizeof(buf)-1,MSG_U(Q_MISSBASE
			,"Home base %s does not exist\n"
			 "Do you want to create it"),base);
		if (xconf_yesno(MSG_U(T_MISSBASE,"Missing home base"),buf
			,help_homebase)==MENU_YES){
			ret = file_mkdir (base,0,0,0755,NULL);
		}
	}
	return ret;
}


/*
	Edit the specification of a user.
	Return -1 if the user escape without accepting the changes.
	Return 0 if the record was changed
	Return 1 if the record must be deleted
*/
PUBLIC int GROUP::edit(USERS &users, GROUPS &groups)
{
	bool is_new = gid == -1;
	DIALOG dia;
	dia.newf_title (MSG_R(T_BASE),1,"",MSG_R(T_BASE));
	int field_name = dia.getnb();
	dia.newf_str (MSG_U(F_GROUPNAME,"Group name"),name);
	if (is_new) gid = groups.getnew();
	dia.newf_num (MSG_U(F_GROUPID,"Group ID"),gid);
	SSTRING altmem;
	int nbmem = tbmem.getnb();
	for (int i=0; i<nbmem; i++){
		altmem.append (tbmem.getitem(i)->get());
		if (i < nbmem - 1) altmem.append (" ");
	}
	int field_altmem = dia.getnb();
	dia.newf_str (MSG_U(F_ALTMEM,"Alternate members(opt)"),altmem);

	QUOTA_EDIT qedit (is_new ? (char*)NULL : name.get(), QUOTA_GROUP);
	qedit.setupdia (dia,MSG_U(T_GDISKQUOTA,"Group disk quota"));

	QUOTA_EDIT qeditd (is_new ? (char*)NULL : name.get(), QUOTA_USERDEF_G);
	qeditd.setupdia (dia,MSG_U(T_UDEFDISKQUOTA,"Members default disk quota"));

	char createhome = 1;
	SSTRING homebase;
	int createperm = 0700;
	if (!is_new){
		const char *nam = name.get();
		createhome = linuxconf_getvalnum (K_CREATEHOME,nam,1);
		createperm = linuxconf_getvalnum (K_CREATEPERM,nam,createperm);
		homebase.setfrom (linuxconf_getval (K_HOMEBASE,nam));
	}

	dia.newf_title (MSG_U(T_HOMEDIRS,"Directories"),1,"",MSG_R(T_HOMEDIRS));
	dia.newf_chk (MSG_U(F_DIFFDIR,"Different directory"),createhome
		,MSG_U(I_CREATEHOME,"for each members"));
	int field_home = dia.getnb();
	dia.newf_str (MSG_U(F_HOMEBASE,"Home base directory"),homebase);
	dia.newf_octnum (MSG_U(F_CREATEPERM,"Creation permissions"),createperm);


	int field = 0;
	int ret = -1;
	while (1){
		MENU_STATUS code = dia.edit(MSG_U(T_GROUPSPEC,"Group specification")
			,MSG_U(INTRO_GROUPSPEC,"You must specify at least the name\n")
			,help_group
			,field
			,MENUBUT_ACCEPT|MENUBUT_CANCEL|MENUBUT_DEL);
		if (code == MENU_CANCEL || code == MENU_ESCAPE){
			dia.restore();
			break;
		}else if (perm_rootaccess(
			MSG_U(P_GROUPDB,"to maintain the group database"))){
			if (code == MENU_DEL){
				if (xconf_areyousure(MSG_U(Q_DELGROUP
					,"Confirm deletion of group definition"))){
					ret = 1;
					break;
				}
			}else if (name.strchr(':')!=NULL){
				xconf_error (MSG_R(E_NO2PT));
				field = field_name;
			}else if (altmem.strchr(':')!=NULL){
				xconf_error (MSG_R(E_NO2PT));
				field = field_altmem;
			}else if (!homebase.is_empty()
				&& file_type(homebase.get()) != -1
				&& file_type(homebase.get()) != 1){
				xconf_error (MSG_U(E_IVLDDIR,"Home base %s exist\n"
					"but is not a directory"),homebase.get());
				field = field_home;
			}else{
				int newgid = gid;
				int nb;
				char **members = group_parsemem (altmem.get(),nb);
				const char *nam = name.get();
				GROUP tmp(nam,passwd.get(),newgid,members);
				if (tmp.check(users,groups,this)==0
					&& group_checkhomebase(homebase.get())!=-1){
					qedit.setname (name.get());
					qedit.save(NULL);
					qeditd.setname (name.get());
					qeditd.save(NULL);
					settbmem (members);
					// Update conf.linuxconf only if the value
					// differs from the default
					linuxconf_setcursys (subsys_policies);
					if (createhome){
						linuxconf_removeall (K_CREATEHOME,nam);
					}else{
						linuxconf_replace (K_CREATEHOME,nam,0);
					}
					if (homebase.is_empty()){
						linuxconf_removeall (K_HOMEBASE,nam);
					}else{
						linuxconf_replace (K_HOMEBASE,nam,homebase);
					}
					if (createperm == 0700){
						linuxconf_removeall (K_CREATEPERM,nam);
					}else{
						linuxconf_replace (K_CREATEPERM,nam,createperm);
					}
					if (linuxconf_save() != -1){
						ret = 0;
						break;
					}
				}
				tbstr_free (members,nb);
				free (members);
			}
		}
	}
	return ret;
}

/*
	Return the base directory where home must be created for a group member.
	Return NULL if not defined for this group
*/
const char *group_gethomebase (const char *gname)
{
	return linuxconf_getval (K_HOMEBASE,gname);
}

/*
	Return true if a home directory is needed for members of this group
*/
bool group_homeneeded(const char *gname)
{
	return linuxconf_getvalnum (K_CREATEHOME,gname,1) != 0;
}

/*
	Return the creation permission bits for the home directory of
	members of this group
*/
int group_getcreateperm(const char *gname)
{
	return linuxconf_getvalnum (K_CREATEPERM,gname
		,policies_getcreateperm());

}


/*
	Remove one user as a member of a group (if there)
	Return != 0 if the user was a member of that group.
*/
PUBLIC int GROUP::delmember(const char *user)
{
	int ret = 0;
	int n = tbmem.getnb();
	for (int i=0; i<n; i++){
		if (tbmem.getitem(i)->cmp(user)==0){
			tbmem.remove_del (i);
			ret = 1;
			break;
		}
	}
	return ret;
}

/*
	Add one user as a member of a group
*/
PUBLIC void GROUP::addmember(const char *user)
{
	tbmem.add (new SSTRING(user));
}

/*
	Return true is a user is a member of this group (alternate group)
*/
PUBLIC bool GROUP::is_member(const char *user)
{
	bool ret = false;
	int n = tbmem.getnb();
	for (int i=0; i<n; i++){
		if (tbmem.getitem(i)->cmp(user)==0){
			ret = true;
			break;
		}
	}
	return ret;
}
