#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <limits.h>
#include <sys/stat.h>
#include <popen.h>
#include "netadm.h"
#include "netadm.m"
#include "../paths.h"
#include "internal.h"

static NETADM_HELP_FILE help_export ("export");

static bool debug=false;

static void n_debug (const char *ctl, ...)
{
	if (debug){
		va_list list;
		va_start (list,ctl);
		vfprintf (stderr,ctl,list);
		va_end (list);
	}
}
/*
	Establish the connection with the remote end.
	At this point, we have a link. Now we must send the password
	and verify the protocol version
*/
static int export_connect (POPEN &pop, const char *pass)
{
	int ret = -1;
	bool sentver = false;
	bool sentpass = false;
	bool expectack = false;
	FILE *fout = pop.getfout();
	bool done = false;
	while (!done && pop.wait(100)>0){
		char line[200];
		while (pop.readout(line,sizeof(line)-1)!=-1){
			int code = atoi(line);
			if (code == PROTO_C_PROMPT){
				if (expectack){
					xconf_error (MSG_U(E_OUTOFSYNC,"Protocol out of sync"));
					done = true;
					break;
				}else if (!sentver){
					n_debug("Sending protocol version: %s\n",PROTO_VERSION);
					fprintf (fout,"version %s\n",PROTO_VERSION);
					expectack = true;
					sentver = true;
				}else if (!sentpass){
					n_debug ("Sending password\n");
					fprintf (fout,"password %s\n",pass);
					expectack = true;
					sentpass = true;
				}
			}else if (code == PROTO_C_ERROR){
				line[60] = '\0';
				xconf_error (MSG_U(E_EXPORTERR
					,"Export error\n"
					 "%s")
					,line);
				done = true;
				break;
			}else if (code == PROTO_C_COMMENT){
				dialog_consout ("%s",line);
			}else if (code == PROTO_C_ACK){
				if (!expectack){
					xconf_error (MSG_R(E_OUTOFSYNC));
					done = true;
					break;
				}else{
					n_debug ("Receiving ack: %s",line);
					expectack = false;
					if (sentpass){
						ret = 0;
						done = true;
						break;
					}
				}
			}else{
				break;
			}
		}
	}
	n_debug ("Initial connection: %s\n",ret==-1 ? "failure" : "success");
	return ret;
}

/*
	Read the list of file to export and the MD5 sum of each
	Return -1 if any error
*/
static int export_readfiles(
	const char *group,
	SSTRINGS &files,
	SSTRINGS &sums)
{
	int ret = -1;
	char indexsum[PATH_MAX];
	snprintf (indexsum,PATH_MAX-1,"%s/%s/index.sum"
		,ETC_LINUXCONF_GROUPS,group);
	FILE *fin = xconf_fopen (indexsum,"r");
	if (fin != NULL){
		char line[2*PATH_MAX];
		while (fgets(line,sizeof(line)-1,fin)!=NULL){
			strip_end (line);
			char path[PATH_MAX],sum[PATH_MAX];
			if (sscanf (line,"%s %s",path,sum)==2){
				files.add (new SSTRING (path));
				sums.add (new SSTRING (sum));
			}
		}
		fclose (fin);
		ret = 0;
	}
	return ret;
}



/*
	Send all file needed to the member of the admin group
*/
static int export_send (POPEN &pop, const char *group)
{
	int ret = -1;
	SSTRINGS subsyss;
	SSTRINGS files;
	SSTRINGS sums;
	if (netadm_readsubsyss(group,subsyss)!=-1
		&& export_readfiles(group,files,sums)!=-1){
		files.add (new SSTRING("/index.subsys"));
		FILE *fout = pop.getfout();
		bool expectack = false;
		bool sentput = false;
		bool sentdim = false;
		bool sentfile = false;
		int nofile = 0;
		char filepath[PATH_MAX];
		bool done = false;
		while (!done){
			char line[200];
			while (pop.readout(line,sizeof(line)-1)!=-1){
				int code = atoi(line);
				if (code == PROTO_C_PROMPT){
					if (expectack){
						xconf_error (MSG_R(E_OUTOFSYNC));
						done = true;
						break;
					}else if (!sentput){
						//expectack = true;
						const char *f = files.getitem(nofile)->get();
						dialog_consout ("%s\n",f);
						sprintf (filepath,"%s/%s%s"
							,ETC_LINUXCONF_GROUPS,group
							,f);
						nofile++;
						n_debug ("Sending file %s\n",filepath);
						fprintf (fout,"putfile %s\n",filepath);
						sentput = true;
					}else if (!sentdim){
						struct stat st;
						if (stat(filepath,&st)==-1){
							xconf_error (MSG_U(E_STAT,"Can't stat file\n%s")
								,filepath);
							done = true;
							break;
						}else{
							n_debug ("File dimension: %ld\n",st.st_size);
							fprintf (fout,"dimension %ld\n",st.st_size);
							expectack = true;
							sentdim = true;
						}
					}
				}else if (code == PROTO_C_ERROR){
					xconf_error (MSG_R(E_EXPORTERR),line);
					done = true;
					break;
				}else if (code == PROTO_C_COMMENT){
					dialog_consout ("%s",line);
				}else if (code == PROTO_C_ACK){
					if (!expectack){
						xconf_error (MSG_R(E_OUTOFSYNC));
						done = true;
						break;
					}else if (sentfile){
						n_debug ("Receing ack for sendfile\n");
						expectack = false;
						sentfile = false;
						if (nofile == files.getnb()){
							ret = 0;
							done = true;
							break;
						}
						
					}else if (sentdim){
						n_debug ("Receing ack for send dimension, now sending data\n");
						// expectack = false;
						// At the end of the copy, another acknoledge will come
						FILE *fdata = xconf_fopen (filepath,"r");
						if (fdata == NULL){
							done = true;
							break;
						}else{
							char buf[30000];
							int len;
							while ((len=fread(buf,1,sizeof(buf),fdata))>0){
								fwrite (buf,1,len,fout);
							}
							fclose (fdata);
							sentdim = false;
							sentput = false;
							sentfile = true;
						}
					}else{
						n_debug ("File received: %s",line);
						expectack = false;
					}
				}else{
					break;
				}
			}
			if (done || pop.wait(100)<=0) break;
		}
	}
	n_debug ("Sending files status: %s\n",ret==-1 ? "failure" : "success");
	return ret;
}

/*
	Send one command by respecting the PROMPT and ACK protocol
*/
static int export_sendcmd (POPEN &pop, const char *cmd)
{
	int ret = -1;
	FILE *fout = pop.getfout();
	bool expectack = false;
	bool done = false;
	while (!done){
		char line[200];
		while (pop.readout(line,sizeof(line)-1)!=-1){
			int code = atoi(line);
			if (code == PROTO_C_PROMPT){
				if (expectack){
					xconf_error (MSG_R(E_OUTOFSYNC));
					done = true;
					break;
				}else{
					fprintf (fout,"%s\n",cmd);
					expectack = true;
				}
			}else if (code == PROTO_C_ERROR){
				xconf_error (MSG_R(E_EXPORTERR),line);
				done = true;
				break;
			}else if (code == PROTO_C_COMMENT){
				dialog_consout ("%s",line);
			}else if (code == PROTO_C_ACK){
				if (!expectack){
					xconf_error (MSG_R(E_OUTOFSYNC));
				}else{
					ret = 0;
				}
				done = true;
				break;
			}
			if (done || pop.wait(100)<=0) break;
		}
	}
	return ret;
}


PRIVATE int CLUSTER::exportone(
	const char *member,
	const char *pass)
{
	int ret = -1;
	char arg[2*PATH_MAX];
	snprintf (arg,sizeof(arg)-1,"-l netadm %s",member);
	n_debug ("Executing: ssh %s\n",arg);
	POPEN pop ("ssh",arg);
	if (pop.isok()){
		FILE *fout = pop.getfout();
		if (export_connect(pop,pass)!=-1
			&& export_send (pop,id.get())!=-1){
			char buf[1000];
			snprintf (buf,sizeof(buf)-1
				,MSG_U(I_EXPORTOK
					,"The administration group %s\n"
					 "has been successfully copied to machine\n"
					 "%s.\n"
					 "Do you want to install the various new config files\n"
					 "immediatly and activate the changes ?")
				,id.get(),member);
			if (dialog_yesno(MSG_U(T_EXPORTOK,"Export completed")
				,buf
				,help_export)==MENU_YES){
				char import_cmd[PATH_MAX];
				sprintf (import_cmd,"import %s",id.get());
				if (export_sendcmd (pop,import_cmd)!=-1
					&& export_sendcmd (pop,"update")!=-1){
					ret = 0;
					xconf_notice (MSG_U(N_UPDATED,"Station %s updated"),member);
				}
			}else{
				ret = 0;
			}
			fprintf (fout,"quit\n");
		}
		pop.close();
	}
	return ret;
}

PUBLIC int CLUSTER::exportall()
{
	int ret = -1;
	for (int i=0; i<members.getnb(); i++){
		const char *m = members.getitem(i)->get();
		DIALOG dia;
		SSTRING pass;
		dia.newf_pass (MSG_U(F_ROOTPASS,"root password"),pass);
		char do_debug=0;
		dia.newf_chk  ("",do_debug,MSG_U(I_DEBUG,"Enable protocol debuggin"));
		char buf[200];
		snprintf (buf,sizeof(buf)-1
			,MSG_U(I_EXPORTINGTO,"Exporting to %s\n"
			 "\n"
			 "Please enter root password so linuxconf\n"
			 "can connect to the netadm special account\n"
			 "using ssh (secure shell)")
			,m);
		if (dia.edit(m,buf,help_nil)==MENU_ACCEPT){
			debug = do_debug != 0;
			ret |= exportone (m,pass.get());
		}
	}
	return ret;
}

