#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "diadef.h"
#include "dialog.h"
#include "dialog.m"
#include "../diajava/proto.h"
#include "../diajava/diajava.h"

enum FLDNUM_TYPE{ FLDNUM_DEC, FLDNUM_HEX, FLDNUM_OCT, FLDNUM_DBL };

class FIELD_NUM: public FIELD_STRING_BASE{
protected:
	double *dblval;
	int    *intval;
	double dbl_backup;
	int int_backup;
	FLDNUM_TYPE type;
	int nbdecimals;
	/*~PROTOBEG~ FIELD_NUM */
public:
	FIELD_NUM (const char *_prompt,
		 double *_dblval,
		 int *_intval,
		 FLDNUM_TYPE _type,
		 int _nbdecimals);
protected:
	void dokey (WINDOW *w, int key, FIELD_MSG&msg);
public:
	void html_draw (int nof);
	int post_validate (void);
	void reload (const char *dianame, int nof);
	void restore (void);
	void save (void);
private:
	bool validchar (char car);
public:
	/*~PROTOEND~ FIELD_NUM */
};

static const char *tb_format[]={"%d","%x","%o","%f"};

PUBLIC FIELD_NUM::FIELD_NUM(
	const char *_prompt,
	double *_dblval,
	int *_intval,
	FLDNUM_TYPE _type,
	int _nbdecimals)
	: FIELD_STRING_BASE (_prompt, _dblval != NULL ? 30 : 10)
{
	type = _type;
	dbl_backup = 0;
	int_backup = 0;
	dblval = _dblval;
	intval = _intval;
	if (_dblval != NULL){
		dbl_backup = *_dblval;
		int_backup = (int)dbl_backup;
	}else{
		int_backup = *_intval;
	}
	nbdecimals = _nbdecimals;
	if (_nbdecimals == 0){
		sprintf (buf, tb_format[type],int_backup);
	}else{
		sprintf (buf, "%.*f",nbdecimals,dbl_backup);
	}
}

PRIVATE bool FIELD_NUM::validchar (char car)
{
	bool ret = false;
	if (type == FLDNUM_DEC){
		ret = isdigit(car);
	}else if (type == FLDNUM_HEX){
		ret = isxdigit(car);
	}else if (type == FLDNUM_OCT){
		ret = car >= '0' && car < '8';
	}
	return ret;
}

PROTECTED  void FIELD_NUM::dokey(WINDOW *w, int key, FIELD_MSG &msg)
{
	int charok = validchar (key);
	if (charok
		|| (key >= KEY_MIN && key <= KEY_MAX)
		|| key < ' '
		|| (x.input + x.scroll == 0 && key == '-')
		|| (nbdecimals > 0 && key == '.')){
		FIELD_STRING_BASE::dokey(w,key,msg);
	}
}
	
/*
	Draw the field with the prompt
*/
PUBLIC void FIELD_NUM::html_draw(int nof)
{
	char key[100];
	format_htmlkey (key,nof);
	html_printf ("<tr><td>%s<td>",prompt);
	html_defvar ("integer",key,buf,"size=10 maxlength=20");
	html_defvarcur (key,buf);
}



PUBLIC void FIELD_NUM::save()
{
	if (dblval != NULL){
		sscanf (buf,"%lf",dblval);
	}else{
		sscanf (buf,tb_format[type] ,intval);
	}
}

PUBLIC void FIELD_NUM::restore()
{
	if (dblval != NULL){
		*dblval = dbl_backup;
	}else{
		*intval = int_backup;
	}
}
PUBLIC void FIELD_NUM::reload(const char *dianame, int nof)
{
	if (dblval != NULL){
		if (nbdecimals != 0){
			sprintf (buf, "%.*f",nbdecimals,dbl_backup);
		}else{
			sprintf (buf,"%d",(int)*dblval);
		}
	}else{
		sprintf (buf,tb_format[type],*intval);
	}
	sendval (dianame,nof,'S',buf);
}
PUBLIC int FIELD_NUM::post_validate()
{
	int ret = 0;
	const char *pt = buf;
	if (*pt == '-') pt++;
	while (*pt != '\0'){
		int charok = validchar (*pt);
		if (*pt == '.'){
			if (nbdecimals == 0){
				xconf_error (MSG_U(E_NOPT
					,"No decimal point allowed"));
				ret = -1;
				break;
			}
		}else if (!charok){
			xconf_error (MSG_U(E_NONDIGIT
				,"Only digits and the minus sign are allowed"));
			ret = -1;
			break;
		}
		pt++;
	}
	return ret;
}



/*
	Add a numerical field to the dialog.
*/
PUBLIC FIELD *DIALOG::newf_num(
	const char *prompt,
	int &val)
{
	FIELD_NUM *s = new FIELD_NUM(prompt,NULL,&val,FLDNUM_DEC,0);
	add (s);
	return s;
}
/*
	Add an hexadecimal field to the dialog.
*/
PUBLIC FIELD *DIALOG::newf_hexnum(
	const char *prompt,
	int &val)
{
	FIELD_NUM *s = new FIELD_NUM(prompt,NULL,&val,FLDNUM_HEX,0);
	add (s);
	return s;
}
/*
	Add an octal field to the dialog.
*/
PUBLIC FIELD *DIALOG::newf_octnum(
	const char *prompt,
	int &val)
{
	FIELD_NUM *s = new FIELD_NUM(prompt,NULL,&val,FLDNUM_OCT,0);
	add (s);
	return s;
}

/*
	Add a floating point field to the dialog.
*/
PUBLIC FIELD *DIALOG::newf_dbl(
	const char *prompt,
	double &val,
	int nbdecimals)
{
	FIELD_NUM *s = new FIELD_NUM(prompt,&val,NULL,FLDNUM_DBL,nbdecimals);
	add (s);
	return s;
}

class FIELD_SLIDER: public FIELD_NUM{
protected:
	int minval,maxval;		// To scale le slider
	/*~PROTOBEG~ FIELD_SLIDER */
public:
	FIELD_SLIDER (const char *_prompt,
		 int &_val,
		 int _minval,
		 int _maxval);
	void gui_draw (int nof, int &level);
	/*~PROTOEND~ FIELD_SLIDER */
};

PUBLIC FIELD_SLIDER::FIELD_SLIDER(
	const char *_prompt,
	int &_val,
	int _minval,
	int _maxval)
	: FIELD_NUM (_prompt,NULL,&_val,FLDNUM_DEC,0)
{
	minval = _minval;
	maxval = _maxval;
}


PUBLIC void FIELD_SLIDER::gui_draw(int nof, int &level)
{
	if (diajava_slider){
		if (prompt[0] != '\0'){
			diagui_send_Label (prompt);
		}else{
			diagui_sendcmd (P_Skip,"1\n");
		}
		if (readonly){
			diagui_send_Label (buf);
		}else{
			diagui_sendcmd (P_Slider,"S%d 200 %d %d %s\n",nof
				,minval,maxval,buf);
		}
	}else{
		FIELD_NUM::gui_draw(nof,level);
	}
}

/*
	Add a horizontal slider field (integer input) to the dialog.
*/
PUBLIC FIELD *DIALOG::newf_slider(
	const char *prompt,
	int &val,
	int minval,
	int maxval)
{
	FIELD_SLIDER *s = new FIELD_SLIDER(prompt,val,minval,maxval);
	add (s);
	return s;
}

class FIELD_GAUGE: public FIELD_NUM{
	int range;
	/*~PROTOBEG~ FIELD_GAUGE */
public:
	FIELD_GAUGE (const char *_prompt,
		 int &_val,
		 int _range);
	void drawtxt (WINDOW *dialog);
	void gui_draw (int nof, int &);
	void html_draw (int);
	/*~PROTOEND~ FIELD_GAUGE */
};

PUBLIC FIELD_GAUGE::FIELD_GAUGE(
	const char *_prompt,
	int &_val,
	int _range)
	: FIELD_NUM (_prompt,NULL,&_val,FLDNUM_DEC,0)
{
	range = _range;
}

PUBLIC void FIELD_GAUGE::drawtxt (WINDOW *dialog)
{
	wattrset(dialog, inputbox_attr);
	wmove(dialog, box.y,box.x);
	int stop = *intval * box.width/range;
	for (int i=0; i<box.width; i++){
		waddstr (dialog,i<stop ? "-" : " ");
	}
}
/*
	Draw the gauge in html. I guess we can do something smart here.
	But so far, there is no big interest for a gauge in html. This
	will come at some point.
*/
PUBLIC void FIELD_GAUGE::html_draw(int )
{
	html_printf ("<tr><td>%s<td>",prompt);
	int stop = *intval * range/box.width;
	for (int i=0; i<stop; i++){
		html_printf ("-");
	}
	html_printf ("\n");
}


PUBLIC void FIELD_GAUGE::gui_draw(int nof, int &)
{
	if (prompt[0] != '\0'){
		diagui_send_Label (prompt);
	}else{
		diagui_sendcmd (P_Skip,"1\n");
	}
	if (diajava_gauge){
		diagui_sendcmd (P_Gauge,"S%d 200 %d %s\n",nof,range,buf);
	}else{
		char tmp[20];
		snprintf (tmp,sizeof(tmp)-1,"%s/%d  ",buf,range);
		diagui_send_Label (tmp);
	}
}

/*
	Add a horizontal slider field (integer input) to the dialog.
*/
PUBLIC FIELD *DIALOG::newf_gauge(
	const char *prompt,
	int &val,
	int range)
{
	FIELD_GAUGE *s = new FIELD_GAUGE(prompt,val,range);
	add (s);
	return s;
}


