#! /usr/bin/python

# Processor for language definition files
__copyright__ = '''
 Copyright (C) 1999 The Software in the Public Interest (SPI)
 Written by Michael Sobolev <mss@transas.com>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
'''

import sys, os, time, string
from xml.parsers.xmlproc import xmlval,catalog,xcatalog

# Import aliases..
from iconv import convert

# This function should return a correct contents (!) for C string constant
def escape (str):
    return str

def from_utf8 (charset, what):
    return convert ('UTF-8', charset, what)

class Storage:
    def __init__ (self, name):
        self.name = name
        self.ref = []

    def add (self, what):
        try:
            self.ref.index (what)
        except:
            self.ref.append (what)

    def reference (self, what):
        return '%s[%s]' % (self.name, self.ref.index (what))

    def declaration (self):
        return 'static char *%s[] = { %s };' % (self.name, string.joinfields (map (lambda x : '"%s"' % x, self.ref), ', '))

class Language:
    ename = ""          # English name for the language
    acm = ""            # Charset name (acutally, acm name!) to use for displaying choice for this language
    font = ""           # Font name 

    name = ""           # Language name in the appropriate character set
    hint = ""           # Help line to show when the users hightlights the language

    list = None

    def __init__ (self, fonts, acms, attrs):
        self.fonts = fonts
        self.acms = acms

        self.ename = attrs['name']
        self.charset = string.upper (attrs['charset'])
        self.acm = attrs['acm']
        self.font = attrs['font']

        self.fonts.add (self.font)
        self.acms.add (self.acm)

        self.name = ""
        self.hint = ""

        self.list = None

    def definition_name (self):
        return string.lower (self.ename)

    def definition (self):
        return '{ "%s", "%s", NULL, NULL, "%s", NULL }' % (escape (self.ename), from_utf8 (self.charset, escape (self.name)), from_utf8 (self.charset, escape (self.hint)))

    def initialization (self, output, name):
        output.write ('\t%s.font = %s;\n' % (name, self.fonts.reference (self.font)))
        output.write ('\t%s.acm = %s;\n' % (name, self.acms.reference (self.acm)))

    def __cmp__ (self, other):
        if self.name < other.name:
            result = -1
        elif self.name == other.name:
            result = 0
        else:
            result = 1

        return result

class List:
    name = ""           # List name (window title when choosing from this list)

    items = []

    def __init__ (self):
        self.name = ""
        self.items = []

    def definition (self, output, charset, name):
        output.write ('static struct language_list %s = {\n' % name)

        output.write ('\t"%s",\n' % from_utf8 (charset, self.name))
        output.write ('\t{\n')

        for item in self.items:
            if type (item) == type (1):     # index
                output.write ('    { 1, NULL }')
            else:
                output.write ('    { 2, NULL }')

            output.write (',\n')

        output.write ('    { 0 }\n')
        output.write ('\t}\n')
        output.write ('};\n\n')

        for i in range (len (self.items)):
            item = self.items[i]

            if type (item) != type (1):     # index
                item.definition (output, charset, name + '_' + str (i))

    def initialization (self, output, name):
        for i in range (len (self.items)):
            item = self.items[i]

            if type (item) == type (1):     # index
                output.write ('\t%s.items[%s].p = all_items + %s;\n' % (name, i, item))
            else:
                output.write ('\t%s.items[%s].p = &%s;\n' % (name, i, name + '_' + str (i)))

        for i in range (len (self.items)):
            item = self.items[i]

            if type (item) != type (1):     # index
                item.initialization (output, name + '_' + str (i))

class Item:
    locale = ""         # Locale name.  It looks like it's impossible to deduct the locale name from other parameters...
    acm = ""            # Charset to use for this very language variant
    fontname = ""       # Font to use for this very language variant
    keymap = ""         # Name of prefered keymap
    messages = ""       # The name of file with messages for dbootstrap program

    def __init__ (self, charset, fonts, acms, keymaps, locales, msgcats, attrs):
        self.fonts = fonts
        self.acms = acms
        self.keymaps = keymaps
        self.locales = locales
        self.msgcats = msgcats

        self.name = ""
        self.charset = charset

        self.locale = attrs['locale']
        self.acm = attrs['acm']
        self.font = attrs['font']
        self.keymap = attrs['keymap']
        self.msgcat = attrs['msgcat']

        self.fonts.add (self.font)
        self.acms.add (self.acm)
        self.keymaps.add (self.keymap)
        self.locales.add (self.locale)
        self.msgcats.add (self.msgcat)

    def __cmp__ (self, other):
        if self.name < other.name:
            result = -1
        elif self.name == other.name:
            result = 0
        else:
            result = 1

        return result

    def definition (self):
        return '{ "%s", NULL, NULL, NULL, NULL, NULL }' % from_utf8 (self.charset, self.name)

    def initialization (self, output, name):
        output.write ('\t%s.locale = %s;\n' % (name, self.locales.reference (self.locale)))
        output.write ('\t%s.font = %s;\n' % (name, self.fonts.reference (self.font)))
        output.write ('\t%s.acm = %s;\n' % (name, self.acms.reference (self.acm)))
        output.write ('\t%s.keymap = %s;\n' % (name, self.keymaps.reference (self.keymap)))
        output.write ('\t%s.msgcat = %s;\n' % (name, self.msgcats.reference (self.msgcat)))

class ErrorHandler (xmlval.ErrorHandler):
    def __init__(self,locator):
        xmlval.ErrorHandler.__init__(self,locator)
        self.errors=0
        self.warnings=0

    def get_location(self):
        return "%s:%d:%d" % (self.locator.get_current_sysid(),self.locator.get_line(), self.locator.get_column())
        
    def warning(self,msg):
        sys.stderr.write ("WARNING ON %s: %s\n" % (self.get_location(),msg))
        self.warnings=self.warnings+1

    def error(self,msg):
        self.fatal(msg)
        
    def fatal (self,msg):
        sys.stderr.write ("%s: %s\n" % (self.get_location(), msg))
        self.errors=self.errors+1

class Application (xmlval.Application):
    def __init__ (self, arch):
        xmlval.Application.__init__ (self)

        # I may want to check this value.
        self.arch = arch

        self.charset = None
        self.langs = []

        self.states = []
        self.state = None

        self.tempo = []

        self.items = []

        self.keymaps = Storage ('keymaps')
        self.fonts = Storage ('fonts')
        self.acms = Storage ('acms')
        self.locales = Storage ('locales')
        self.msgcats = Storage ('msgcats')

        self.ignore = None

    # +++ wrappers +++

    def handle_start_tag (self, name, attrs):
        # sys.stderr.write ('STAG: %s (%s)\n' % (name, attrs))

        try:
            processor = getattr (self, 'start_' + name)
        except AttributeError:
            processor = None

        if processor:
            processor (attrs)
        else:
            raise 'There is no processor for start tag of element %s' % name

    def handle_end_tag (self, name):
        # sys.stderr.write ('ETAG: %s\n' % name)

        try:
            processor = getattr (self, 'end_' + name)
        except AttributeError:
            processor = None

        if processor:
            processor ()
        else:
            raise 'There is no processor for end tag of element %s' % name

    def handle_data (self, data, start, end):
        # sys.stderr.write ('DATA: %s\n' % data[start : end])

        self.do_text (data[start : end])

    # --- wrappers ---

    # +++ helpers +++

    def push (self):
        ''' pushes the value of self.what onto stack '''
        self.tempo.insert (0, self.what)

    def pop (self):
        ''' pops the value of self.what from the stack.  the previous value of self.what is returned '''
        value = self.what
        self.what = self.tempo[0]

        del self.tempo[0]

        return value

    def enter (self, state):
        self.states.insert (0, self.state)

        self.state = state

    def leave (self):
        self.state = self.states[0]

        del self.states[0]

    def check (self, state):
        assert self.state == state

    # +++ helpers +++

    def start_languages (self, attrs):
        pass

    def end_languages (self):
        pass

    def start_language (self, attrs):
        self.enter ('language')

        self.what = Language (self.fonts, self.acms, attrs)
        self.charset = self.what.charset

    def end_language (self):
        self.langs.append (self.what)

        self.leave ()

    def start_name (self, attrs):
        self.enter ('name')

    def end_name (self):
        self.leave ()

    def start_item (self, attrs):
        self.enter ('item')

        self.push ()

        self.what = Item (self.charset, self.fonts, self.acms, self.keymaps, self.locales, self.msgcats, attrs)

    def end_item (self):
        self.leave ()

        current = self.pop ()

        no = len (self.items)

        self.items.append (current)

        self.what.items.append (no)

    def start_hint (self, attrs):
        self.enter ('hint')

    def end_hint (self):
        self.leave ()

    def start_arch (self, attrs):
        # print 'arch:', attrs['name']

        self.ignore = self.arch not in string.split (attrs['name'])

        print 'arch:', self.ignore

    def end_arch (self):
        pass

    def start_list (self, attrs):
        self.push ()

        self.what = List ()

        self.enter ('list')

    def end_list (self):
        self.leave ()

        current = self.pop ()

        current.items.sort ()

        if self.state == 'language':
            self.what.list = current
        else:
            self.what.items.append (current)

    def do_text (self, text):
        if self.state in [ 'font', 'keymap', 'acm' ]:
            self.data = text
        elif self.state == 'name':
            self.what.name = self.what.name + text
        elif self.state == 'hint':
            self.what.hint = self.what.hint + text
        else:
            raise 'Do not know where to put the data in state %s!' % self.state

    def dump (self, output):
        output.write (self.fonts.declaration ())
        output.write ('\n')

        output.write (self.acms.declaration ())
        output.write ('\n')

        output.write (self.locales.declaration ())
        output.write ('\n')

        output.write (self.keymaps.declaration ())
        output.write ('\n')

        output.write (self.msgcats.declaration ())
        output.write ('\n')

        output.write ('\n')

        self.langs.sort ()

        langs = len (self.langs)

        output.write ('static struct language_item all_items[] = {\n')

        next = None

        for item in self.items:
            if next:
                output.write (',\n')
            else:
                next = 1

            output.write ('\t%s' % item.definition ())

        output.write ('\n};\n\n')

        output.write ('''\
static struct language_definition all_languages[] = {
''')
        for lang in self.langs:
            output.write ('\t%s,\n' % lang.definition ())

        output.write ('\t{ NULL }\n')

        output.write ('''};\n
''')

        for lang in self.langs:
            lang.list.definition (output, lang.charset, lang.definition_name ())

        output.write ('''\
const struct language_definition *
available_languages (void)
{\n
''')
        for i in range (len (self.langs)):
            lang = self.langs[i]
            name = lang.definition_name ()

            lang.initialization (output, 'all_languages[%s]' % i)

            output.write ('    all_languages[%s].list = &%s;\n' % (i, name))

            lang.list.initialization (output, name)

            output.write ('\n')

        for i in range (len (self.items)):
            item = self.items[i]

            item.initialization (output, 'all_items[%s]' % i)

        output.write ('''\
    return all_languages;
}
''')

argc = len (sys.argv)

if argc < 3:
    print 'usage: pl.py arch input [ output ]'
    sys.exit (1)

app = Application (sys.argv[1])
p = xmlval.XMLValidator ()
err = ErrorHandler (p)

p.set_error_handler (err)
p.set_application (app)

p.parse_resource (sys.argv[2])

if err.errors + err.warnings > 0:
    sys.stderr.write ('The file is not valid.  Aborting...\n')
    sys.exit (1)

if argc == 4:
    outname = sys.argv[3]
else:
    outname = os.path.splitext (sys.argv[2])[0] + '.c'

outfile = open (outname, 'w')

outfile.write ('''\
/*
 This is an automatically generated file.  DO NOT CHANGE!
%s
 Source file: %s
 Date: %s
 */

#include <stdlib.h>

#include "langs.h"

''' % (__copyright__, app.locator.get_current_sysid (), time.ctime (time.time ())))

app.dump (outfile)

outfile.close ()
