#!/bin/sh

# Create release directory and copy files over
# Made into a shell script to add flexibility

# See setup_image_names() in utilities/dbootstrap/main.c to get the
# idea of what sort of directory structure you want to end up with.
# The most important things to get right are the rescue disk and the
# drivers.tgz file, as these are what's used by network installs.

# Basically the rescue disk lives in
# [subarch]/images-size/[flavor]/rescue.bin ([] stuff is optional -
# note that images-size (i.e. images-1.44, images-1.20) is *not*
# optional, as it will default to 1.44 if not set in the bootloader
# configuration)

# In contrast, drivers.tgz and linux (the bare kernel) live in
# [subarch]/[flavor].  This really isn't the best layout but it works
# well enough in most cases.

. ./common.sh
set -e
#set -x

# parse command line params
if [ $# -lt 3 ]; then
  echo "release.sh <arch> <dist> <version>"
  exit 1
fi

arch="$1"
dist="$2"
version="$3"
shift;shift;shift
# arch specific settings
otherargs="$*"

# Set up the environment
MAKE=${MAKE:-make}
tools_dir=${TOOLS_DIR:-/archive/debian/tools}
TOP=$(pwd)
release=$TOP/release
pkgver="`dpkg-parsechangelog | awk -- '/^Version/ {print $2}'`"
suffix="_${pkgver}_${arch}"
bfinstall="bf-archive-install${suffix}.sh"

do_common_pretasks()
{
  rm -rf $release
  mkdir -p $release

  $MAKE release/README.txt
  $MAKE release/READ-pl.txt
  cp base-contents.txt $release/

  #---------------------------------------------------------------------
  # standard image handling
  #---------------------------------------------------------------------
  # strategy is to copy files to 'release' then pick off from there
  cp *.bin base*.tgz drivers*.tgz $release/

  # Unfortunately there's not a standard way of specifying
  # subarchitectures and flavors in the image file names... so we work
  # around this on a per-architecture basis below.  Yes this will be
  # fixed.

  cd $release

  mkdir images-2.88
  mkdir images-1.44
  mkdir images-1.20

  for f in resc2880*.bin; do
    [ ! -f $f ] || mv $f images-2.88/rescue${f#resc2880}
  done

  for f in resc1440*.bin; do
    [ ! -f $f ] || mv $f images-1.44/rescue${f#resc1440}
  done
  for f in root1440*.bin; do
    [ ! -f $f ] || mv $f images-1.44/root${f#root1440}
  done
  for f in drv14*.bin; do	
    [ ! -f $f ] || mv $f images-1.44/driver${f#drv14}
  done
  for f in base14-*.bin; do
    mv $f images-1.44/base${f#base14}
  done

  for f in resc1200*.bin; do
    [ ! -f $f ] || mv $f images-1.20/rescue${f#resc1200}
  done
  for f in root1200*.bin; do
    [ ! -f $f ] || mv $f images-1.20/root${f#root1200}
  done
  for f in drv12*.bin; do
    [ ! -f $f ] || mv $f images-1.20/driver${f#drv12}
  done
  for f in base12-*.bin; do	\
    [ ! -f $f ] || mv $f images-1.20/base${f#base12}
  done

  rmdir images-2.88 2>/dev/null || true	# in case empty
  rmdir images-1.44 2>/dev/null || true	# in case empty
  rmdir images-1.20 2>/dev/null || true	# in case empty

  cd $TOP

  # make the stub for the archive installer script
  sed "s/%arch%/$arch/g; s/%dist%/$dist/g; s/%version%/$version/g;" \
	< debian/bf-archive-install.sh > $release/$bfinstall
  chmod a+x $release/$bfinstall

  #---------------------------------------------------------------------
  # docs and doc tar-ball
  #---------------------------------------------------------------------
  mkdir $release/doc
  make -C documentation DESTDIR=$release/doc install

  tar cz --remove-files -f $release/bf-doc${suffix}.tar.gz -C $release doc
  echo "tar -xzf bf-doc${suffix}.tar.gz ; rm -f bf-doc${suffix}.tar.gz" >>	\
	$release/$bfinstall
  rm -rf $release/doc
}


do_archspecific()
{
  # Architecture specific stuff
  case $arch in
  i386)
  	do_arch_i386 $otherargs
	;;
  sparc)
  	do_arch_sparc $otherargs
	;;
  powerpc)
  	do_arch_powerpc $otherargs
	;;
  arm)
  	do_arch_arm $otherargs
	;;
  m68k)
  	do_arch_m68k $otherargs
	;;
  alpha)
  	do_arch_alpha $otherargs
	;;
  *)
  	echo "Unsupported arch $arch"
	exit 1
  esac
}

do_common_posttasks()
{
  #--------------------------------------------------------
  # standard post stuff
  #--------------------------------------------------------
  # this script is self-destructing
  echo "find . -type d -exec chmod 775 \{\} \;" >> $release/$bfinstall
  echo "find . -type f -exec chmod 664 \{\} \;" >> $release/$bfinstall
  echo "rm -f $bfinstall" >> $release/$bfinstall
}


do_arch_i386()
{
  #--------------------------------------------------------
  # i386 installation
  #--------------------------------------------------------
  mv $release/base-contents.txt $release/basecont.txt

  # Language support for rescue disk
  # text files and fonts for syslinux and a batch file
  archive="$1"
  shift
  langs=`cd scripts/rescue/messages; ls -d ??`
  langfiles="readme.txt debian.txt f1.txt f2.txt f3.txt f4.txt f5.txt f6.txt f7.txt f8.txt f9.txt f10.txt"
  debug "Using versions from $archive"
  KERNEL_VERSION=`basename $archive | sed -e 's/kernel-image-//' -e 's/_.*//'`
  DEBIAN_KERNEL_IMAGE=`basename $archive .deb | sed -e s/_$i386\$//`

  mkdir -p $release/dosutils
  mkdir -p $release/lang
  mkdir -p $release/lang/fonts

  sed "s/LANGUAGES/`echo $langs`/" < scripts/dos/setlang.bat > $release/dosutils/setlang.bat
  cp scripts/rootdisk/consolefonts/*.psf $release/lang/fonts/
  for lang in ${langs}
  do
    debug "Processing files for language ${lang}"
    mkdir $release/lang/${lang}
    for file in ${langfiles}
    do
      if [ -f scripts/rescue/messages/${lang}/${file} ]; then
        debug "m4 processing ${file}"
        m4 -EP -D__kernel_version__=$KERNEL_VERSION \
              -D__kernel_image__=$DEBIAN_KERNEL_IMAGE \
              -D__debianversion__=$version \
              -D__arch__=i386 scripts/rescue/messages/${lang}/${file} \
		| sed s/$/$'\r'/ > $release/lang/${lang}/${file}
        testexit
      fi
    done
  done

  # DOS tools provided for i386 only
  recode latin1..ibmpc <scripts/dos/install.bat >$release/install.bat

  cd $release/dosutils
  miniunzip $tools_dir/rawrite2.zip rawrite2.exe 
  miniunzip $tools_dir/rawrite2.zip rawrite2.txt
  miniunzip -p $tools_dir/lodlin16.zip LODLIN16/LOADLIN.EXE >loadlin.exe
  cp -f LODLIN16/LOADLIN.EXE loadlin.exe
  rm -Rf LODLIN16
  cd $TOP

  cp linux $release/

  gunzip -c $TOP/config.gz > $release/kernel-config
  gunzip -c $TOP/config.gz > $release/images-1.44/kernel-config
  cp $release/images-1.44/kernel-config $release/images-2.88/kernel-config

  # safe "flavor"
  for size in 1.20 1.44; do
    safedir=$release/images-$size/safe
    mkdir -p $safedir
    (cd $safedir && 
	ln -s ../driver-* . &&
	mv ../rescue-s.bin rescue.bin)
    cp $TOP/documentation/README-safe $safedir/README.txt
  done

  for _flavor in compact udma66 idepci; do
    mkdir $release/$_flavor
    cp linux$_flavor $release/$_flavor/linux
    recode latin1..ibmpc <scripts/dos/install-$_flavor.bat >$release/$_flavor/install.bat
    mv $release/drivers$_flavor.tgz $release/$_flavor/drivers.tgz
    gunzip -c $TOP/config$_flavor.gz > $release/$_flavor/kernel-config  

    for size in 1.44 2.88; do
      fromdir=$release/images-$size
      destdir=$release/images-$size/$_flavor
  
      mkdir -p $destdir
      mv $fromdir/rescue$_flavor.bin $destdir/rescue.bin
      cp $TOP/documentation/README-$_flavor $destdir/README.txt
      cp $TOP/documentation/README-$_flavor $release/$_flavor/README.txt
      gunzip -c $TOP/config$_flavor.gz > $destdir/kernel-config
      
      if [ -f $fromdir/root$_flavor.bin ]; then
  	mv $fromdir/root$_flavor.bin $destdir/root.bin
      fi
      for i in $fromdir/driver$_flavor-?.bin; do
        if [ -f $i ]; then
          base=$(basename $i | sed -e s,$_flavor,,)
          mv $i $destdir/$base
        fi
      done
    done
  
    rm -f release/images-1.20/driver$_flavor-{1,2,3,4}.bin   # what's this file doing?
  
    rm -f $release/root$_flavor.bin
  done

  rm -f $release/root.bin

  # Set permissions
  chmod -R a+r $release
  
  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f | 			\
    grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  ### make tarballs, and the script steps to undo them
  cd $release
  tar cz --remove-files -f bf-images-1.20${suffix}.tar.gz images-1.20
  echo "tar -xzf bf-images-1.20${suffix}.tar.gz ; rm -f bf-images-1.20${suffix}.tar.gz" >> \
  	$bfinstall
  rm -rf images-1.20

  tar cz --remove-files -f bf-images-1.44${suffix}.tar.gz images-1.44
  echo "tar -xzf bf-images-1.44${suffix}.tar.gz ; rm -f bf-images-1.44${suffix}.tar.gz" >> \
  	$bfinstall
  rm -rf images-1.44

  tar cz --remove-files -f bf-common${suffix}.tar.gz *.tgz images-2.88 compact udma66 idepci dosutils kernel-config
  echo "tar -xzf bf-common${suffix}.tar.gz ; rm -f bf-common${suffix}.tar.gz" >> \
  	$bfinstall
  rmdir images-2.88/compact
  rmdir images-2.88/udma66
  rmdir images-2.88/idepci
  rmdir images-2.88
  rmdir compact
  rmdir udma66
  rmdir idepci
  rmdir dosutils

  tar cz --remove-files -f bf-misc${suffix}.tar.gz \
  	README.txt install.bat linux md5sum.txt basecont.txt \
        lang \
	READ-??.txt
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
  	$bfinstall
  rm -rf lang

  cd $TOP
}

do_arch_sparc()
{
  mv $release/root.bin $release/images-1.44

  for subarch in sun4u sun4cdm sun4dm-pci; do
    # Per subarch images
    mkdir -p $release/${subarch}/images-1.44/
    mv $release/images-1.44/rescue-${subarch}.bin $release/${subarch}/images-1.44/rescue.bin
    (cd $release/images-1.44 &&
      for f in driver-${subarch}*.bin; do
	[ ! -f $f ] || mv $f ../${subarch}/images-1.44/driver${f#driver-${subarch}};
      done)
    mv $release/drivers-${subarch}.tgz $release/${subarch}/drivers.tgz
    # the linux kernel is mainly dedicated to net boot: convert it to
    # a.out (required by the SUN TFTP boot process) and pad the result to
    # be 4-bytes aligned (required by some old bootproms).
    if [ -f linux-${subarch} ]; then
      gzip -dcf linux-${subarch} > $release/${subarch}/linux
      elftoaout -o $release/${subarch}/linux-a.out $release/${subarch}/linux ||
        mv $release/${subarch}/linux $release/${subarch}/linux-a.out
      rm -f $release/${subarch}/linux
      size=`ls -l release/${subarch}/linux-a.out | awk '{print $5}'`
      if rem=`expr \( 4 - $size % 4 \) % 4`; then
        dd if=/dev/zero bs=1 count=$rem >> $release/${subarch}/linux-a.out
      fi
    fi
    # Install the per subarch tftpboot images
    cp tftpboot-${subarch}.img $release/${subarch}/tftpboot.img
  done

  # install netserver miniroot
  cp root.tar.gz $release/

  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f | 			\
	grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  # make tarballs for subarches, etc.
  for dir in images-1.44 sun4cdm sun4u sun4dm-pci; do
    (cd $release && tar cz --remove-files -f bf-${dir}${suffix}.tar.gz $dir)
    echo "tar -xzf bf-${dir}${suffix}.tar.gz ; rm -f bf-${dir}${suffix}.tar.gz" >> \
	$release/$bfinstall
    rm -rf $release/${dir}
  done

  (cd $release && tar cz --remove-files -f bf-common${suffix}.tar.gz \
	*.txt *.tgz root.tar.gz)
  echo "tar -xzf bf-common${suffix}.tar.gz ; rm -f bf-common${suffix}.tar.gz" >> \
	$release/$bfinstall

}

do_arch_m68k()
{
  #--------------------------------------------------------
  # m68k installation
  #--------------------------------------------------------
  # m68k particulars; we support 4 m68k platforms so far, all different. If 
  # you ate lunch just recently, simply skip this section.
  #
  archive=$1
  rootfiles="README.txt READ-pl.txt base-contents.txt md5sum.txt base*.tgz"

  for subarch in amiga atari bvme6000 mvme16x mvme147 mac; do
    for size in 1.44 2.88; do
      mkdir -p $release/$subarch/images-$size/
      [ ! -f $release/images-$size/rescue$subarch.bin ] || \
          mv $release/images-$size/rescue$subarch.bin \
               $release/$subarch/images-$size/rescue.bin
      [ ! -f $release/images-$size/root$subarch.bin   ] || \
          mv $release/images-$size/root$subarch.bin \
               $release/$subarch/images-$size/root.bin
      if [ -d $release/images-$size ]; then
        cd $release/images-$size
        for f in driver$subarch*.bin; do
          [ ! -f $f ] || mv $f ../$subarch/images-$size/driver${f#driver$subarch};
        done
        cd ../$subarch/images-$size
        if [ -f driver-1.bin -a ! -f driver-2.bin ]; then
          mv driver-1.bin driver.bin;
        fi
        cd $TOP
      fi
      rmdir $release/$subarch/images-$size 2>/dev/null || true	# in case empty
      rmdir $release/images-$size          2>/dev/null || true	# in case empty
    done
    [ ! -f $release/drivers$subarch.tgz ] || \
	    mv $release/drivers$subarch.tgz $release/$subarch/drivers.tgz
    [ ! -f $release/root$subarch.bin    ] || \
        mv $release/root$subarch.bin    $release/$subarch/root.bin
    [ ! -f $TOP/linux$subarch      ] || \
        cp $TOP/linux$subarch      $release/$subarch/linux
    [ ! -f $TOP/sys_map$subarch.gz ] || \
        cp $TOP/sys_map$subarch.gz $release/$subarch/sysmap.gz
    rmdir $release/$subarch  2>/dev/null || true	# in case empty
  done

  #--------------------------------------------------------
  # m68k/atari 
  #--------------------------------------------------------
  if [ -d $release/atari ]; then
    #
    # Atari install kit: lzh archive is no more, we use tgz now
    #
    cp m68k-specials/bootstra.ttp             $release/atari/bootstra.prg
    echo "-s -k linux -r root.bin root=/dev/ram video=keep load_ramdisk=1" \
                                            > $release/atari/bootargs
    cp m68k-specials/debian-atari.txt         $release/atari/install.doc

    (cd $release && tar czf $release/atariinstall.tgz atari)

    # 
    # Atari: loose files
    #
    cp m68k-specials/rawwrite.ttp $release
    rootfiles="$rootfiles rawwrite.ttp atariinstall.tgz"
  fi

  #--------------------------------------------------------
  # m68k/amiga
  #--------------------------------------------------------
  if [ -d $release/amiga ]; then
    #
    # Amiga install kit: lzh archive is no more, we use tgz now
    #
    # unpack AmigaOS magic from m68k support directory
    (cd $release && tar xvfz $TOP/m68k-specials/amiga-kit.tgz)
    cp m68k-specials/debian-amiga.txt         $release/amiga/install.doc
    rootfiles="$rootfiles amiga.info"

    (cd $release && tar czf $release/amigainstall.tgz amiga amiga.info)

    #
    # Amiga: loose files
    #
    cp m68k-specials/dmesg		$release
    chmod a+x $release/dmesg
    cp m68k-specials/dmesg.readme	$release
    rootfiles="$rootfiles dmesg dmesg.readme amigainstall.tgz"
  fi

  #--------------------------------------------------------
  # m68k/mac
  #--------------------------------------------------------
  if [ -d $release/mac ]; then
    # Mac install kit: HFS floppy image, to be packaged as StuffIt archive
    #

    # create HFS floppy image
    dd if=/dev/zero of=$release/mac/macinstall-floppy-hfs.img bs=1024 count=2880
    hformat -l "Debian/68k" $release/mac/macinstall-floppy-hfs.img

    # mount it (':' is top dir)
    hmount $release/mac/macinstall-floppy-hfs.img

    # copy kernel/sysmap/ramdisk as raw
    hcopy -r linuxmac :linux
    hcopy -r sys_mapmac.gz :System-Map.gz
    hcopy -r rootmac.bin :root.bin

    # MacOS magic; useful for HFS CD I hope
    #
    # hybrid CD note:
    # mount this HFS floppy image using the mount command
    # mount -t hfs -o loop,fork=netatalk,afpd
    # copy over both Penguin-15 and .AppleDouble/Penguin-15
    # copy over both Penguin Prefs and .AppleDouble/Penguin Prefs
    # (.AppleDouble files go into .AppleDouble subdir of target dir)
    #
    # make CD image using the command
    # mkhybrid -map <mapfile> --netatalk -a -j -hfs -r -V <label> -o <image> <source dir>
    #
    # map file contains stuff like
    # # ext. xlate  creator  type    comment
    # .hqx   Ascii  'BnHx'   'TEXT'  "BinHex file"
    # .mov   Raw    'TVOD'   'MooV'  "QuickTime Movie"
    # .deb   Raw    'Debn'   'bina'  "Debian package"
    # .bin   Raw    'Debn'   'bina'  "Floppy or ramdisk image"
    # *      Ascii  'ttxt'   'TEXT'  "Text file"
    #
    # Note that we can't use MacBinary mapping for .bin files due to
    # name clashes with the binary image files, so we have to provide 
    # all encoded Mac files in BinHex format.
    #
    # Kudos Brad Midgley, <brad@pht.com> for the tricks!
    #
    # copy booter as BinHex
    # hcopy -b m68k-specials/Penguin-17.hqx :
    hcopy -b m68k-specials/Penguin-18.hqx :
    hcopy -b m68k-specials/Penguin-Colors.hqx :
    hcopy -b m68k-specials/Penguin.doc.hqx :

    # copy Preferences as BinHex
    # hcopy -b m68k-specials/Penguin_Prefs_floppy.hqx :"Penguin Prefs"

    # copy install doc as text
    hcopy -t m68k-specials/debian-mac.txt :Install.doc
    humount $release/mac/macinstall-floppy-hfs.img

# FIXME:
# why do we do all this hfs magic, if we do not need the image in the end?
# will the image be needed for an improved debian-cd?
    rm $release/mac/macinstall-floppy-hfs.img

    # package Mac booter stuff for the CD install dir
    # (tar format until I can come up with something nicer)
#    mkdir -p $release/mac/install/
    cp m68k-specials/Penguin-17.hqx $release/mac/
    cp m68k-specials/Penguin-18.hqx $release/mac/
    cp m68k-specials/Penguin-Colors.hqx $release/mac/
    cp m68k-specials/Penguin.doc.hqx $release/mac/
#    mkdir -p $release/mac/install/mac/
    # cp m68k-specials/Penguin-17.hqx $release/mac/install/mac/
    # cp m68k-specials/Penguin_Prefs_CD.hqx $release/mac/install/mac/
    # cp m68k-specials/Penguin_Prefs_CD_autoboot.hqx $release/mac/install/mac/

#    dd if=/dev/zero of=$release/macinstall-temp.img bs=1024 count=1440
#    hformat $release/macinstall-temp.img
#
#    # mount it (':' is top dir)
#    hmount $release/macinstall-temp.img
#    hcopy -b m68k-specials/Penguin_Prefs_CD.hqx :"Penguin Prefs"
#    hcopy -b m68k-specials/Penguin_Prefs_CD_autoboot.hqx :"Penguin Prefs (autoboot)"
#    hcopy -b :"Penguin Prefs" $release/mac/install/mac/
#    hcopy -b :"Penguin Prefs (autoboot)" $release/mac/install/mac/
#    humount $release/macinstall-temp.img
#    rm $release/macinstall-temp.img

    cp m68k-specials/debian-mac.txt $release/mac/

    (cd $release && tar czf $release/macinstall.tgz mac)

    #
    # Mac: loose files
    #
    rootfiles="$rootfiles macinstall.tgz"
  fi

  #--------------------------------------------------------
  # m68k/vme bvme6000, mvme16x (no tftp support for mvme147)
  #--------------------------------------------------------
  # VME stuff
  #
  if [ -d $release/bvme6000 -o -d $release/mvme16x ]; then
     rm -rf /var/tmp/extract-tftplilo
     mkdir /var/tmp/extract-tftplilo
     dpkg-deb --extract $archive/admin/m68k-vme-tftplilo_*.deb /var/tmp/extract-tftplilo
  fi

  #--------------------------------------------------------
  # m68k/vme: bvme6000
  #--------------------------------------------------------
  if [ -d $release/bvme6000 ]; then
    cp /var/tmp/extract-tftplilo/boot/tftplilo.bvme $release/bvme6000
    zcat /var/tmp/extract-tftplilo/usr/share/doc/m68k-vme-tftplilo/README.gz \
                                                >$release/bvme6000/tftplilo.txt
    cp m68k-specials/tftplilo.conf               $release/bvme6000/tftplilo.conf
    cp m68k-specials/bvmbug-G.bin                $release/bvme6000/bvmbug-G.bin
    cp m68k-specials/bvmbug.txt                  $release/bvme6000/bvmbug.txt
    cp m68k-specials/debian-vme.txt              $release/bvme6000/install.txt
    mv $release/bvme6000/linux                   $release/bvme6000/linuxbvme6000
    mv $release/bvme6000/root.bin                $release/bvme6000/rootbvme6000.bin
  fi

  #--------------------------------------------------------
  # m68k/vme: mvme16x
  #--------------------------------------------------------
  if [ -d $release/mvme16x ]; then
    cp /var/tmp/extract-tftplilo/boot/tftplilo.mvme $release/mvme16x
    zcat /var/tmp/extract-tftplilo/usr/share/doc/m68k-vme-tftplilo/README.gz \
                                                >$release/mvme16x/tftplilo.txt
    cp m68k-specials/tftplilo.conf               $release/mvme16x/tftplilo.conf
    cp m68k-specials/debian-vme.txt              $release/mvme16x/install.txt
    mv $release/mvme16x/linux                    $release/mvme16x/linuxmvme16x
    mv $release/mvme16x/root.bin                 $release/mvme16x/rootmvme16x.bin
  fi

  #--------------------------------------------------------
  # m68k/vme: mvme147
  #--------------------------------------------------------
  if [ -d $release/mvme147 ]; then
    cp m68k-specials/debian-vme.txt              $release/mvme147/install.txt
    mv $release/mvme147/linux                    $release/mvme147/linuxmvme147
    mv $release/mvme147/root.bin                 $release/mvme147/rootmvme147.bin
  fi

  if [ -d $release/bvme6000 -o -d $release/mvme16x -o -d $release/mvme147 ];
  then
    #
    # VME: loose files
    #
    cp m68k-specials/rawrite2.exe               $release/
    rootfiles="$rootfiles rawrite2.exe"

    # clean up
    rm -rf /var/tmp/extract-tftplilo
  fi

  #--------------------------------------------------------
  # m68k magic
  #--------------------------------------------------------
  #
  # Copy the m68k magic used for this build to $release/
  #
  mkdir $release/source
  ( cd m68k-specials
    tar czf $release/source/m68k-support.tar.gz * --exclude CVS
  )

  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f | 			\
	grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  # make tarball of root files
  ( cd $release
    tar cz --remove-files -f bf-root${suffix}.tar.gz $rootfiles
    echo "tar -xzf bf-root${suffix}.tar.gz ; rm -f bf-root${suffix}.tar.gz" >> \
        $release/$bfinstall
  )

  # make tarballs for subarches, etc.
  for dir in images-1.44 source \
             amiga atari bvme6000 mvme16x mvme147 mac; do
	if [ -d $release/$dir ]; then
      (cd $release && tar cz --remove-files -f bf-${dir}${suffix}.tar.gz $dir)
      echo "tar -xzf bf-${dir}${suffix}.tar.gz ; rm -f bf-${dir}${suffix}.tar.gz" >> \
        $release/$bfinstall
      rm -rf $release/${dir}
    fi
  done
}

do_arch_powerpc()
{
  #--------------------------------------------------------
  # PowerPC install documentation
  #--------------------------------------------------------
  $MAKE -C powerpc-specials clean
  $MAKE -C powerpc-specials

  #--------------------------------------------------------
  # CHRP install kit
  #--------------------------------------------------------
  mkdir -p $release/chrp/images-1.44
  [ ! -f linuxchrp ] || cp linuxchrp $release/chrp/linux
  [ ! -f sys_mapchrp.gz ] || cp sys_mapchrp.gz $release/chrp/sysmap.gz
  mv $release/images-1.44/rescuechrp.bin $release/chrp/images-1.44/rescue.bin
  mv $release/images-1.44/rootchrp.bin $release/chrp/images-1.44/root.bin
  mv $release/driverschrp.tgz $release/chrp/drivers.tgz
  for dfile in $release/images-1.44/driverchrp-?.bin; do 
    bdfile=`basename $dfile`
    dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
    mv $release/images-1.44/$bdfile $release/chrp/images-1.44/driver-$dnum.bin
  done
  cp powerpc-specials/bootargs-chrp $release/chrp/bootargs
  cp powerpc-specials/install-chrp.en.txt $release/chrp/install.txt

  #--------------------------------------------------------
  # PMac install kit: HFS floppy image
  #--------------------------------------------------------
  mkdir -p $release/powermac/images-1.44
  if [ -f linuxpmac.gz ]; then zcat linuxpmac.gz > $release/powermac/linux; fi
  if [ -f sys_mappmac.gz ]; then cp sys_mappmac.gz $release/powermac/sysmap.gz ; fi

  sh powerpc-specials/miBoot/mkboot.sh $release/powermac/images-1.44/boot-floppy-hfs.img

  mv $release/images-1.44/rescuepmac.bin $release/powermac/images-1.44/rescue.bin
  mv $release/images-1.44/rootpmac.bin $release/powermac/images-1.44/root.bin
  mv $release/driverspmac.tgz $release/powermac/drivers.tgz
  for dfile in $release/images-1.44/driverpmac-?.bin; do 
    bdfile=`basename $dfile`
    dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
    mv $release/images-1.44/$bdfile $release/powermac/images-1.44/driver-$dnum.bin
  done
  #	$MAKE -C powerpc-specials/coffboot ../../rescue-pmac.coff
  #	cp rescue-pmac.coff $release/powermac/rescue-pmac.coff
  cp $release/powermac/images-1.44/root.bin $release/powermac/ramdisk.image.gz
  cp powerpc-specials/BootX*.sit $release/powermac/
  cp powerpc-specials/bootvars*sit.hqx $release/powermac/
  cp powerpc-specials/bootargs-pmac $release/powermac/bootargs
  cp powerpc-specials/install-pmac.en.txt $release/powermac/install.txt

  #--------------------------------------------------------
  # PReP install kit
  #--------------------------------------------------------
  mkdir -p $release/prep/images-1.44
  if [ -f linuxprep ]; then cp linuxprep $release/prep/linux ; fi
  if [ -f sys_mapprep.gz ]; then cp sys_mapprep.gz $release/prep/sysmap.gz ; fi
  mv $release/bootprepfull.bin $release/prep/bootfull.bin
  mv $release/bootprep.bin $release/prep/images-1.44/boot.bin
  mv $release/images-1.44/rescueprep.bin $release/prep/images-1.44/rescue.bin
  mv $release/images-1.44/rootprep.bin $release/prep/images-1.44/root.bin
  mv $release/driversprep.tgz $release/prep/drivers.tgz
  for dfile in $release/images-1.44/driverprep-?.bin; do 
    bdfile=`basename $dfile`; \
    dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
    mv $release/images-1.44/$bdfile $release/prep/images-1.44/driver-$dnum.bin
  done 
  cp powerpc-specials/install-prep.en.txt $release/prep/install.txt

  #--------------------------------------------------------
  # APUS stuff
  #--------------------------------------------------------
  mkdir -p $release/apus/images-1.44
  tar xzvf powerpc-specials/bootstrap-apus.tgz -C $release/apus
  if [ -f linuxapus ]; then cp linuxapus $release/apus/linux ; fi
  if [ -f sys_mapapus.gz ]; then cp sys_mapapus.gz $release/apus/sysmap.gz ; fi
  mv $release/images-1.44/rescueapus.bin $release/apus/images-1.44/rescue.bin
  mv $release/images-1.44/rootapus.bin $release/apus/images-1.44/root.bin
  mv $release/driversapus.tgz $release/apus/drivers.tgz
  for dfile in $release/images-1.44/driverapus-?.bin; do 
    bdfile=`basename $dfile`; \
    dnum=`echo $bdfile | cut -d - -f 2 | cut -c 1`
    mv $release/images-1.44/$bdfile $release/apus/images-1.44/driver-$dnum.bin
  done 
  cp powerpc-specials/install-apus.en.txt $release/apus/install.txt    

  #--------------------------------------------------------
  rm -f $release/root*.bin
  # make md5sum before we tar up the contents
  (cd $release && md5sum `find . -type f | 		\
	grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)
  #--------------------------------------------------------
  # Multitudinous inner tarballs
  #--------------------------------------------------------
  (cd $release && tar cz --remove-files -f bf-arch${suffix}.tar.gz chrp powermac prep apus)
  echo "tar -xzf bf-arch${suffix}.tar.gz ; rm -f bf-arch${suffix}.tar.gz" >> \
  	$release/$bfinstall
  rm -rf $release/{chrp,powermac,prep,apus}

  (cd $release && tar cz --remove-files -f bf-common${suffix}.tar.gz images-1.44 *.tgz)
  echo "tar -xzf bf-common${suffix}.tar.gz ; rm -f bf-common${suffix}.tar.gz" >> \
  $release/$bfinstall
  (cd $release && rm -rf images-1.44)

  (cd $release && tar cz --remove-files -f bf-misc${suffix}.tar.gz \
	README.txt READ-??.txt md5sum.txt base-contents.txt)
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
  $release/$bfinstall
}

do_arch_alpha()
{

# Alpha directory structure should look like:
#
# images-1.44/root.bin (generic)
# images-1.44/rescue.bin (generic)
# images-1.44/drivers.tgz (generic)
# images-1.44/driver-?.bin (generic)
# ${subarch}/
#            drivers.tgz
#            linux
#            images-1.44/
#	                 rescue.bin
#                        driver-?.bin
#                        milo.bin
# MILO/
#      linload.exe
#      ldmilo.exe
#      (milo various stuff)
# APB/
#     (apb various stuff - might be moved around a bit)
# md5sum.txt
# install.txt
# bootlx (needed for CDs)

  mv $release/root.bin $release/images-1.44
  mkdir -p $release/MILO
  cp linload.exe ldmilo.exe $release/MILO

  mkdir -p $release/APB
  if [ -L apb.exe ]; then
      cp apb.exe up1000.pal $release/APB/
      for i in apb_*; do cp $i $release/APB/${i#apb_}; done
      cat >$release/APB/apb.cfg <<EOF
debian_install|bootl up1000.pal linux load_ramdisk=1
EOF
  fi

  for subarch in $*
  do
      if [ "$subarch" = "generic" ]; then
	  targetdir=$release
      else
	  targetdir=$release/$subarch
      fi
      # All subarchs have rescue disks
      if [ -f $release/images-1.44/rescue_${subarch}.bin ]; then
	  if [ "$subarch" != "generic" ]; then
	    subarch_dirs="$subarch_dirs $subarch"
	  fi
	  mkdir -p $targetdir/images-1.44
	  mv $release/images-1.44/rescue_${subarch}.bin \
	      $targetdir/images-1.44/rescue.bin
	  mv $release/driversalpha_${subarch}.tgz \
	      $targetdir/drivers.tgz
	  cp linuxalpha_${subarch} $targetdir/linux
	  for drvdisc in $release/images-1.44/driveralpha_${subarch}-?.bin
	  do
	      drv_number=`echo $drvdisc|sed "s/.*driveralpha_${subarch}-\([1-9]\).bin/\1/"`
	      mv $drvdisc $targetdir/images-1.44/driver-${drv_number}.bin
	  done
      fi
      # Some have MILO disks (but we don't necessarily build these)
      if [ -f $release/milo_${subarch}.bin ]; then
	  subarch_dirs="$subarch_dirs $subarch"
	  mkdir -p $targetdir/images-1.44
	  mv $release/milo_${subarch}.bin $targetdir/images-1.44/milo.bin
      fi
      # Also MILO executables
      if [ -L milo_${subarch} ]; then
	  cp milo_${subarch} $release/MILO/${subarch}
      fi
  done

  # install netserver miniroot
  cp root.tar.gz $release/
  [ ! -f root-graphical.tar.gz ] || cp root-graphical.tar.gz $release/

  # install tftpboot image
  cp tftpboot.img $release/
  [ ! -f tftpboot-graphical.img ] || cp tftpboot-graphical.img $release/

  # install bootlx for CDs (yes, you must have aboot 0.7 installed)
  cp /boot/bootlx $release/

  # make md5sum before tarring up
  (cd $release && md5sum `find . -type f | 		\
      grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  # Now, a plethora of inner tarballs
  (cd $release && tar cz --remove-files -f bf-generic${suffix}.tar.gz \
	images-1.44 linux tftpboot.img root.tar.gz *.tgz)
  echo "tar -xzf bf-generic${suffix}.tar.gz ; rm -f bf-generic${suffix}.tar.gz" >> \
      $release/$bfinstall
  rmdir $release/images-1.44

  (cd $release && tar cz --remove-files -f bf-others${suffix}.tar.gz $subarch_dirs)
  echo "tar -xzf bf-others${suffix}.tar.gz ; rm -f bf-others${suffix}.tar.gz" >> \
      $release/$bfinstall
  for subarch in $subarch_dirs; do rm -rf $release/$subarch; done

  (cd $release && tar cz --remove-files -f bf-misc${suffix}.tar.gz \
      README.txt READ-??.txt md5sum.txt base-contents.txt MILO APB)
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
      $release/$bfinstall
  (cd $release && rm -rf MILO APB)
}

do_arch_arm()
{
  # install subarch specifics
  for subarch in riscpc netwinder; do
    mkdir -p $release/$subarch/images-1.44
    mv $release/images-1.44/rescue-$subarch.bin $release/$subarch/images-1.44/rescue.bin
    mv $release/root-$subarch.bin $release/$subarch/images-1.44/root.bin
    mv $release/drivers-$subarch.tgz $release/$subarch/drivers.tgz
    for nr in 1 2 3; do
      if [ -f $release/images-1.44/driver-$subarch-$nr.bin ]; then mv $release/images-1.44/driver-$subarch-$nr.bin $release/$subarch/images-1.44/driver-$nr.bin; fi
    done
  done

  # install CATS tftp image
  mkdir -p $release/cats
  cp tftpboot-cats.img $release/cats/tftpboot.img

  # install NetWinder tftp image
  cp tftpboot-netwinder.img $release/netwinder/tftpboot.img

  # install NetWinder nfs root
  cp root-netwinder.tar.gz $release/netwinder/root.tar.gz

  # install NetWinder raw kernel
  cp linux-netwinder $release/netwinder/linux

  # install RiscPC raw kernel
  cp linux-riscpc $release/riscpc/linux

  # make md5sum before tarring up all the contents
  (cd $release && md5sum `find . -type f | 			\
    grep -v '\(md5sum.txt\|/doc/\|bf-\)'` > md5sum.txt)

  ### make tarballs, and the script steps to undo them
  ( cd $release
  for dir in images-1.44 cats netwinder riscpc ; do
  tar cz --remove-files -f bf-${dir}${suffix}.tar.gz ${dir}
  echo "tar -xzf bf-${dir}${suffix}.tar.gz ; rm -f bf-${dir}${suffix}.tar.gz" >> \
  	$bfinstall
  rm -rf ${dir}
  done

  tar cz --remove-files -f bf-misc${suffix}.tar.gz \
  	README.txt md5sum.txt base-contents.txt
  echo "tar -xzf bf-misc${suffix}.tar.gz ; rm -f bf-misc${suffix}.tar.gz" >> \
  	$bfinstall
  )
}
	
# main
do_common_pretasks
do_archspecific
do_common_posttasks
exit 0
