/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl.functions;
import com.kvisco.xsl.*;
import com.kvisco.util.List;
import org.w3c.dom.Node;

/**
 * A implementation of the "substring()" function call
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class Substring extends FunctionCall {
    
    /**
     * Creates a new Substring FunctionCall
    **/
    public Substring() {
        super(Names.SUBSTRING_FN);
    } //-- Substring
    
    /**
     * Evalutes this FunctionCall using the given Node as
     * the context. This method will always return a StringResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        List params = getParameterList();
        
        int nparams = params.size();
        if ((nparams < 2) || (nparams > 3))
            throw new InvalidExprException(INVALID_NUMBER_PARAMS+this);
            
        ExprResult exprResult = ((Expr)params.get(0)).evaluate(context, ps);
        
        String str = StringExpr.toStringResult(exprResult).toString();
        
        //-- calculate start index
        int startIdx = 1;
        exprResult = ((Expr)params.get(1)).evaluate(context, ps);
        double dbl = exprResult.numberValue();
        
        if ( Double.isNaN(dbl) || Double.isInfinite(dbl) )
            return new StringResult("");
        
        startIdx = (int)Math.round(dbl);
        
        //-- calculate end index
        int length   = str.length();
        if (nparams == 3) {
            exprResult = ((Expr)params.get(2)).evaluate(context, ps);
            dbl = exprResult.numberValue();
            
            if (Double.isNaN(dbl)) return new StringResult("");
            if (Double.isInfinite(dbl)) return new StringResult(str);
            
            length = (int)Math.round(dbl);
        }
        
        //-- adjust startIdx and length 
        //-- since XSL String indexing starts at 1, but
        //-- java String indexing starts at 0
        int endIdx = startIdx+length-1;
        if (startIdx > 0) --startIdx;
        else startIdx = 0;
        
        return new StringResult(str.substring(startIdx,endIdx));
    } //-- evaluate
    
} //-- Substring

