/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import org.w3c.dom.Node;

/**
 * Represents a TreeFragment result
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class TreeFragmentResult implements ExprResult {

    private Node value = null;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new TreeFragmentResult with the default value
    **/
    public TreeFragmentResult() {
        super();        
    } //-- TreeFragmentResult
    
    /**
     * Creates a new TreeFragmentResult with the given value
     * @param value the Node value of this ExprResult
    **/
    public TreeFragmentResult(Node value) {
        this.value = value;
    } //-- TreeFragmentResult
    
      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Returns the value of this ExprResult as a boolean
     * @return the value of this ExprResult as a boolean
    **/
    public boolean booleanValue() {
        if (value == null) return false;
        switch (value.getNodeType()) {
            case Node.DOCUMENT_FRAGMENT_NODE:
                return value.hasChildNodes();
            default:
                return true;
        } 
    } //-- booleanValue
    
    /**
     * Returns true if the given ExprResult is of the same type
     * of this ExprResult, and their values are equal.
     * @return true if the given ExprResult is of the same type
     * of this ExprResult, and their values are equal. otherwise
     * false.
    **/
    public boolean equals(ExprResult exprResult) {
        if (exprResult == null) return false;
        if (value == null) return false;
        Node node = ((TreeFragmentResult)exprResult).getValue();
        return (value == node);
    } //-- equals
    
    /**
     * Returns the value of this TreeFragmentResult
     * @return the value of this TreeFragmentResult
    **/
    public Node getValue() {
        return value;
    } //-- getValue
    
    /**
     * Returns the type of this ExprResult
     * @return the type of this ExprResult
     * @see ExprResult
    **/
    public short getResultType() {
        return ExprResult.TREE_FRAGMENT;
    } //-- getResultType
    
    /**
     * Returns the value of this ExprResult as a double
     * @return the value of this ExprResult as a double
    **/
    public double numberValue() {
        StringResult sr 
            = new StringResult(XSLObject.getNodeValue(value));
        return sr.numberValue();
    } //-- numberValue
    
    /**
     * Sets the value of this TreeFragmentResult
     * @param value the Node value to use as the result value
    **/
    public void setValue(Node value) {
        this.value = value;
    } //-- setTreeFragmentValue
    
    /**
     * Returns the Java Object this ExprResult represents
     * @return the Java Object this ExprResult represents
    **/
    public Object toJavaObject() {
        return value;
    } //-- toJavaObject
    
    /**
     * Returns the value of this ExprResult as a String
     * @return the value of this ExprResult as a String
    **/
    public String toString() {
        return XSLObject.getNodeValue(value);
    } //--toString
    
} //-- TreeFragmentResult