/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */
package com.kvisco.xsl;

import org.w3c.dom.*;

import java.util.Hashtable;

/**
 * A class that represents an XSL StringExpr
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class StringExpr implements Expr {

    public final static char L_CURLY_BRACKET = '{';
    public final static char R_CURLY_BRACKET = '}';
    

    /**
     * The SelectExpr of the node to return the value for.
    **/
    private Expr expr = null;

      //----------------/
     //- Constructors -/
    //----------------/

    /**
     * Creates a new StringExpr using the given String
     * @param exprString the String to create the StringExpr from
     * @exception InvalidExprException
    **/
    public StringExpr(String exprString) throws InvalidExprException {
        super();
        parse(exprString);
    }

    
    /**
     * Evalutes this StringExpr using the given Node as
     * the context
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        if (expr != null) {
            ExprResult exprResult = expr.evaluate(context, ps);
            return toStringResult(exprResult);
        }
        else {
            return new StringResult("");
        }
    } //-- evaluate

      //---------------------/
     //- Protected Methods -/
    //---------------------/

    public short getExprType() { 
        return Expr.STRING; 
    } //-- getExprType
    
    protected Expr getExpr() {
        return this.expr;
    } //-- getSelectExpr

    /**
     * Retrieves the value of the node (Element or Attribute)
     * matched by the pattern of this StringExpr.
     * @param context the node to find "relative" matches from.
     * @param rpState the ProcessorState for accessing the current processing
     * environment
     * @return the String value of the selected node.
    **/
    protected String getValue(Node context, ProcessorState rpState) {

        try {
            return evaluate(context, rpState).toString();
        }
        catch(InvalidExprException iee) {};
        return "";
    } //-- getValue
    
    
    public static StringResult toStringResult(ExprResult exprResult) {
        
        
        if (exprResult == null)
            return new StringResult("null");
        
        switch(exprResult.getResultType()) {
            case ExprResult.BOOLEAN:
            case ExprResult.NUMBER:
            case ExprResult.TREE_FRAGMENT:
                return new StringResult(exprResult.toString());
            case ExprResult.STRING:
                return (StringResult)exprResult;
            case ExprResult.NODE_SET:
                String value = "";
                NodeSet nodeSet = (NodeSet)exprResult;
                if (nodeSet.size() > 0)
                    value = XSLObject.getNodeValue(nodeSet.get(0));
                return new StringResult(value);
            default:
                break;
        }
        
        return new StringResult("null");
        
    } //-- toStringResult
    /**
     * Returns the String representation of this StringExpr
     * @return the String representation of this StringExpr
    **/
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append(L_CURLY_BRACKET);
        if (expr != null) sb.append(expr.toString());
        sb.append(R_CURLY_BRACKET);
        return sb.toString();
    } //-- toString

      //-------------------/
     //- Private Methods -/
    //-------------------/
    
    /**
     * Parses the given String into this StringExpr
     * @param attValue the String to parse
     * @exception InvalidExprException
    **/
    private void parse(String attValue) throws InvalidExprException {
        this.expr = ExpressionParser.createExpr(attValue);
    } //-- parse
    
} //-- StringExpr